'use strict';

import should from 'should';
import sinon from 'sinon';
import StreamingMetaApiConnection from './streamingMetaApiConnection';
import StreamingMetaApiConnectionInstance from './streamingMetaApiConnectionInstance';
import RpcMetaApiConnectionInstance from './rpcMetaApiConnectionInstance';
import ConnectionRegistry from './connectionRegistry';
import TerminalState from './terminalState';

/**
 * @test {ConnectionRegistry}
 */
describe('ConnectionRegistry', () => {

  let sandbox;
  let registry, clock;
  let metaApiWebsocketClient = {
    addSynchronizationListener: () => {},
    addReconnectListener: () => {},
    removeSynchronizationListener: () => {},
    removeAccountCache: () => {},
    removeReconnectListener: () => {},
    subscribe: () => {},
    regionsByAccounts: {},
    unsubscribe: () => {},
    addAccountCache: () => {},
    ensureSubscribe: () => {}
  };
  let storage = {
    lastHistoryOrderTime: () => new Date('2020-01-01T00:00:00.000Z'),
    lastDealTime: () => new Date('2020-01-02T00:00:00.000Z'),
    loadDataFromDisk: () => ({deals: [], historyOrders: []})
  };
  let unsubscribeStub;

  before(() => {
    sandbox = sinon.createSandbox();
  });
  
  beforeEach(() => {
    clock = sandbox.useFakeTimers({
      shouldAdvanceTime: true,
      now: new Date('2020-10-05T07:00:00.000Z')
    });
    registry = new ConnectionRegistry({}, metaApiWebsocketClient);
    sandbox.stub(StreamingMetaApiConnection.prototype, 'initialize').resolves();
    sandbox.stub(StreamingMetaApiConnection.prototype, 'subscribe').resolves();
    sandbox.stub(TerminalState.prototype, 'close').returns();
    unsubscribeStub = sandbox.stub(metaApiWebsocketClient, 'unsubscribe').resolves();
  });

  afterEach(() => {
    sandbox.restore();
  });

  /**
   * @test {ConnectionRegistry#connectStreaming}
   */
  it('should create streaming connection', async () => {
    let account = {id: 'id', region: 'vint-hill', accountRegions: {'vint-hill': 'id', 'new-york': 'idReplica'}};
    let connection = registry.connectStreaming(account, storage);
    await connection.connect();
    (connection instanceof StreamingMetaApiConnectionInstance).should.be.true();
    sinon.assert.match(registry._streamingConnections, sinon.match.has('id', connection._metaApiConnection));
  });

  /**
   * @test {ConnectionRegistry#removeStreaming}
   */
  it('should disconnect streaming connection', async () => {
    let account = {id: 'id', region: 'vint-hill', accountRegions: {'vint-hill': 'id', 'new-york': 'idReplica'}};
    registry.connectStreaming(account, storage);
    await registry.removeStreaming(account);
    sinon.assert.calledWith(unsubscribeStub, 'id');
    sinon.assert.calledWith(unsubscribeStub, 'idReplica');
  });

  /**
   * @test {ConnectionRegistry#removeStreaming}
   */
  it('should not disconnect until both streaming and rpc connections are closed', async () => {
    let account = {id: 'id', region: 'vint-hill', accountRegions: {'vint-hill': 'id', 'new-york': 'idReplica'}};
    registry.connectStreaming(account, storage);
    registry.connectStreaming(account, storage);
    registry.connectRpc(account);
    await registry.removeStreaming(account);
    sinon.assert.notCalled(unsubscribeStub);
    await registry.removeStreaming(account);
    sinon.assert.notCalled(unsubscribeStub);
    await registry.removeRpc(account);
    sinon.assert.calledWith(unsubscribeStub, 'id');
    sinon.assert.calledWith(unsubscribeStub, 'idReplica');
  });

  /**
   * @test {ConnectionRegistry#connectRpc}
   */
  it('should create rpc connection', async () => {
    let account = {id: 'id', region: 'vint-hill', accountRegions: {'vint-hill': 'id', 'new-york': 'idReplica'}};
    let connection = registry.connectRpc(account);
    (connection instanceof RpcMetaApiConnectionInstance).should.be.true();
    sinon.assert.match(registry._rpcConnections, sinon.match.has('id', connection._metaApiConnection));
  });

  /**
   * @test {ConnectionRegistry#removeRpc}
   */
  it('should disconnect rpc connection', async () => {
    let account = {id: 'id', region: 'vint-hill', accountRegions: {'vint-hill': 'id', 'new-york': 'idReplica'}};
    registry.connectRpc(account, storage);
    await registry.removeRpc(account);
    sinon.assert.calledWith(unsubscribeStub, 'id');
    sinon.assert.calledWith(unsubscribeStub, 'idReplica');
  });

  /**
   * @test {ConnectionRegistry#removeRpc}
   */
  it('should not disconnect until both rpc and streaming connections are closed', async () => {
    let account = {id: 'id', region: 'vint-hill', accountRegions: {'vint-hill': 'id', 'new-york': 'idReplica'}};
    registry.connectRpc(account);
    registry.connectRpc(account);
    registry.connectStreaming(account);
    await registry.removeRpc(account);
    sinon.assert.notCalled(unsubscribeStub);
    await registry.removeRpc(account);
    sinon.assert.notCalled(unsubscribeStub);
    await registry.removeStreaming(account);
    sinon.assert.calledWith(unsubscribeStub, 'id');
    sinon.assert.calledWith(unsubscribeStub, 'idReplica');
  });

  /**
   * @test {ConnectionRegistry#removeRpc}
   */
  it('should close connection instances', async () => {
    let account = {id: 'id', region: 'vint-hill', accountRegions: {'vint-hill': 'id', 'new-york': 'idReplica'}};
    let rpc1 = registry.connectRpc(account);
    let rpc2 = registry.connectRpc(account);
    let rpc3 = registry.connectRpc(account);
    let streaming1 = registry.connectStreaming(account);
    let streaming2 = registry.connectStreaming(account);
    let streaming3 = registry.connectStreaming(account);
    await rpc1.connect();
    await streaming1.connect();
    const stubs = {
      rpc1: sandbox.spy(rpc1, 'close'),
      rpc2: sandbox.spy(rpc2, 'close'),
      rpc3: sandbox.spy(rpc3, 'close'),
      streaming1: sandbox.spy(streaming1, 'close'),
      streaming2: sandbox.spy(streaming2, 'close'),
      streaming3: sandbox.spy(streaming3, 'close'),
    };
    sinon.assert.match(Object.keys(registry.streamingConnections).length, 1);
    sinon.assert.match(Object.keys(registry.rpcConnections).length, 1);
    registry.closeAllInstances('id');
    await clock.tickAsync(1000);
    sinon.assert.calledOnce(stubs.rpc1);
    sinon.assert.calledOnce(stubs.rpc2);
    sinon.assert.calledOnce(stubs.rpc3);
    sinon.assert.calledOnce(stubs.streaming1);
    sinon.assert.calledOnce(stubs.streaming2);
    sinon.assert.calledOnce(stubs.streaming3);
    sinon.assert.match(Object.keys(registry.streamingConnections).length, 0);
    sinon.assert.match(Object.keys(registry.rpcConnections).length, 0);
  });

});
