'use strict';

import HttpClient from '../httpClient';
import sinon from 'sinon';
import TokenManagementClient from './tokenManagement.client.es6';

/**
 * @test {TokenManagementClient}
 */
describe('TokenManagementClient', () => {

  const profileApiUrl = 'https://profile-api-v1.agiliumtrade.agiliumtrade.ai';
  const token = 'header.payload.sign';

  let tokenManagementClient;
  let httpClient;
  let domainClient;
  let sandbox;
  let requestStub;

  before(() => {
    httpClient = new HttpClient();
    sandbox = sinon.createSandbox();
  });

  beforeEach(() => {
    domainClient = {
      token,
      domain: 'agiliumtrade.agiliumtrade.ai',
    };
    tokenManagementClient = new TokenManagementClient(httpClient, domainClient);
    requestStub = sandbox.stub(httpClient, 'request');
  });

  afterEach(() => {
    sandbox.restore();
  });

  /**
   * @test {TokenManagementClient#getAccessRules}
   */
  it('should retrieve access rules manifest from API', async () => {
    let expected = [{ 
      id: 'trading-account-management-api',
      description: 'Trading account management API',
      application: 'trading-account-management-api',
      services: [{ description: 'REST API', service: 'rest' }],
      methodGroups: [{ description: 'All method groups', group: '*', 
        methods: [{description: 'All methods', method: '*'}]
      }],
      roles: [{ description: 'Read-only access to resources', roles: ['reader'] }],
      entities: [{ description: 'All entities', entity: '*' }]
    }];

    requestStub.resolves(expected);
    let profiles = await tokenManagementClient.getAccessRules();
    profiles.should.equal(expected);
    sandbox.assert.calledOnceWithExactly(httpClient.request, {
      url: `${profileApiUrl}/access-rule-manifest`,
      method: 'GET',
      headers: {
        'auth-token': token
      },
      json: true,
    }, 'getAccessRules');
  });

  /**
   * @test {TokenManagementClient#getAccessRules}
   */
  it('should not retrieve access rules manifest from API with account token', async () => {
    domainClient.token = 'token';
    tokenManagementClient = new TokenManagementClient(httpClient, domainClient);
    try {
      await tokenManagementClient.getAccessRules();
      sandbox.assert.fail();
    } catch (error) {
      error.message.should.equal(
        'You can not invoke getAccessRules method, because you have connected with account access token. ' +
        'Please use API access token from https://app.metaapi.cloud/api-access/generate-token page ' + 
        'to invoke this method.'
      );
    }
  });

  /**
   * @test {TokenManagementClient#narrowDownToken}
   */
  it('should narrow down token via API', async () => {
    let payload = {
      accessRules: [{
        id: 'trading-account-management-api',
        application: 'trading-account-management-api',
        service: 'rest',
        resources: [{entity: 'account', id: 'accountId'}],
        methodGroups: [{group: 'account-management', methods: [{method: 'getAccount'}]}],
        roles: ['reader'],
      }]
    };
    let expected = {
      token: 'token'
    };
    requestStub.resolves(expected);
    let response = await tokenManagementClient.narrowDownToken(payload);
    response.token.should.equal(expected.token);
    sandbox.assert.calledOnceWithExactly(httpClient.request, {
      url: `${profileApiUrl}/users/current/narrow-down-auth-token`,
      method: 'POST',
      data: payload,
      headers: {
        'auth-token': token
      },
      json: true,
    }, 'narrowDownToken');
  });

  /**
   * @test {TokenManagementClient#narrowDownToken}
   */
  it('should narrow down token with validity hours via API', async () => {
    let payload = {
      accessRules: [{
        id: 'trading-account-management-api',
        application: 'trading-account-management-api',
        service: 'rest',
        resources: [{entity: 'account', id: 'accountId'}],
        methodGroups: [{group: 'account-management', methods: [{method: 'getAccount'}]}],
        roles: ['reader'],
      }]
    };
    const validityInHours = 5;
    let expected = {
      token: 'token'
    };
    requestStub.resolves(expected);
    let response = await tokenManagementClient.narrowDownToken(payload, validityInHours);
    response.token.should.equal(expected.token);
    sandbox.assert.calledOnceWithExactly(httpClient.request, {
      url: `${profileApiUrl}/users/current/narrow-down-auth-token?validity-in-hours=${validityInHours}`,
      method: 'POST',
      data: payload,
      headers: {
        'auth-token': token
      },
      json: true,
    }, 'narrowDownToken');
  });

  it('should not narrow down via API with account token', async () => {
    domainClient.token = 'token';
    tokenManagementClient = new TokenManagementClient(httpClient, domainClient);
    try {
      await tokenManagementClient.narrowDownToken({});
      sandbox.assert.fail();
    } catch (error) {
      error.message.should.equal(
        'You can not invoke narrowDownToken method, because you have connected with account access token. ' +
        'Please use API access token from https://app.metaapi.cloud/api-access/generate-token page ' + 
        'to invoke this method.'
      );
    }
  });

});
