import MetatraderAccountClient, { MetatraderAccountReplicaDto, UpdatedMetatraderAccountReplicaDto } from "../clients/metaApi/metatraderAccount.client";
import MetatraderAccount from "./metatraderAccount";
import {Reliability, State, ConnectionStatus} from '../clients/metaApi/metatraderAccount.client'

/**
 * Implements a MetaTrader account replica entity
 */
export default class MetatraderAccountReplica {
  
  /**
   * Constructs a MetaTrader account replica entity
   * @param {MetatraderAccountReplicaDto} data MetaTrader account data
   * @param {MetatraderAccount} primaryAccount primary MetaTrader account
   * @param {MetatraderAccountClient} metatraderAccountClient MetaTrader account REST API client
   */
  constructor(data: MetatraderAccountReplicaDto, primaryAccount: MetatraderAccount, metatraderAccountClient: MetatraderAccountClient);
  
  /**
   * Returns account replica id
   * @return {string} unique account replica id
   */
  get id(): string;

  /**
   * Returns current account replica state. One of CREATED, DEPLOYING, DEPLOYED, DEPLOY_FAILED, UNDEPLOYING,
   * UNDEPLOYED, UNDEPLOY_FAILED, DELETING, DELETE_FAILED, REDEPLOY_FAILED, DRAFT
   * @return {State} current account replica state
   */
  get state(): State;

  /**
   * Returns MetaTrader magic to place trades using
   * @return {number} MetaTrader magic to place trades using
   */
  get magic(): number;

  /**
   * Returns terminal & broker connection status, one of CONNECTED, DISCONNECTED, DISCONNECTED_FROM_BROKER
   * @return {ConnectionStatus} terminal & broker connection status
   */
  get connectionStatus(): ConnectionStatus;
  
  /**
   * Returns quote streaming interval in seconds 
   * @return {number} quote streaming interval in seconds
   */
  get quoteStreamingIntervalInSeconds(): number;
  
  /**
   * Returns symbol provided by broker 
   * @return {string} any symbol provided by broker
   */
  get symbol(): string;
  
  /**
   * Returns reliability value. Possible values are regular and high
   * @return {Reliability} account replica reliability value
   */
  get reliability(): Reliability;
  
  /**
   * Returns user-defined account replica tags
   * @return {Array<string>} user-defined account replica tags
   */
  get tags(): Array<string>;

  /**
   * Returns extra information which can be stored together with your account replica
   * @return {Object} extra information which can be stored together with your account replica
   */
  get metadata(): Object;

   /**
   * Returns number of resource slots to allocate to account replica. Allocating extra resource slots
   * results in better account performance under load which is useful for some applications. E.g. if you have many
   * accounts copying the same strategy via CopyFactory API, then you can increase resourceSlots to get a lower trade
   * copying latency. Please note that allocating extra resource slots is a paid option. Please note that high
   * reliability accounts use redundant infrastructure, so that each resource slot for a high reliability account
   * is billed as 2 standard resource slots.
   * @return {number} number of resource slots to allocate to account replica
   */
  get resourceSlots(): number;

  /**
   * Returns the number of CopyFactory 2 resource slots to allocate to account replica.
   * Allocating extra resource slots results in lower trade copying latency. Please note that allocating extra resource
   * slots is a paid option. Please also note that CopyFactory 2 uses redundant infrastructure so that
   * each CopyFactory resource slot is billed as 2 standard resource slots. You will be billed for CopyFactory 2
   * resource slots only if you have added your account replica to CopyFactory 2 by specifying copyFactoryRoles field.
   * @return {number} number of CopyFactory 2 resource slots to allocate to account replica
   */
  get copyFactoryResourceSlots(): number;

  /**
   * Returns account replica region
   * @return {string} account replica region value
   */
  get region(): string;

  /**
   * Returns the time account replica was created at, in ISO format
   * @returns {string} the time account replica was created at, in ISO format
   */
  get createdAt(): Date;

  /**
   * Returns primary MetaTrader account of the replica from DTO
   * @return {MetatraderAccount} primary MetaTrader account of the replica from DTO
   */
  get primaryAccountFromDto(): MetatraderAccount;

  /**
   * Returns primary MetaTrader account of the replica
   * @return {MetatraderAccount} primary MetaTrader account of the replica
   */
  get primaryAccount(): MetatraderAccount;

  /**
   * Updates account replica data
   * @param {MetatraderAccountReplicaDto} data MetaTrader account replica data 
   */
  updateData(data: MetatraderAccountReplicaDto): void;

  /**
   * Removes a trading account replica and stops the API server serving the replica
   * @return {Promise} promise resolving when account replica is scheduled for deletion
   */
  remove(): Promise<any>;

  /**
   * Starts API server and trading terminal for trading account replica.
   * This request will be ignored if the replica is already deployed
   * @returns {Promise} promise resolving when account replica is scheduled for deployment
   */
  deploy(): Promise<any>;
  
  /**
   * Stops API server and trading terminal for trading account replica.
   * The request will be ignored if trading account replica is already undeployed
   * @returns {Promise} promise resolving when account replica is scheduled for undeployment
   */
  undeploy(): Promise<any>;
  
  /**
   * Redeploys trading account replica. This is equivalent to undeploy immediately followed by deploy.
   * @returns {Promise} promise resolving when account replica is scheduled for redeployment
   */
  redeploy(): Promise<any>;
  
  /**
   * Increases trading account reliability in order to increase the expected account uptime.
   * The account will be temporary stopped to perform this action.
   * Note that increasing reliability is a paid option
   * @returns {Promise} promise resolving when account replica reliability is increased
   */
  increaseReliability(): Promise<any>;
  
   /**
   * Waits until API server has finished deployment and account replica reached the DEPLOYED state
   * @param {number} timeoutInSeconds wait timeout in seconds, default is 5m
   * @param {number} intervalInMilliseconds interval between account replica reloads while waiting for a change, default is 1s
   * @return {Promise} promise which resolves when account replica is deployed
   */
  waitDeployed(timeoutInSeconds?: number, intervalInMilliseconds?: number): Promise<any>;
  
  /**
   * Waits until API server has finished undeployment and account replica reached the UNDEPLOYED state
   * @param {number} timeoutInSeconds wait timeout in seconds, default is 5m
   * @param {number} intervalInMilliseconds interval between account replica reloads while waiting for a change, default is 1s
   * @return {Promise} promise which resolves when account replica is deployed
   */
  waitUndeployed(timeoutInSeconds?: number, intervalInMilliseconds?: number): Promise<any>;
  
  /**
   * Waits until account replica has been deleted
   * @param {number} timeoutInSeconds wait timeout in seconds, default is 5m
   * @param {number} intervalInMilliseconds interval between account replica reloads while waiting for a change, default is 1s
   * @return {Promise} promise which resolves when account replica is deleted
   */
  waitRemoved(timeoutInSeconds?: number, intervalInMilliseconds?: number): Promise<any>;
  
  /**
   * Waits until API server has connected to the terminal and terminal has connected to the broker
   * @param {number} timeoutInSeconds wait timeout in seconds, default is 5m
   * @param {number} intervalInMilliseconds interval between account replica reloads while waiting for a change, default is 1s
   * @return {Promise} promise which resolves when API server is connected to the broker
   */
  waitConnected(timeoutInSeconds?: number, intervalInMilliseconds?: number): Promise<any>;
  
  /**
   * Updates trading account replica
   * @param {UpdatedMetatraderAccountReplicaDto} metatraderAccount updated account replica information
   * @return {Promise} promise resolving when account replica is updated
   */
  update(metatraderAccount: UpdatedMetatraderAccountReplicaDto): Promise<any>;
}
