"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return RpcMetaApiConnection;
    }
});
const _logger = /*#__PURE__*/ _interop_require_default(require("../logger"));
const _metaApiConnection = /*#__PURE__*/ _interop_require_default(require("./metaApiConnection"));
const _timeoutError = /*#__PURE__*/ _interop_require_default(require("../clients/timeoutError"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let RpcMetaApiConnection = class RpcMetaApiConnection extends _metaApiConnection.default {
    /**
   * Opens the connection. Can only be called the first time, next calls will be ignored.
   * @param {string} instanceId connection instance id
   * @return {Promise} promise resolving when the connection is opened
   */ async connect(instanceId) {
        if (!this._openedInstances.includes(instanceId)) {
            this._openedInstances.push(instanceId);
        }
        if (!this._opened) {
            this._opened = true;
            const accountRegions = this._account.accountRegions;
            this._websocketClient.addAccountCache(this._account.id, accountRegions);
            Object.keys(accountRegions).forEach((region)=>{
                if (!this._options.region || this._options.region === region) {
                    this._websocketClient.ensureSubscribe(accountRegions[region], 0);
                    this._websocketClient.ensureSubscribe(accountRegions[region], 1);
                }
            });
        }
    }
    /**
   * Closes the connection. The instance of the class should no longer be used after this method is invoked.
   * @param {string} instanceId connection instance id
   */ async close(instanceId) {
        if (this._opened) {
            this._openedInstances = this._openedInstances.filter((id)=>id !== instanceId);
            if (!this._openedInstances.length && !this._closed) {
                clearInterval(this._refreshJob);
                await this._connectionRegistry.removeRpc(this.account);
                this._websocketClient.removeSynchronizationListener(this.account.id, this);
                this._websocketClient.removeAccountCache(this.account.id);
                this._websocketClient.removeReconnectListener(this);
                this._closed = true;
            }
        }
    }
    /**
   * Invoked when connection to MetaTrader terminal established
   * @param {String} instanceIndex index of an account instance connected
   * @param {Number} replicas number of account replicas launched
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ async onConnected(instanceIndex, replicas) {
        const state = this._getState(instanceIndex);
        state.synchronized = true;
        const region = this.getRegion(instanceIndex);
        this.cancelRefresh(region);
    }
    /**
   * Invoked when connection to MetaTrader terminal terminated
   * @param {String} instanceIndex index of an account instance connected
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ async onDisconnected(instanceIndex) {
        const state = this._getState(instanceIndex);
        state.synchronized = false;
        this._logger.debug(`${this._account.id}:${instanceIndex}: disconnected from broker`);
    }
    /**
   * Invoked when a stream for an instance index is closed
   * @param {String} instanceIndex index of an account instance connected
   */ async onStreamClosed(instanceIndex) {
        delete this._stateByInstanceIndex[instanceIndex];
    }
    /**
   * Returns flag indicating status of state synchronization with MetaTrader terminal
   * @returns {Boolean} a flag indicating status of state synchronization with MetaTrader terminal
   */ isSynchronized() {
        return Object.values(this._stateByInstanceIndex).map((instance)=>instance.synchronized).includes(true);
    }
    /**
   * Waits until synchronization to RPC application is completed
   * @param {Number} timeoutInSeconds synchronization timeout in seconds. Defaults to 5 minutes
   * @return {Promise} promise which resolves when synchronization to RPC application is completed
   * @throws {TimeoutError} if application failed to synchronize with the teminal within timeout allowed
   */ async waitSynchronized(timeoutInSeconds = 300) {
        this._checkIsConnectionActive();
        const startTime = Date.now();
        let synchronized = this.isSynchronized();
        while(!synchronized && startTime + timeoutInSeconds * 1000 > Date.now()){
            await new Promise((res)=>setTimeout(res, 1000));
            synchronized = this.isSynchronized();
        }
        if (!synchronized) {
            throw new _timeoutError.default("Timed out waiting for MetaApi to synchronize to MetaTrader account " + this._account.id);
        }
        // eslint-disable-next-line
        while(true){
            try {
                await this._websocketClient.waitSynchronized(this._account.id, undefined, "RPC", 5, "RPC");
                break;
            } catch (err) {
                if (Date.now() > startTime + timeoutInSeconds * 1000) {
                    throw err;
                }
            }
        }
    }
    /**
   * Invoked when connection to MetaApi websocket API restored after a disconnect
   * @param {String} region reconnected region
   * @param {Number} instanceNumber reconnected instance number
   * @return {Promise} promise which resolves when connection to MetaApi websocket API restored after a disconnect
   */ async onReconnected(region, instanceNumber) {
        const instanceTemplate = `${region}:${instanceNumber}`;
        Object.keys(this._stateByInstanceIndex).filter((key)=>key.startsWith(`${instanceTemplate}:`)).forEach((key)=>{
            delete this._stateByInstanceIndex[key];
        });
    }
    _getState(instanceIndex) {
        if (!this._stateByInstanceIndex[instanceIndex]) {
            this._stateByInstanceIndex[instanceIndex] = {
                instanceIndex,
                synchronized: false
            };
        }
        return this._stateByInstanceIndex[instanceIndex];
    }
    /**
   * Constructs MetaApi MetaTrader RPC Api connection
   * @param {MetaApiOpts} options MetaApi options
   * @param {MetaApiWebsocketClient} websocketClient MetaApi websocket client
   * @param {MetatraderAccount} account MetaTrader account id to connect to
   * @param {ConnectionRegistry} connectionRegistry metatrader account connection registry
   */ constructor(options, websocketClient, account, connectionRegistry){
        super(options, websocketClient, account, "RPC");
        this._connectionRegistry = connectionRegistry;
        this._websocketClient.addSynchronizationListener(account.id, this);
        this._stateByInstanceIndex = {};
        this._openedInstances = [];
        Object.values(account.accountRegions).forEach((replicaId)=>this._websocketClient.addReconnectListener(this, replicaId));
        this._logger = _logger.default.getLogger("MetaApiConnection");
    }
};

//# sourceMappingURL=data:application/json;base64,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