"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return ConnectionHealthMonitor;
    }
});
const _synchronizationListener = /*#__PURE__*/ _interop_require_default(require("../clients/metaApi/synchronizationListener"));
const _moment = /*#__PURE__*/ _interop_require_default(require("moment"));
const _reservoir = /*#__PURE__*/ _interop_require_default(require("./reservoir/reservoir"));
const _logger = /*#__PURE__*/ _interop_require_default(require("../logger"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let ConnectionHealthMonitor = class ConnectionHealthMonitor extends _synchronizationListener.default {
    /**
   * Stops health monitor
   */ stop() {
        this._logger.debug(`${this._connection.account.id}: Stopping the monitor`);
        clearTimeout(this._updateQuoteHealthStatusInterval);
        clearTimeout(this._measureUptimeInterval);
    }
    /**
   * Invoked when a symbol price was updated
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderSymbolPrice} price updated MetaTrader symbol price
   */ onSymbolPriceUpdated(instanceIndex, price) {
        try {
            let brokerTimestamp = (0, _moment.default)(price.brokerTime).toDate().getTime();
            this._priceUpdatedAt = new Date();
            this._offset = this._priceUpdatedAt.getTime() - brokerTimestamp;
        } catch (err) {
            // eslint-disable-next-line no-console
            this._logger.error(`${this._connection.account.id}: Failed to update quote ` + "streaming health status on price update", err);
        }
    }
    /**
   * Invoked when a server-side application health status is received from MetaApi
   * @param {String} instanceIndex index of an account instance connected
   * @param {HealthStatus} status server-side application health status
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onHealthStatus(instanceIndex, status) {
        this._serverHealthStatus["" + instanceIndex] = status;
    }
    /**
   * Invoked when connection to MetaTrader terminal terminated
   * @param {String} instanceIndex index of an account instance connected
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onDisconnected(instanceIndex) {
        delete this._serverHealthStatus["" + instanceIndex];
    }
    /**
   * Returns server-side application health status
   * @return {HealthStatus} server-side application health status
   */ get serverHealthStatus() {
        let result;
        for (let s of Object.values(this._serverHealthStatus)){
            if (!result) {
                result = s;
            } else {
                for (let field of Object.keys(s)){
                    result[field] = result[field] || s[field];
                }
            }
        }
        return result || {};
    }
    /**
   * Connection health status
   * @typedef {Object} ConnectionHealthStatus
   * @property {Boolean} connected flag indicating successful connection to API server
   * @property {Boolean} connectedToBroker flag indicating successful connection to broker
   * @property {Boolean} quoteStreamingHealthy flag indicating that quotes are being streamed successfully from the
   * broker
   * @property {Boolean} synchronized flag indicating a successful synchronization
   * @property {Boolean} healthy flag indicating overall connection health status
   * @property {String} message health status message
   */ /**
   * Returns health status
   * @returns {ConnectionHealthStatus} connection health status
   */ // eslint-disable-next-line complexity
    get healthStatus() {
        let status = {
            connected: this._connection.terminalState.connected,
            connectedToBroker: this._connection.terminalState.connectedToBroker,
            quoteStreamingHealthy: this._quotesHealthy,
            synchronized: this._connection.synchronized
        };
        status.healthy = status.connected && status.connectedToBroker && status.quoteStreamingHealthy && status.synchronized;
        let message;
        if (status.healthy) {
            message = "Connection to broker is stable. No health issues detected.";
        } else {
            message = "Connection is not healthy because ";
            let reasons = [];
            if (!status.connected) {
                reasons.push("connection to API server is not established or lost");
            }
            if (!status.connectedToBroker) {
                reasons.push("connection to broker is not established or lost");
            }
            if (!status.synchronized) {
                reasons.push("local terminal state is not synchronized to broker");
            }
            if (!status.quoteStreamingHealthy) {
                reasons.push("quotes are not streamed from the broker properly");
            }
            message = message + reasons.join(" and ") + ".";
        }
        status.message = message;
        return status;
    }
    /**
   * Returns uptime in percents measured over specific periods of time
   * @returns {Object} uptime in percents measured over specific periods of time
   */ get uptime() {
        let uptime = {};
        for (let e of Object.entries(this._uptimeReservoirs)){
            uptime[e[0]] = e[1].getStatistics().average;
        }
        return uptime;
    }
    _measureUptime() {
        try {
            Object.values(this._uptimeReservoirs).forEach((r)=>r.pushMeasurement(this._connection.terminalState.connected && this._connection.terminalState.connectedToBroker && this._connection.synchronized && this._quotesHealthy ? 100 : 0));
        } catch (err) {
            // eslint-disable-next-line no-console
            this._logger.error("failed to measure uptime for account " + this._connection.account.id, err);
        }
    }
    // eslint-disable-next-line complexity
    _updateQuoteHealthStatus() {
        try {
            let serverDateTime = (0, _moment.default)(new Date(Date.now() - this._offset));
            let serverTime = serverDateTime.format("HH:mm:ss.SSS");
            let dayOfWeek = serverDateTime.day();
            let daysOfWeek = {
                0: "SUNDAY",
                1: "MONDAY",
                2: "TUESDAY",
                3: "WEDNESDAY",
                4: "THURSDAY",
                5: "FRIDAY",
                6: "SATURDAY"
            };
            let inQuoteSession = false;
            if (!this._priceUpdatedAt) {
                this._priceUpdatedAt = new Date();
            }
            if (!(this._connection.subscribedSymbols || []).length) {
                this._priceUpdatedAt = new Date();
            }
            for (let symbol of this._connection.subscribedSymbols || []){
                let specification = this._connection.terminalState.specification(symbol) || {};
                let quoteSessions = (specification.quoteSessions || [])[daysOfWeek[dayOfWeek]] || [];
                for (let session of quoteSessions){
                    if (session.from <= serverTime && session.to >= serverTime) {
                        inQuoteSession = true;
                    }
                }
            }
            this._quotesHealthy = !this._connection.subscribedSymbols.length || !inQuoteSession || Date.now() - this._priceUpdatedAt.getTime() < this._minQuoteInterval;
        } catch (err) {
            // eslint-disable-next-line no-console
            this._logger.error("failed to update quote streaming health status for account " + this._connection.account.id, err);
        }
    }
    _getRandomTimeout() {
        return (Math.random() * 59 + 1) * 1000;
    }
    /**
   * Constructs the listener
   * @param {StreamingMetaApiConnection} connection MetaApi connection instance
   */ constructor(connection){
        super();
        this._connection = connection;
        const updateQuoteHealthStatusInterval = ()=>{
            this._updateQuoteHealthStatus();
            this._updateQuoteHealthStatusInterval = setTimeout(updateQuoteHealthStatusInterval.bind(this), this._getRandomTimeout());
        };
        this._updateQuoteHealthStatusInterval = setTimeout(updateQuoteHealthStatusInterval.bind(this), this._getRandomTimeout());
        const measureUptimeInterval = ()=>{
            this._measureUptime();
            this._measureUptimeInterval = setTimeout(measureUptimeInterval.bind(this), this._getRandomTimeout());
        };
        this._measureUptimeInterval = setTimeout(measureUptimeInterval.bind(this), this._getRandomTimeout());
        this._minQuoteInterval = 60000;
        this._serverHealthStatus = {};
        this._uptimeReservoirs = {
            "5m": new _reservoir.default(300, 5 * 60 * 1000),
            "1h": new _reservoir.default(600, 60 * 60 * 1000),
            "1d": new _reservoir.default(24 * 60, 24 * 60 * 60 * 1000),
            "1w": new _reservoir.default(24 * 7, 7 * 24 * 60 * 60 * 1000)
        };
        this._logger = _logger.default.getLogger("ConnectionHealthMonitor");
    }
};

//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIjxhbm9uPiJdLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbmltcG9ydCBTeW5jaHJvbml6YXRpb25MaXN0ZW5lciBmcm9tICcuLi9jbGllbnRzL21ldGFBcGkvc3luY2hyb25pemF0aW9uTGlzdGVuZXInO1xuaW1wb3J0IG1vbWVudCBmcm9tICdtb21lbnQnO1xuaW1wb3J0IFJlc2Vydm9pciBmcm9tICcuL3Jlc2Vydm9pci9yZXNlcnZvaXInO1xuaW1wb3J0IExvZ2dlck1hbmFnZXIgZnJvbSAnLi4vbG9nZ2VyJztcblxuLyoqXG4gKiBUcmFja3MgY29ubmVjdGlvbiBoZWFsdGggc3RhdHVzXG4gKi9cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIENvbm5lY3Rpb25IZWFsdGhNb25pdG9yIGV4dGVuZHMgU3luY2hyb25pemF0aW9uTGlzdGVuZXIge1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3RzIHRoZSBsaXN0ZW5lclxuICAgKiBAcGFyYW0ge1N0cmVhbWluZ01ldGFBcGlDb25uZWN0aW9ufSBjb25uZWN0aW9uIE1ldGFBcGkgY29ubmVjdGlvbiBpbnN0YW5jZVxuICAgKi9cbiAgY29uc3RydWN0b3IoY29ubmVjdGlvbikge1xuICAgIHN1cGVyKCk7XG4gICAgdGhpcy5fY29ubmVjdGlvbiA9IGNvbm5lY3Rpb247XG4gICAgY29uc3QgdXBkYXRlUXVvdGVIZWFsdGhTdGF0dXNJbnRlcnZhbCA9ICgpID0+IHtcbiAgICAgIHRoaXMuX3VwZGF0ZVF1b3RlSGVhbHRoU3RhdHVzKCk7XG4gICAgICB0aGlzLl91cGRhdGVRdW90ZUhlYWx0aFN0YXR1c0ludGVydmFsID0gc2V0VGltZW91dCh1cGRhdGVRdW90ZUhlYWx0aFN0YXR1c0ludGVydmFsLmJpbmQodGhpcyksXG4gICAgICAgIHRoaXMuX2dldFJhbmRvbVRpbWVvdXQoKSk7XG4gICAgfTtcbiAgICB0aGlzLl91cGRhdGVRdW90ZUhlYWx0aFN0YXR1c0ludGVydmFsID0gc2V0VGltZW91dCh1cGRhdGVRdW90ZUhlYWx0aFN0YXR1c0ludGVydmFsLmJpbmQodGhpcyksXG4gICAgICB0aGlzLl9nZXRSYW5kb21UaW1lb3V0KCkpO1xuICAgIGNvbnN0IG1lYXN1cmVVcHRpbWVJbnRlcnZhbCA9ICgpID0+IHtcbiAgICAgIHRoaXMuX21lYXN1cmVVcHRpbWUoKTtcbiAgICAgIHRoaXMuX21lYXN1cmVVcHRpbWVJbnRlcnZhbCA9IHNldFRpbWVvdXQobWVhc3VyZVVwdGltZUludGVydmFsLmJpbmQodGhpcyksXG4gICAgICAgIHRoaXMuX2dldFJhbmRvbVRpbWVvdXQoKSk7XG4gICAgfTtcbiAgICB0aGlzLl9tZWFzdXJlVXB0aW1lSW50ZXJ2YWwgPSBzZXRUaW1lb3V0KG1lYXN1cmVVcHRpbWVJbnRlcnZhbC5iaW5kKHRoaXMpLFxuICAgICAgdGhpcy5fZ2V0UmFuZG9tVGltZW91dCgpKTtcbiAgICB0aGlzLl9taW5RdW90ZUludGVydmFsID0gNjAwMDA7XG4gICAgdGhpcy5fc2VydmVySGVhbHRoU3RhdHVzID0ge307XG4gICAgdGhpcy5fdXB0aW1lUmVzZXJ2b2lycyA9IHtcbiAgICAgICc1bSc6IG5ldyBSZXNlcnZvaXIoMzAwLCA1ICogNjAgKiAxMDAwKSxcbiAgICAgICcxaCc6IG5ldyBSZXNlcnZvaXIoNjAwLCA2MCAqIDYwICogMTAwMCksXG4gICAgICAnMWQnOiBuZXcgUmVzZXJ2b2lyKDI0ICogNjAsIDI0ICogNjAgKiA2MCAqIDEwMDApLFxuICAgICAgJzF3JzogbmV3IFJlc2Vydm9pcigyNCAqIDcsIDcgKiAyNCAqIDYwICogNjAgKiAxMDAwKSxcbiAgICB9O1xuICAgIHRoaXMuX2xvZ2dlciA9IExvZ2dlck1hbmFnZXIuZ2V0TG9nZ2VyKCdDb25uZWN0aW9uSGVhbHRoTW9uaXRvcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0b3BzIGhlYWx0aCBtb25pdG9yXG4gICAqL1xuICBzdG9wKCkge1xuICAgIHRoaXMuX2xvZ2dlci5kZWJ1ZyhgJHt0aGlzLl9jb25uZWN0aW9uLmFjY291bnQuaWR9OiBTdG9wcGluZyB0aGUgbW9uaXRvcmApO1xuICAgIGNsZWFyVGltZW91dCh0aGlzLl91cGRhdGVRdW90ZUhlYWx0aFN0YXR1c0ludGVydmFsKTtcbiAgICBjbGVhclRpbWVvdXQodGhpcy5fbWVhc3VyZVVwdGltZUludGVydmFsKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJbnZva2VkIHdoZW4gYSBzeW1ib2wgcHJpY2Ugd2FzIHVwZGF0ZWRcbiAgICogQHBhcmFtIHtTdHJpbmd9IGluc3RhbmNlSW5kZXggaW5kZXggb2YgYW4gYWNjb3VudCBpbnN0YW5jZSBjb25uZWN0ZWRcbiAgICogQHBhcmFtIHtNZXRhdHJhZGVyU3ltYm9sUHJpY2V9IHByaWNlIHVwZGF0ZWQgTWV0YVRyYWRlciBzeW1ib2wgcHJpY2VcbiAgICovXG4gIG9uU3ltYm9sUHJpY2VVcGRhdGVkKGluc3RhbmNlSW5kZXgsIHByaWNlKSB7XG4gICAgdHJ5IHtcbiAgICAgIGxldCBicm9rZXJUaW1lc3RhbXAgPSBtb21lbnQocHJpY2UuYnJva2VyVGltZSkudG9EYXRlKCkuZ2V0VGltZSgpO1xuICAgICAgdGhpcy5fcHJpY2VVcGRhdGVkQXQgPSBuZXcgRGF0ZSgpO1xuICAgICAgdGhpcy5fb2Zmc2V0ID0gdGhpcy5fcHJpY2VVcGRhdGVkQXQuZ2V0VGltZSgpIC0gYnJva2VyVGltZXN0YW1wO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWNvbnNvbGVcbiAgICAgIHRoaXMuX2xvZ2dlci5lcnJvcihgJHt0aGlzLl9jb25uZWN0aW9uLmFjY291bnQuaWR9OiBGYWlsZWQgdG8gdXBkYXRlIHF1b3RlIGAgKyBcbiAgICAgICAgJ3N0cmVhbWluZyBoZWFsdGggc3RhdHVzIG9uIHByaWNlIHVwZGF0ZScsIGVycik7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEludm9rZWQgd2hlbiBhIHNlcnZlci1zaWRlIGFwcGxpY2F0aW9uIGhlYWx0aCBzdGF0dXMgaXMgcmVjZWl2ZWQgZnJvbSBNZXRhQXBpXG4gICAqIEBwYXJhbSB7U3RyaW5nfSBpbnN0YW5jZUluZGV4IGluZGV4IG9mIGFuIGFjY291bnQgaW5zdGFuY2UgY29ubmVjdGVkXG4gICAqIEBwYXJhbSB7SGVhbHRoU3RhdHVzfSBzdGF0dXMgc2VydmVyLXNpZGUgYXBwbGljYXRpb24gaGVhbHRoIHN0YXR1c1xuICAgKiBAcmV0dXJuIHtQcm9taXNlfSBwcm9taXNlIHdoaWNoIHJlc29sdmVzIHdoZW4gdGhlIGFzeW5jaHJvbm91cyBldmVudCBpcyBwcm9jZXNzZWRcbiAgICovXG4gIG9uSGVhbHRoU3RhdHVzKGluc3RhbmNlSW5kZXgsIHN0YXR1cykge1xuICAgIHRoaXMuX3NlcnZlckhlYWx0aFN0YXR1c1snJyArIGluc3RhbmNlSW5kZXhdID0gc3RhdHVzO1xuICB9XG5cbiAgLyoqXG4gICAqIEludm9rZWQgd2hlbiBjb25uZWN0aW9uIHRvIE1ldGFUcmFkZXIgdGVybWluYWwgdGVybWluYXRlZFxuICAgKiBAcGFyYW0ge1N0cmluZ30gaW5zdGFuY2VJbmRleCBpbmRleCBvZiBhbiBhY2NvdW50IGluc3RhbmNlIGNvbm5lY3RlZFxuICAgKiBAcmV0dXJuIHtQcm9taXNlfSBwcm9taXNlIHdoaWNoIHJlc29sdmVzIHdoZW4gdGhlIGFzeW5jaHJvbm91cyBldmVudCBpcyBwcm9jZXNzZWRcbiAgICovXG4gIG9uRGlzY29ubmVjdGVkKGluc3RhbmNlSW5kZXgpIHtcbiAgICBkZWxldGUgdGhpcy5fc2VydmVySGVhbHRoU3RhdHVzWycnICsgaW5zdGFuY2VJbmRleF07XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBzZXJ2ZXItc2lkZSBhcHBsaWNhdGlvbiBoZWFsdGggc3RhdHVzXG4gICAqIEByZXR1cm4ge0hlYWx0aFN0YXR1c30gc2VydmVyLXNpZGUgYXBwbGljYXRpb24gaGVhbHRoIHN0YXR1c1xuICAgKi9cbiAgZ2V0IHNlcnZlckhlYWx0aFN0YXR1cygpIHtcbiAgICBsZXQgcmVzdWx0O1xuICAgIGZvciAobGV0IHMgb2YgT2JqZWN0LnZhbHVlcyh0aGlzLl9zZXJ2ZXJIZWFsdGhTdGF0dXMpKSB7XG4gICAgICBpZiAoIXJlc3VsdCkge1xuICAgICAgICByZXN1bHQgPSBzO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgZm9yIChsZXQgZmllbGQgb2YgT2JqZWN0LmtleXMocykpIHtcbiAgICAgICAgICByZXN1bHRbZmllbGRdID0gcmVzdWx0W2ZpZWxkXSB8fCBzW2ZpZWxkXTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gcmVzdWx0IHx8IHt9O1xuICB9XG5cbiAgLyoqXG4gICAqIENvbm5lY3Rpb24gaGVhbHRoIHN0YXR1c1xuICAgKiBAdHlwZWRlZiB7T2JqZWN0fSBDb25uZWN0aW9uSGVhbHRoU3RhdHVzXG4gICAqIEBwcm9wZXJ0eSB7Qm9vbGVhbn0gY29ubmVjdGVkIGZsYWcgaW5kaWNhdGluZyBzdWNjZXNzZnVsIGNvbm5lY3Rpb24gdG8gQVBJIHNlcnZlclxuICAgKiBAcHJvcGVydHkge0Jvb2xlYW59IGNvbm5lY3RlZFRvQnJva2VyIGZsYWcgaW5kaWNhdGluZyBzdWNjZXNzZnVsIGNvbm5lY3Rpb24gdG8gYnJva2VyXG4gICAqIEBwcm9wZXJ0eSB7Qm9vbGVhbn0gcXVvdGVTdHJlYW1pbmdIZWFsdGh5IGZsYWcgaW5kaWNhdGluZyB0aGF0IHF1b3RlcyBhcmUgYmVpbmcgc3RyZWFtZWQgc3VjY2Vzc2Z1bGx5IGZyb20gdGhlXG4gICAqIGJyb2tlclxuICAgKiBAcHJvcGVydHkge0Jvb2xlYW59IHN5bmNocm9uaXplZCBmbGFnIGluZGljYXRpbmcgYSBzdWNjZXNzZnVsIHN5bmNocm9uaXphdGlvblxuICAgKiBAcHJvcGVydHkge0Jvb2xlYW59IGhlYWx0aHkgZmxhZyBpbmRpY2F0aW5nIG92ZXJhbGwgY29ubmVjdGlvbiBoZWFsdGggc3RhdHVzXG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSBtZXNzYWdlIGhlYWx0aCBzdGF0dXMgbWVzc2FnZVxuICAgKi9cblxuICAvKipcbiAgICogUmV0dXJucyBoZWFsdGggc3RhdHVzXG4gICAqIEByZXR1cm5zIHtDb25uZWN0aW9uSGVhbHRoU3RhdHVzfSBjb25uZWN0aW9uIGhlYWx0aCBzdGF0dXNcbiAgICovXG4gIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBjb21wbGV4aXR5XG4gIGdldCBoZWFsdGhTdGF0dXMoKSB7XG4gICAgbGV0IHN0YXR1cyA9IHtcbiAgICAgIGNvbm5lY3RlZDogdGhpcy5fY29ubmVjdGlvbi50ZXJtaW5hbFN0YXRlLmNvbm5lY3RlZCxcbiAgICAgIGNvbm5lY3RlZFRvQnJva2VyOiB0aGlzLl9jb25uZWN0aW9uLnRlcm1pbmFsU3RhdGUuY29ubmVjdGVkVG9Ccm9rZXIsXG4gICAgICBxdW90ZVN0cmVhbWluZ0hlYWx0aHk6IHRoaXMuX3F1b3Rlc0hlYWx0aHksXG4gICAgICBzeW5jaHJvbml6ZWQ6IHRoaXMuX2Nvbm5lY3Rpb24uc3luY2hyb25pemVkXG4gICAgfTtcbiAgICBzdGF0dXMuaGVhbHRoeSA9IHN0YXR1cy5jb25uZWN0ZWQgJiYgc3RhdHVzLmNvbm5lY3RlZFRvQnJva2VyICYmIHN0YXR1cy5xdW90ZVN0cmVhbWluZ0hlYWx0aHkgJiZcbiAgICAgIHN0YXR1cy5zeW5jaHJvbml6ZWQ7XG4gICAgbGV0IG1lc3NhZ2U7XG4gICAgaWYgKHN0YXR1cy5oZWFsdGh5KSB7XG4gICAgICBtZXNzYWdlID0gJ0Nvbm5lY3Rpb24gdG8gYnJva2VyIGlzIHN0YWJsZS4gTm8gaGVhbHRoIGlzc3VlcyBkZXRlY3RlZC4nO1xuICAgIH0gZWxzZSB7XG4gICAgICBtZXNzYWdlID0gJ0Nvbm5lY3Rpb24gaXMgbm90IGhlYWx0aHkgYmVjYXVzZSAnO1xuICAgICAgbGV0IHJlYXNvbnMgPSBbXTtcbiAgICAgIGlmICghc3RhdHVzLmNvbm5lY3RlZCkge1xuICAgICAgICByZWFzb25zLnB1c2goJ2Nvbm5lY3Rpb24gdG8gQVBJIHNlcnZlciBpcyBub3QgZXN0YWJsaXNoZWQgb3IgbG9zdCcpO1xuICAgICAgfVxuICAgICAgaWYgKCFzdGF0dXMuY29ubmVjdGVkVG9Ccm9rZXIpIHtcbiAgICAgICAgcmVhc29ucy5wdXNoKCdjb25uZWN0aW9uIHRvIGJyb2tlciBpcyBub3QgZXN0YWJsaXNoZWQgb3IgbG9zdCcpO1xuICAgICAgfVxuICAgICAgaWYgKCFzdGF0dXMuc3luY2hyb25pemVkKSB7XG4gICAgICAgIHJlYXNvbnMucHVzaCgnbG9jYWwgdGVybWluYWwgc3RhdGUgaXMgbm90IHN5bmNocm9uaXplZCB0byBicm9rZXInKTtcbiAgICAgIH1cbiAgICAgIGlmICghc3RhdHVzLnF1b3RlU3RyZWFtaW5nSGVhbHRoeSkge1xuICAgICAgICByZWFzb25zLnB1c2goJ3F1b3RlcyBhcmUgbm90IHN0cmVhbWVkIGZyb20gdGhlIGJyb2tlciBwcm9wZXJseScpO1xuICAgICAgfVxuICAgICAgbWVzc2FnZSA9IG1lc3NhZ2UgKyByZWFzb25zLmpvaW4oJyBhbmQgJykgKyAnLic7XG4gICAgfVxuICAgIHN0YXR1cy5tZXNzYWdlID0gbWVzc2FnZTtcbiAgICByZXR1cm4gc3RhdHVzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdXB0aW1lIGluIHBlcmNlbnRzIG1lYXN1cmVkIG92ZXIgc3BlY2lmaWMgcGVyaW9kcyBvZiB0aW1lXG4gICAqIEByZXR1cm5zIHtPYmplY3R9IHVwdGltZSBpbiBwZXJjZW50cyBtZWFzdXJlZCBvdmVyIHNwZWNpZmljIHBlcmlvZHMgb2YgdGltZVxuICAgKi9cbiAgZ2V0IHVwdGltZSgpIHtcbiAgICBsZXQgdXB0aW1lID0ge307XG4gICAgZm9yIChsZXQgZSBvZiBPYmplY3QuZW50cmllcyh0aGlzLl91cHRpbWVSZXNlcnZvaXJzKSkge1xuICAgICAgdXB0aW1lW2VbMF1dID0gZVsxXS5nZXRTdGF0aXN0aWNzKCkuYXZlcmFnZTtcbiAgICB9XG4gICAgcmV0dXJuIHVwdGltZTtcbiAgfVxuXG4gIF9tZWFzdXJlVXB0aW1lKCkge1xuICAgIHRyeSB7XG4gICAgICBPYmplY3QudmFsdWVzKHRoaXMuX3VwdGltZVJlc2Vydm9pcnMpLmZvckVhY2gociA9PiByLnB1c2hNZWFzdXJlbWVudCh0aGlzLl9jb25uZWN0aW9uLnRlcm1pbmFsU3RhdGUuY29ubmVjdGVkICYmXG4gICAgICAgIHRoaXMuX2Nvbm5lY3Rpb24udGVybWluYWxTdGF0ZS5jb25uZWN0ZWRUb0Jyb2tlciAmJiB0aGlzLl9jb25uZWN0aW9uLnN5bmNocm9uaXplZCAmJlxuICAgICAgICB0aGlzLl9xdW90ZXNIZWFsdGh5ID8gMTAwIDogMCkpO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWNvbnNvbGVcbiAgICAgIHRoaXMuX2xvZ2dlci5lcnJvcignZmFpbGVkIHRvIG1lYXN1cmUgdXB0aW1lIGZvciBhY2NvdW50ICcgK1xuICAgICAgICB0aGlzLl9jb25uZWN0aW9uLmFjY291bnQuaWQsIGVycik7XG4gICAgfVxuICB9XG5cbiAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGNvbXBsZXhpdHlcbiAgX3VwZGF0ZVF1b3RlSGVhbHRoU3RhdHVzKCkge1xuICAgIHRyeSB7XG4gICAgICBsZXQgc2VydmVyRGF0ZVRpbWUgPSBtb21lbnQobmV3IERhdGUoRGF0ZS5ub3coKSAtIHRoaXMuX29mZnNldCkpO1xuICAgICAgbGV0IHNlcnZlclRpbWUgPSBzZXJ2ZXJEYXRlVGltZS5mb3JtYXQoJ0hIOm1tOnNzLlNTUycpO1xuICAgICAgbGV0IGRheU9mV2VlayA9IHNlcnZlckRhdGVUaW1lLmRheSgpO1xuICAgICAgbGV0IGRheXNPZldlZWsgPSB7XG4gICAgICAgIDA6ICdTVU5EQVknLFxuICAgICAgICAxOiAnTU9OREFZJyxcbiAgICAgICAgMjogJ1RVRVNEQVknLFxuICAgICAgICAzOiAnV0VETkVTREFZJyxcbiAgICAgICAgNDogJ1RIVVJTREFZJyxcbiAgICAgICAgNTogJ0ZSSURBWScsXG4gICAgICAgIDY6ICdTQVRVUkRBWSdcbiAgICAgIH07XG4gICAgICBsZXQgaW5RdW90ZVNlc3Npb24gPSBmYWxzZTtcbiAgICAgIGlmICghdGhpcy5fcHJpY2VVcGRhdGVkQXQpIHtcbiAgICAgICAgdGhpcy5fcHJpY2VVcGRhdGVkQXQgPSBuZXcgRGF0ZSgpO1xuICAgICAgfVxuICAgICAgaWYgKCEodGhpcy5fY29ubmVjdGlvbi5zdWJzY3JpYmVkU3ltYm9scyB8fCBbXSkubGVuZ3RoKSB7XG4gICAgICAgIHRoaXMuX3ByaWNlVXBkYXRlZEF0ID0gbmV3IERhdGUoKTtcbiAgICAgIH1cbiAgICAgIGZvciAobGV0IHN5bWJvbCBvZiB0aGlzLl9jb25uZWN0aW9uLnN1YnNjcmliZWRTeW1ib2xzIHx8IFtdKSB7XG4gICAgICAgIGxldCBzcGVjaWZpY2F0aW9uID0gdGhpcy5fY29ubmVjdGlvbi50ZXJtaW5hbFN0YXRlLnNwZWNpZmljYXRpb24oc3ltYm9sKSB8fCB7fTtcbiAgICAgICAgbGV0IHF1b3RlU2Vzc2lvbnMgPSAoc3BlY2lmaWNhdGlvbi5xdW90ZVNlc3Npb25zIHx8IFtdKVtkYXlzT2ZXZWVrW2RheU9mV2Vla11dIHx8IFtdO1xuICAgICAgICBmb3IgKGxldCBzZXNzaW9uIG9mIHF1b3RlU2Vzc2lvbnMpIHtcbiAgICAgICAgICBpZiAoc2Vzc2lvbi5mcm9tIDw9IHNlcnZlclRpbWUgJiYgc2Vzc2lvbi50byA+PSBzZXJ2ZXJUaW1lKSB7XG4gICAgICAgICAgICBpblF1b3RlU2Vzc2lvbiA9IHRydWU7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgICB0aGlzLl9xdW90ZXNIZWFsdGh5ID0gIXRoaXMuX2Nvbm5lY3Rpb24uc3Vic2NyaWJlZFN5bWJvbHMubGVuZ3RoIHx8ICFpblF1b3RlU2Vzc2lvbiB8fFxuICAgICAgICAoRGF0ZS5ub3coKSAtIHRoaXMuX3ByaWNlVXBkYXRlZEF0LmdldFRpbWUoKSA8IHRoaXMuX21pblF1b3RlSW50ZXJ2YWwpO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWNvbnNvbGVcbiAgICAgIHRoaXMuX2xvZ2dlci5lcnJvcignZmFpbGVkIHRvIHVwZGF0ZSBxdW90ZSBzdHJlYW1pbmcgaGVhbHRoIHN0YXR1cyBmb3IgYWNjb3VudCAnICtcbiAgICAgICAgdGhpcy5fY29ubmVjdGlvbi5hY2NvdW50LmlkLCBlcnIpO1xuICAgIH1cbiAgfVxuXG4gIF9nZXRSYW5kb21UaW1lb3V0KCkge1xuICAgIHJldHVybiAoTWF0aC5yYW5kb20oKSAqIDU5ICsgMSkgKiAxMDAwO1xuICB9XG5cbn1cbiJdLCJuYW1lcyI6WyJDb25uZWN0aW9uSGVhbHRoTW9uaXRvciIsIlN5bmNocm9uaXphdGlvbkxpc3RlbmVyIiwic3RvcCIsIl9sb2dnZXIiLCJkZWJ1ZyIsIl9jb25uZWN0aW9uIiwiYWNjb3VudCIsImlkIiwiY2xlYXJUaW1lb3V0IiwiX3VwZGF0ZVF1b3RlSGVhbHRoU3RhdHVzSW50ZXJ2YWwiLCJfbWVhc3VyZVVwdGltZUludGVydmFsIiwib25TeW1ib2xQcmljZVVwZGF0ZWQiLCJpbnN0YW5jZUluZGV4IiwicHJpY2UiLCJicm9rZXJUaW1lc3RhbXAiLCJtb21lbnQiLCJicm9rZXJUaW1lIiwidG9EYXRlIiwiZ2V0VGltZSIsIl9wcmljZVVwZGF0ZWRBdCIsIkRhdGUiLCJfb2Zmc2V0IiwiZXJyIiwiZXJyb3IiLCJvbkhlYWx0aFN0YXR1cyIsInN0YXR1cyIsIl9zZXJ2ZXJIZWFsdGhTdGF0dXMiLCJvbkRpc2Nvbm5lY3RlZCIsInNlcnZlckhlYWx0aFN0YXR1cyIsInJlc3VsdCIsInMiLCJPYmplY3QiLCJ2YWx1ZXMiLCJmaWVsZCIsImtleXMiLCJoZWFsdGhTdGF0dXMiLCJjb25uZWN0ZWQiLCJ0ZXJtaW5hbFN0YXRlIiwiY29ubmVjdGVkVG9Ccm9rZXIiLCJxdW90ZVN0cmVhbWluZ0hlYWx0aHkiLCJfcXVvdGVzSGVhbHRoeSIsInN5bmNocm9uaXplZCIsImhlYWx0aHkiLCJtZXNzYWdlIiwicmVhc29ucyIsInB1c2giLCJqb2luIiwidXB0aW1lIiwiZSIsImVudHJpZXMiLCJfdXB0aW1lUmVzZXJ2b2lycyIsImdldFN0YXRpc3RpY3MiLCJhdmVyYWdlIiwiX21lYXN1cmVVcHRpbWUiLCJmb3JFYWNoIiwiciIsInB1c2hNZWFzdXJlbWVudCIsIl91cGRhdGVRdW90ZUhlYWx0aFN0YXR1cyIsInNlcnZlckRhdGVUaW1lIiwibm93Iiwic2VydmVyVGltZSIsImZvcm1hdCIsImRheU9mV2VlayIsImRheSIsImRheXNPZldlZWsiLCJpblF1b3RlU2Vzc2lvbiIsInN1YnNjcmliZWRTeW1ib2xzIiwibGVuZ3RoIiwic3ltYm9sIiwic3BlY2lmaWNhdGlvbiIsInF1b3RlU2Vzc2lvbnMiLCJzZXNzaW9uIiwiZnJvbSIsInRvIiwiX21pblF1b3RlSW50ZXJ2YWwiLCJfZ2V0UmFuZG9tVGltZW91dCIsIk1hdGgiLCJyYW5kb20iLCJjb25zdHJ1Y3RvciIsImNvbm5lY3Rpb24iLCJ1cGRhdGVRdW90ZUhlYWx0aFN0YXR1c0ludGVydmFsIiwic2V0VGltZW91dCIsImJpbmQiLCJtZWFzdXJlVXB0aW1lSW50ZXJ2YWwiLCJSZXNlcnZvaXIiLCJMb2dnZXJNYW5hZ2VyIiwiZ2V0TG9nZ2VyIl0sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7OztlQVVxQkE7OztnRkFSZTsrREFDakI7a0VBQ0c7K0RBQ0k7Ozs7OztBQUtYLElBQUEsQUFBTUEsMEJBQU4sTUFBTUEsZ0NBQWdDQyxnQ0FBdUI7SUFrQzFFOztHQUVDLEdBQ0RDLE9BQU87UUFDTCxJQUFJLENBQUNDLE9BQU8sQ0FBQ0MsS0FBSyxDQUFDLENBQUMsRUFBRSxJQUFJLENBQUNDLFdBQVcsQ0FBQ0MsT0FBTyxDQUFDQyxFQUFFLENBQUMsc0JBQXNCLENBQUM7UUFDekVDLGFBQWEsSUFBSSxDQUFDQyxnQ0FBZ0M7UUFDbERELGFBQWEsSUFBSSxDQUFDRSxzQkFBc0I7SUFDMUM7SUFFQTs7OztHQUlDLEdBQ0RDLHFCQUFxQkMsYUFBYSxFQUFFQyxLQUFLLEVBQUU7UUFDekMsSUFBSTtZQUNGLElBQUlDLGtCQUFrQkMsSUFBQUEsZUFBTSxFQUFDRixNQUFNRyxVQUFVLEVBQUVDLE1BQU0sR0FBR0MsT0FBTztZQUMvRCxJQUFJLENBQUNDLGVBQWUsR0FBRyxJQUFJQztZQUMzQixJQUFJLENBQUNDLE9BQU8sR0FBRyxJQUFJLENBQUNGLGVBQWUsQ0FBQ0QsT0FBTyxLQUFLSjtRQUNsRCxFQUFFLE9BQU9RLEtBQUs7WUFDWixzQ0FBc0M7WUFDdEMsSUFBSSxDQUFDbkIsT0FBTyxDQUFDb0IsS0FBSyxDQUFDLENBQUMsRUFBRSxJQUFJLENBQUNsQixXQUFXLENBQUNDLE9BQU8sQ0FBQ0MsRUFBRSxDQUFDLHlCQUF5QixDQUFDLEdBQzFFLDJDQUEyQ2U7UUFDL0M7SUFDRjtJQUVBOzs7OztHQUtDLEdBQ0RFLGVBQWVaLGFBQWEsRUFBRWEsTUFBTSxFQUFFO1FBQ3BDLElBQUksQ0FBQ0MsbUJBQW1CLENBQUMsS0FBS2QsY0FBYyxHQUFHYTtJQUNqRDtJQUVBOzs7O0dBSUMsR0FDREUsZUFBZWYsYUFBYSxFQUFFO1FBQzVCLE9BQU8sSUFBSSxDQUFDYyxtQkFBbUIsQ0FBQyxLQUFLZCxjQUFjO0lBQ3JEO0lBRUE7OztHQUdDLEdBQ0QsSUFBSWdCLHFCQUFxQjtRQUN2QixJQUFJQztRQUNKLEtBQUssSUFBSUMsS0FBS0MsT0FBT0MsTUFBTSxDQUFDLElBQUksQ0FBQ04sbUJBQW1CLEVBQUc7WUFDckQsSUFBSSxDQUFDRyxRQUFRO2dCQUNYQSxTQUFTQztZQUNYLE9BQU87Z0JBQ0wsS0FBSyxJQUFJRyxTQUFTRixPQUFPRyxJQUFJLENBQUNKLEdBQUk7b0JBQ2hDRCxNQUFNLENBQUNJLE1BQU0sR0FBR0osTUFBTSxDQUFDSSxNQUFNLElBQUlILENBQUMsQ0FBQ0csTUFBTTtnQkFDM0M7WUFDRjtRQUNGO1FBQ0EsT0FBT0osVUFBVSxDQUFDO0lBQ3BCO0lBRUE7Ozs7Ozs7Ozs7R0FVQyxHQUVEOzs7R0FHQyxHQUNELHNDQUFzQztJQUN0QyxJQUFJTSxlQUFlO1FBQ2pCLElBQUlWLFNBQVM7WUFDWFcsV0FBVyxJQUFJLENBQUMvQixXQUFXLENBQUNnQyxhQUFhLENBQUNELFNBQVM7WUFDbkRFLG1CQUFtQixJQUFJLENBQUNqQyxXQUFXLENBQUNnQyxhQUFhLENBQUNDLGlCQUFpQjtZQUNuRUMsdUJBQXVCLElBQUksQ0FBQ0MsY0FBYztZQUMxQ0MsY0FBYyxJQUFJLENBQUNwQyxXQUFXLENBQUNvQyxZQUFZO1FBQzdDO1FBQ0FoQixPQUFPaUIsT0FBTyxHQUFHakIsT0FBT1csU0FBUyxJQUFJWCxPQUFPYSxpQkFBaUIsSUFBSWIsT0FBT2MscUJBQXFCLElBQzNGZCxPQUFPZ0IsWUFBWTtRQUNyQixJQUFJRTtRQUNKLElBQUlsQixPQUFPaUIsT0FBTyxFQUFFO1lBQ2xCQyxVQUFVO1FBQ1osT0FBTztZQUNMQSxVQUFVO1lBQ1YsSUFBSUMsVUFBVSxFQUFFO1lBQ2hCLElBQUksQ0FBQ25CLE9BQU9XLFNBQVMsRUFBRTtnQkFDckJRLFFBQVFDLElBQUksQ0FBQztZQUNmO1lBQ0EsSUFBSSxDQUFDcEIsT0FBT2EsaUJBQWlCLEVBQUU7Z0JBQzdCTSxRQUFRQyxJQUFJLENBQUM7WUFDZjtZQUNBLElBQUksQ0FBQ3BCLE9BQU9nQixZQUFZLEVBQUU7Z0JBQ3hCRyxRQUFRQyxJQUFJLENBQUM7WUFDZjtZQUNBLElBQUksQ0FBQ3BCLE9BQU9jLHFCQUFxQixFQUFFO2dCQUNqQ0ssUUFBUUMsSUFBSSxDQUFDO1lBQ2Y7WUFDQUYsVUFBVUEsVUFBVUMsUUFBUUUsSUFBSSxDQUFDLFdBQVc7UUFDOUM7UUFDQXJCLE9BQU9rQixPQUFPLEdBQUdBO1FBQ2pCLE9BQU9sQjtJQUNUO0lBRUE7OztHQUdDLEdBQ0QsSUFBSXNCLFNBQVM7UUFDWCxJQUFJQSxTQUFTLENBQUM7UUFDZCxLQUFLLElBQUlDLEtBQUtqQixPQUFPa0IsT0FBTyxDQUFDLElBQUksQ0FBQ0MsaUJBQWlCLEVBQUc7WUFDcERILE1BQU0sQ0FBQ0MsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxHQUFHQSxDQUFDLENBQUMsRUFBRSxDQUFDRyxhQUFhLEdBQUdDLE9BQU87UUFDN0M7UUFDQSxPQUFPTDtJQUNUO0lBRUFNLGlCQUFpQjtRQUNmLElBQUk7WUFDRnRCLE9BQU9DLE1BQU0sQ0FBQyxJQUFJLENBQUNrQixpQkFBaUIsRUFBRUksT0FBTyxDQUFDQyxDQUFBQSxJQUFLQSxFQUFFQyxlQUFlLENBQUMsSUFBSSxDQUFDbkQsV0FBVyxDQUFDZ0MsYUFBYSxDQUFDRCxTQUFTLElBQzNHLElBQUksQ0FBQy9CLFdBQVcsQ0FBQ2dDLGFBQWEsQ0FBQ0MsaUJBQWlCLElBQUksSUFBSSxDQUFDakMsV0FBVyxDQUFDb0MsWUFBWSxJQUNqRixJQUFJLENBQUNELGNBQWMsR0FBRyxNQUFNO1FBQ2hDLEVBQUUsT0FBT2xCLEtBQUs7WUFDWixzQ0FBc0M7WUFDdEMsSUFBSSxDQUFDbkIsT0FBTyxDQUFDb0IsS0FBSyxDQUFDLDBDQUNqQixJQUFJLENBQUNsQixXQUFXLENBQUNDLE9BQU8sQ0FBQ0MsRUFBRSxFQUFFZTtRQUNqQztJQUNGO0lBRUEsc0NBQXNDO0lBQ3RDbUMsMkJBQTJCO1FBQ3pCLElBQUk7WUFDRixJQUFJQyxpQkFBaUIzQyxJQUFBQSxlQUFNLEVBQUMsSUFBSUssS0FBS0EsS0FBS3VDLEdBQUcsS0FBSyxJQUFJLENBQUN0QyxPQUFPO1lBQzlELElBQUl1QyxhQUFhRixlQUFlRyxNQUFNLENBQUM7WUFDdkMsSUFBSUMsWUFBWUosZUFBZUssR0FBRztZQUNsQyxJQUFJQyxhQUFhO2dCQUNmLEdBQUc7Z0JBQ0gsR0FBRztnQkFDSCxHQUFHO2dCQUNILEdBQUc7Z0JBQ0gsR0FBRztnQkFDSCxHQUFHO2dCQUNILEdBQUc7WUFDTDtZQUNBLElBQUlDLGlCQUFpQjtZQUNyQixJQUFJLENBQUMsSUFBSSxDQUFDOUMsZUFBZSxFQUFFO2dCQUN6QixJQUFJLENBQUNBLGVBQWUsR0FBRyxJQUFJQztZQUM3QjtZQUNBLElBQUksQ0FBQyxBQUFDLENBQUEsSUFBSSxDQUFDZixXQUFXLENBQUM2RCxpQkFBaUIsSUFBSSxFQUFFLEFBQUQsRUFBR0MsTUFBTSxFQUFFO2dCQUN0RCxJQUFJLENBQUNoRCxlQUFlLEdBQUcsSUFBSUM7WUFDN0I7WUFDQSxLQUFLLElBQUlnRCxVQUFVLElBQUksQ0FBQy9ELFdBQVcsQ0FBQzZELGlCQUFpQixJQUFJLEVBQUUsQ0FBRTtnQkFDM0QsSUFBSUcsZ0JBQWdCLElBQUksQ0FBQ2hFLFdBQVcsQ0FBQ2dDLGFBQWEsQ0FBQ2dDLGFBQWEsQ0FBQ0QsV0FBVyxDQUFDO2dCQUM3RSxJQUFJRSxnQkFBZ0IsQUFBQ0QsQ0FBQUEsY0FBY0MsYUFBYSxJQUFJLEVBQUUsQUFBRCxDQUFFLENBQUNOLFVBQVUsQ0FBQ0YsVUFBVSxDQUFDLElBQUksRUFBRTtnQkFDcEYsS0FBSyxJQUFJUyxXQUFXRCxjQUFlO29CQUNqQyxJQUFJQyxRQUFRQyxJQUFJLElBQUlaLGNBQWNXLFFBQVFFLEVBQUUsSUFBSWIsWUFBWTt3QkFDMURLLGlCQUFpQjtvQkFDbkI7Z0JBQ0Y7WUFDRjtZQUNBLElBQUksQ0FBQ3pCLGNBQWMsR0FBRyxDQUFDLElBQUksQ0FBQ25DLFdBQVcsQ0FBQzZELGlCQUFpQixDQUFDQyxNQUFNLElBQUksQ0FBQ0Ysa0JBQ2xFN0MsS0FBS3VDLEdBQUcsS0FBSyxJQUFJLENBQUN4QyxlQUFlLENBQUNELE9BQU8sS0FBSyxJQUFJLENBQUN3RCxpQkFBaUI7UUFDekUsRUFBRSxPQUFPcEQsS0FBSztZQUNaLHNDQUFzQztZQUN0QyxJQUFJLENBQUNuQixPQUFPLENBQUNvQixLQUFLLENBQUMsZ0VBQ2pCLElBQUksQ0FBQ2xCLFdBQVcsQ0FBQ0MsT0FBTyxDQUFDQyxFQUFFLEVBQUVlO1FBQ2pDO0lBQ0Y7SUFFQXFELG9CQUFvQjtRQUNsQixPQUFPLEFBQUNDLENBQUFBLEtBQUtDLE1BQU0sS0FBSyxLQUFLLENBQUEsSUFBSztJQUNwQztJQW5OQTs7O0dBR0MsR0FDREMsWUFBWUMsVUFBVSxDQUFFO1FBQ3RCLEtBQUs7UUFDTCxJQUFJLENBQUMxRSxXQUFXLEdBQUcwRTtRQUNuQixNQUFNQyxrQ0FBa0M7WUFDdEMsSUFBSSxDQUFDdkIsd0JBQXdCO1lBQzdCLElBQUksQ0FBQ2hELGdDQUFnQyxHQUFHd0UsV0FBV0QsZ0NBQWdDRSxJQUFJLENBQUMsSUFBSSxHQUMxRixJQUFJLENBQUNQLGlCQUFpQjtRQUMxQjtRQUNBLElBQUksQ0FBQ2xFLGdDQUFnQyxHQUFHd0UsV0FBV0QsZ0NBQWdDRSxJQUFJLENBQUMsSUFBSSxHQUMxRixJQUFJLENBQUNQLGlCQUFpQjtRQUN4QixNQUFNUSx3QkFBd0I7WUFDNUIsSUFBSSxDQUFDOUIsY0FBYztZQUNuQixJQUFJLENBQUMzQyxzQkFBc0IsR0FBR3VFLFdBQVdFLHNCQUFzQkQsSUFBSSxDQUFDLElBQUksR0FDdEUsSUFBSSxDQUFDUCxpQkFBaUI7UUFDMUI7UUFDQSxJQUFJLENBQUNqRSxzQkFBc0IsR0FBR3VFLFdBQVdFLHNCQUFzQkQsSUFBSSxDQUFDLElBQUksR0FDdEUsSUFBSSxDQUFDUCxpQkFBaUI7UUFDeEIsSUFBSSxDQUFDRCxpQkFBaUIsR0FBRztRQUN6QixJQUFJLENBQUNoRCxtQkFBbUIsR0FBRyxDQUFDO1FBQzVCLElBQUksQ0FBQ3dCLGlCQUFpQixHQUFHO1lBQ3ZCLE1BQU0sSUFBSWtDLGtCQUFTLENBQUMsS0FBSyxJQUFJLEtBQUs7WUFDbEMsTUFBTSxJQUFJQSxrQkFBUyxDQUFDLEtBQUssS0FBSyxLQUFLO1lBQ25DLE1BQU0sSUFBSUEsa0JBQVMsQ0FBQyxLQUFLLElBQUksS0FBSyxLQUFLLEtBQUs7WUFDNUMsTUFBTSxJQUFJQSxrQkFBUyxDQUFDLEtBQUssR0FBRyxJQUFJLEtBQUssS0FBSyxLQUFLO1FBQ2pEO1FBQ0EsSUFBSSxDQUFDakYsT0FBTyxHQUFHa0YsZUFBYSxDQUFDQyxTQUFTLENBQUM7SUFDekM7QUF1TEYifQ==