"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return MetaApiWebsocketClient;
    }
});
const _randomstring = /*#__PURE__*/ _interop_require_default(require("randomstring"));
const _socketioclient = /*#__PURE__*/ _interop_require_default(require("socket.io-client"));
const _timeoutError = /*#__PURE__*/ _interop_require_default(require("../timeoutError"));
const _errorHandler = require("../errorHandler");
const _optionsValidator = /*#__PURE__*/ _interop_require_default(require("../optionsValidator"));
const _notSynchronizedError = /*#__PURE__*/ _interop_require_default(require("./notSynchronizedError"));
const _notConnectedError = /*#__PURE__*/ _interop_require_default(require("./notConnectedError"));
const _tradeError = /*#__PURE__*/ _interop_require_default(require("./tradeError"));
const _packetOrderer = /*#__PURE__*/ _interop_require_default(require("./packetOrderer"));
const _synchronizationThrottler = /*#__PURE__*/ _interop_require_default(require("./synchronizationThrottler"));
const _subscriptionManager = /*#__PURE__*/ _interop_require_default(require("./subscriptionManager"));
const _logger = /*#__PURE__*/ _interop_require_default(require("../../logger"));
const _promiseany = /*#__PURE__*/ _interop_require_default(require("promise.any"));
const _latencyService = /*#__PURE__*/ _interop_require_default(require("./latencyService"));
const _metaApiWebsocketclient = require("./metaApiWebsocket.client");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let PacketLogger;
if (typeof window === "undefined") {
    PacketLogger = require("./packetLogger").default;
}
let MetaApiWebsocketClient = class MetaApiWebsocketClient {
    /**
   * Restarts the account synchronization process on an out of order packet
   * @param {String} accountId account id
   * @param {Number} instanceIndex instance index
   * @param {Number} expectedSequenceNumber expected s/n
   * @param {Number} actualSequenceNumber actual s/n
   * @param {Object} packet packet data
   * @param {Date} receivedAt time the packet was received at
   */ onOutOfOrderPacket(accountId, instanceIndex, expectedSequenceNumber, actualSequenceNumber, packet, receivedAt) {
        const primaryAccountId = this._accountsByReplicaId[accountId];
        if (this._subscriptionManager.isSubscriptionActive(accountId)) {
            const level = this._latencyService.getSynchronizedAccountInstances(primaryAccountId).length ? "debug" : "error";
            this._logger[level]("MetaApi websocket client received an out of order " + `packet type ${packet.type} for account id ${accountId}:${instanceIndex}. Expected s/n ` + `${expectedSequenceNumber} does not match the actual of ${actualSequenceNumber}`);
            this.ensureSubscribe(accountId, instanceIndex);
        }
    }
    /**
   * Patch server URL for use in unit tests
   * @param {String} url patched server URL
   */ set url(url) {
        this._url = url;
    }
    /**
   * Websocket client predefined region
   * @returns {String} predefined region
   */ get region() {
        return this._region;
    }
    /**
   * Returns the list of socket instance dictionaries
   * @return {Object[]} list of socket instance dictionaries
   */ get socketInstances() {
        return this._socketInstances;
    }
    /**
   * Returns the dictionary of socket instances by account ids
   * @return {Object} dictionary of socket instances by account ids
   */ get socketInstancesByAccounts() {
        return this._socketInstancesByAccounts;
    }
    /**
   * Returns the dictionary of account replicas by region
   * @return {Object} dictionary of account replicas by region
   */ get accountReplicas() {
        return this._accountReplicas;
    }
    /**
   * Returns the dictionary of primary account ids by replica ids
   * @return {Object} dictionary of primary account ids by replica ids
   */ get accountsByReplicaId() {
        return this._accountsByReplicaId;
    }
    /**
   * Returns clear account cache job. Used for tests
   * @return {Function} clear account cache job
   */ get clearAccountCacheJob() {
        return this._clearAccountCacheJob.bind(this);
    }
    /**
   * Returns latency service
   * @returns {LatencyService} latency service
   */ get latencyService() {
        return this._latencyService;
    }
    /**
   * Returns the list of subscribed account ids
   * @param {Number} instanceNumber instance index number
   * @param {String} socketInstanceIndex socket instance index
   * @param {String} region server region
   * @return {string[]} list of subscribed account ids
   */ subscribedAccountIds(instanceNumber, socketInstanceIndex, region) {
        const connectedIds = [];
        if (this._socketInstancesByAccounts[instanceNumber]) {
            Object.keys(this._connectedHosts).forEach((instanceId)=>{
                const accountId = instanceId.split(":")[0];
                const accountRegion = this.getAccountRegion(accountId);
                if (!connectedIds.includes(accountId) && this._socketInstancesByAccounts[instanceNumber][accountId] !== undefined && (this._socketInstancesByAccounts[instanceNumber][accountId] === socketInstanceIndex || socketInstanceIndex === undefined) && accountRegion === region) {
                    connectedIds.push(accountId);
                }
            });
        }
        return connectedIds;
    }
    /**
   * Returns websocket client connection status
   * @param {Number} instanceNumber instance index number
   * @param {Number} socketInstanceIndex socket instance index
   * @param {String} region server region
   * @returns {Boolean} websocket client connection status
   */ connected(instanceNumber, socketInstanceIndex, region) {
        const instance = this._socketInstances[region] && this._socketInstances[region][instanceNumber].length > socketInstanceIndex ? this._socketInstances[region][instanceNumber][socketInstanceIndex] : null;
        return instance && instance.socket && instance.socket.connected || false;
    }
    /**
   * Returns list of accounts assigned to instance
   * @param {Number} instanceNumber instance index number
   * @param {String} socketInstanceIndex socket instance index
   * @param {String} region server region
   * @returns 
   */ getAssignedAccounts(instanceNumber, socketInstanceIndex, region) {
        const accountIds = [];
        Object.keys(this._socketInstancesByAccounts[instanceNumber]).forEach((key)=>{
            const accountRegion = this.getAccountRegion(key);
            if (accountRegion === region && this._socketInstancesByAccounts[instanceNumber][key] === socketInstanceIndex) {
                accountIds.push(key);
            }
        });
        return accountIds;
    }
    /**
   * Returns account region by id
   * @param {String} accountId account id
   * @returns {String} account region
   */ getAccountRegion(accountId) {
        return this._regionsByAccounts[accountId] && this._regionsByAccounts[accountId].region;
    }
    /**
   * Adds account cache info
   * @param {String} accountId account id
   * @param {Object} replicas account replicas
   */ addAccountCache(accountId, replicas) {
        this._accountReplicas[accountId] = replicas;
        Object.keys(replicas).forEach((region)=>{
            const replicaId = replicas[region];
            if (!this._regionsByAccounts[replicaId]) {
                this._regionsByAccounts[replicaId] = {
                    region,
                    connections: 1,
                    lastUsed: Date.now()
                };
            } else {
                this._regionsByAccounts[replicaId].connections++;
            }
            this._accountsByReplicaId[replicaId] = accountId;
        });
        this._logger.debug(`${accountId}: added account cache`);
    }
    /**
   * Updates account cache info
   * @param {String} accountId account id
   * @param {Object} replicas account replicas
   */ updateAccountCache(accountId, replicas) {
        const oldReplicas = this._accountReplicas[accountId];
        if (oldReplicas) {
            const connectionCount = this._regionsByAccounts[accountId].connections;
            Object.keys(oldReplicas).forEach((region)=>{
                const replicaId = replicas[region];
                delete this._accountsByReplicaId[replicaId];
                delete this._regionsByAccounts[replicaId];
            });
            this._accountReplicas[accountId] = replicas;
            Object.keys(replicas).forEach((region)=>{
                const replicaId = replicas[region];
                this._regionsByAccounts[replicaId] = {
                    region,
                    connections: connectionCount,
                    lastUsed: Date.now()
                };
                this._accountsByReplicaId[replicaId] = accountId;
            });
            this._logger.debug(`${accountId}: updated account cache`);
        }
    }
    /**
   * Removes account region info
   * @param {String} accountId account id
   */ removeAccountCache(accountId) {
        if (this._regionsByAccounts[accountId]) {
            if (this._regionsByAccounts[accountId].connections > 0) {
                this._regionsByAccounts[accountId].connections--;
            }
        }
    }
    /**
   * Locks subscription for a socket instance based on TooManyRequestsError metadata
   * @param {Number} instanceNumber instance index number
   * @param {String} socketInstanceIndex socket instance index
   * @param {String} region server region
   * @param {Object} metadata TooManyRequestsError metadata
   */ async lockSocketInstance(instanceNumber, socketInstanceIndex, region, metadata) {
        if (metadata.type === "LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_USER") {
            this._subscribeLock = {
                recommendedRetryTime: metadata.recommendedRetryTime,
                lockedAtAccounts: this.subscribedAccountIds(instanceNumber, undefined, region).length,
                lockedAtTime: Date.now()
            };
        } else {
            const subscribedAccounts = this.subscribedAccountIds(instanceNumber, socketInstanceIndex, region);
            if (subscribedAccounts.length === 0) {
                const socketInstance = this.socketInstances[region][instanceNumber][socketInstanceIndex];
                socketInstance.socket.close();
                await this._reconnect(instanceNumber, socketInstanceIndex, region);
            } else {
                const instance = this.socketInstances[region][instanceNumber][socketInstanceIndex];
                instance.subscribeLock = {
                    recommendedRetryTime: metadata.recommendedRetryTime,
                    type: metadata.type,
                    lockedAtAccounts: subscribedAccounts.length
                };
            }
        }
    }
    /**
   * Connects to MetaApi server via socket.io protocol
   * @param {Number} instanceNumber instance index number
   * @param {String} region server region
   * @returns {Promise} promise which resolves when connection is established
   */ async connect(instanceNumber, region) {
        var _this__socketInstances, _region, _this__socketInstances_region, _instanceNumber;
        if (this._region && region !== this._region) {
            throw new _errorHandler.ValidationError(`Trying to connect to ${region} region, but configured with ${this._region}`);
        }
        let clientId = Math.random();
        let resolve;
        let result = new Promise((res, rej)=>{
            resolve = res;
        });
        (_this__socketInstances = this._socketInstances)[_region = region] || (_this__socketInstances[_region] = {});
        (_this__socketInstances_region = this._socketInstances[region])[_instanceNumber = instanceNumber] || (_this__socketInstances_region[_instanceNumber] = []);
        const socketInstanceIndex = this._socketInstances[region][instanceNumber].length;
        const instance = {
            id: socketInstanceIndex,
            reconnectWaitTime: this._socketMinimumReconnectTimeout,
            connected: false,
            requestResolves: {},
            resolved: false,
            connectResult: result,
            sessionId: _randomstring.default.generate(32),
            isReconnecting: false,
            socket: null,
            synchronizationThrottler: new _synchronizationThrottler.default(this, socketInstanceIndex, instanceNumber, region, this._synchronizationThrottlerOpts),
            subscribeLock: null,
            instanceNumber
        };
        instance.connected = true;
        this._socketInstances[region][instanceNumber].push(instance);
        instance.synchronizationThrottler.start();
        const serverUrl = await this._getServerUrl(instanceNumber, socketInstanceIndex, region);
        const socketInstance = (0, _socketioclient.default)(serverUrl, {
            path: "/ws",
            reconnection: true,
            reconnectionDelay: 1000,
            reconnectionDelayMax: 5000,
            reconnectionAttempts: Infinity,
            timeout: this._connectTimeout,
            extraHeaders: {
                "Client-Id": clientId
            },
            query: {
                "auth-token": this._token,
                clientId: clientId,
                protocol: 3
            }
        });
        instance.socket = socketInstance;
        socketInstance.on("connect", async ()=>{
            // eslint-disable-next-line no-console
            this._logger.info(`${region}:${instanceNumber}: MetaApi websocket client connected to the MetaApi server`);
            instance.reconnectWaitTime = this._socketMinimumReconnectTimeout;
            instance.isReconnecting = false;
            if (!instance.resolved) {
                instance.resolved = true;
                resolve();
            } else {
                await this._fireReconnected(instanceNumber, instance.id, region);
            }
            if (!instance.connected) {
                instance.socket.close();
            }
        });
        socketInstance.on("reconnect", async ()=>{
            instance.isReconnecting = false;
            this._logger.info(`${region}:${instanceNumber}: MetaApi websocket client reconnected`);
            await this._fireReconnected(instanceNumber, instance.id, region);
        });
        socketInstance.on("connect_error", async (err)=>{
            // eslint-disable-next-line no-console
            this._logger.error(`${region}:${instanceNumber}: MetaApi websocket client connection error`, err);
            instance.isReconnecting = false;
            await this._reconnect(instanceNumber, instance.id, region);
        });
        socketInstance.on("connect_timeout", async (timeout)=>{
            // eslint-disable-next-line no-console
            this._logger.error(`${region}:${instanceNumber}: MetaApi websocket client connection timeout`);
            instance.isReconnecting = false;
            if (!instance.resolved) {
                await this._reconnect(instanceNumber, instance.id, region);
            }
        });
        socketInstance.on("disconnect", async (reason)=>{
            instance.synchronizationThrottler.onDisconnect();
            // eslint-disable-next-line no-console
            this._logger.info(`${region}:${instanceNumber}: MetaApi websocket client disconnected from the ` + `MetaApi server because of ${reason}`);
            instance.isReconnecting = false;
            await this._reconnect(instanceNumber, instance.id, region);
        });
        socketInstance.on("error", async (error)=>{
            // eslint-disable-next-line no-console
            this._logger.error(`${region}:${instanceNumber}: MetaApi websocket client error`, error);
            instance.isReconnecting = false;
            await this._reconnect(instanceNumber, instance.id, region);
        });
        socketInstance.on("response", (data)=>{
            if (typeof data === "string") {
                data = JSON.parse(data);
            }
            this._logger.debug(()=>`${data.accountId}: Response received: ${JSON.stringify({
                    requestId: data.requestId,
                    timestamps: data.timestamps
                })}`);
            let requestResolve = instance.requestResolves[data.requestId] || {
                resolve: ()=>{},
                reject: ()=>{}
            };
            delete instance.requestResolves[data.requestId];
            this._convertIsoTimeToDate(data);
            requestResolve.resolve(data);
            if (data.timestamps && requestResolve.type) {
                data.timestamps.clientProcessingFinished = new Date();
                for (let listener of this._latencyListeners){
                    Promise.resolve().then(()=>requestResolve.type === "trade" ? listener.onTrade(data.accountId, data.timestamps) : listener.onResponse(data.accountId, requestResolve.type, data.timestamps)).catch((error)=>this._logger.error("Failed to process onResponse event for account " + data.accountId + ", request type " + requestResolve.type, error));
                }
            }
        });
        socketInstance.on("processingError", (data)=>{
            let requestResolve = instance.requestResolves[data.requestId] || {
                resolve: ()=>{},
                reject: ()=>{}
            };
            delete instance.requestResolves[data.requestId];
            requestResolve.reject(this._convertError(data));
        });
        // eslint-disable-next-line complexity
        socketInstance.on("synchronization", async (data)=>{
            if (typeof data === "string") {
                data = JSON.parse(data);
            }
            if (data.instanceIndex && data.instanceIndex !== instanceNumber) {
                this._logger.trace(()=>`${data.accountId}:${data.instanceNumber}: received packet with wrong instance ` + `index via a socket with instance number of ${instanceNumber}, data=${JSON.stringify({
                        type: data.type,
                        sequenceNumber: data.sequenceNumber,
                        sequenceTimestamp: data.sequenceTimestamp,
                        synchronizationId: data.synchronizationId,
                        application: data.application,
                        host: data.host,
                        specificationsUpdated: data.specificationsUpdated,
                        positionsUpdated: data.positionsUpdated,
                        ordersUpdated: data.ordersUpdated,
                        specifications: data.specifications ? (data.specifications || []).length : undefined
                    })}`);
                return;
            }
            if (!this._regionsByAccounts[data.accountId]) {
                this._regionsByAccounts[data.accountId] = {
                    region,
                    connections: 0,
                    lastUsed: Date.now()
                };
            }
            this._logger.trace(()=>`${data.accountId}:${data.instanceIndex}: Sync packet received: ${JSON.stringify({
                    type: data.type,
                    sequenceNumber: data.sequenceNumber,
                    sequenceTimestamp: data.sequenceTimestamp,
                    synchronizationId: data.synchronizationId,
                    application: data.application,
                    host: data.host,
                    specificationsUpdated: data.specificationsUpdated,
                    positionsUpdated: data.positionsUpdated,
                    ordersUpdated: data.ordersUpdated,
                    specifications: data.specifications ? (data.specifications || []).length : undefined
                })}, ` + `active listeners: ${(this._synchronizationListeners[data.accountId] || []).length}`);
            let activeSynchronizationIds = instance.synchronizationThrottler.activeSynchronizationIds;
            if (!data.synchronizationId || activeSynchronizationIds.includes(data.synchronizationId)) {
                if (this._packetLogger) {
                    await this._packetLogger.logPacket(data);
                }
                const ignoredPacketTypes = [
                    "disconnected",
                    "status",
                    "keepalive"
                ];
                if (!this._subscriptionManager.isSubscriptionActive(data.accountId) && !ignoredPacketTypes.includes(data.type)) {
                    this._logger.debug(`${data.accountId}: Packet arrived to inactive connection, attempting` + ` unsubscribe, packet: ${data.type}`);
                    if (this._throttleRequest("unsubscribe", data.accountId, data.instanceIndex, this._unsubscribeThrottlingInterval)) {
                        this.unsubscribe(data.accountId).catch((err)=>{
                            this._logger.warn(`${data.accountId}:${data.instanceIndex || 0}: failed to unsubscribe`, err);
                        });
                    }
                    return;
                }
                this._convertIsoTimeToDate(data);
            } else {
                data.type = "noop";
            }
            this.queuePacket(data);
        });
        return result;
    }
    /**
   * Closes connection to MetaApi server
   */ close() {
        Object.keys(this._socketInstances).forEach((region)=>{
            Object.keys(this._socketInstances[region]).forEach((instanceNumber)=>{
                this._socketInstances[region][instanceNumber].forEach(async (instance)=>{
                    if (instance.connected) {
                        instance.connected = false;
                        await instance.socket.close();
                        for (let requestResolve of Object.values(instance.requestResolves)){
                            requestResolve.reject(new Error("MetaApi connection closed"));
                        }
                        instance.requestResolves = {};
                    }
                });
                this._socketInstancesByAccounts[instanceNumber] = {};
                this._socketInstances[region][instanceNumber] = [];
            });
        });
        this._synchronizationListeners = {};
        this._latencyListeners = [];
        this._packetOrderer.stop();
    }
    /**
   * Stops the client
   */ stop() {
        clearInterval(this._clearAccountCacheInterval);
        clearInterval(this._clearInactiveSyncDataJob);
        this._latencyService.stop();
    }
    /**
   * Returns account information for a specified MetaTrader account.
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {GetAccountInformationOptions} [options] additional request options
   * @returns {Promise<MetatraderAccountInformation>} promise resolving with account information
   */ async getAccountInformation(accountId, options) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getAccountInformation",
            ...options
        });
        return response.accountInformation;
    }
    /**
   * Returns positions for a specified MetaTrader account.
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {GetPositionsOptions} [options] additional request options
   * @returns {Promise<Array<MetatraderPosition>} promise resolving with array of open positions
   */ async getPositions(accountId, options) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getPositions",
            ...options
        });
        return response.positions;
    }
    /**
   * Returns specific position for a MetaTrader account.
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} positionId position id
   * @param {GetPositionOptions} [options] additional request options
   * @return {Promise<MetatraderPosition>} promise resolving with MetaTrader position found
   */ async getPosition(accountId, positionId, options) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getPosition",
            positionId,
            ...options
        });
        return response.position;
    }
    /**
   * Returns open orders for a specified MetaTrader account.
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {GetOrdersOptions} [options] additional request options
   * @return {Promise<Array<MetatraderOrder>>} promise resolving with open MetaTrader orders
   */ async getOrders(accountId, options) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getOrders",
            ...options
        });
        return response.orders;
    }
    /**
   * Returns specific open order for a MetaTrader account.
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} orderId order id (ticket number)
   * @param {GetOrderOptions} [options] additional request options
   * @return {Promise<MetatraderOrder>} promise resolving with metatrader order found
   */ async getOrder(accountId, orderId, options) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getOrder",
            orderId,
            ...options
        });
        return response.order;
    }
    /**
   * MetaTrader history orders search query response
   * @typedef {Object} MetatraderHistoryOrders
   * @property {Array<MetatraderOrder>} historyOrders array of history orders returned
   * @property {Boolean} synchronizing flag indicating that history order initial synchronization is still in progress
   * and thus search results may be incomplete
   */ /**
   * Returns the history of completed orders for a specific ticket number.
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} ticket ticket number (order id)
   * @returns {Promise<MetatraderHistoryOrders>} promise resolving with request results containing history orders found
   */ async getHistoryOrdersByTicket(accountId, ticket) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getHistoryOrdersByTicket",
            ticket
        });
        return {
            historyOrders: response.historyOrders,
            synchronizing: response.synchronizing
        };
    }
    /**
   * Returns the history of completed orders for a specific position id
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} positionId position id
   * @returns {Promise<MetatraderHistoryOrders>} promise resolving with request results containing history orders found
   */ async getHistoryOrdersByPosition(accountId, positionId) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getHistoryOrdersByPosition",
            positionId
        });
        return {
            historyOrders: response.historyOrders,
            synchronizing: response.synchronizing
        };
    }
    /**
   * Returns the history of completed orders for a specific time range
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {Date} startTime start of time range, inclusive
   * @param {Date} endTime end of time range, exclusive
   * @param {Number} offset pagination offset, default is 0
   * @param {Number} limit pagination limit, default is 1000
   * @returns {Promise<MetatraderHistoryOrders>} promise resolving with request results containing history orders found
   */ async getHistoryOrdersByTimeRange(accountId, startTime, endTime, offset = 0, limit = 1000) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getHistoryOrdersByTimeRange",
            startTime,
            endTime,
            offset,
            limit
        });
        return {
            historyOrders: response.historyOrders,
            synchronizing: response.synchronizing
        };
    }
    /**
   * MetaTrader history deals search query response
   * @typedef {Object} MetatraderDeals
   * @property {Array<MetatraderDeal>} deals array of history deals returned
   * @property {Boolean} synchronizing flag indicating that deal initial synchronization is still in progress
   * and thus search results may be incomplete
   */ /**
   * MetaTrader deal
   * @typedef {Object} MetatraderDeal
   * @property {String} id deal id (ticket number)
   * @property {String} type deal type (one of DEAL_TYPE_BUY, DEAL_TYPE_SELL, DEAL_TYPE_BALANCE, DEAL_TYPE_CREDIT,
   * DEAL_TYPE_CHARGE, DEAL_TYPE_CORRECTION, DEAL_TYPE_BONUS, DEAL_TYPE_COMMISSION, DEAL_TYPE_COMMISSION_DAILY,
   * DEAL_TYPE_COMMISSION_MONTHLY, DEAL_TYPE_COMMISSION_AGENT_DAILY, DEAL_TYPE_COMMISSION_AGENT_MONTHLY,
   * DEAL_TYPE_INTEREST, DEAL_TYPE_BUY_CANCELED, DEAL_TYPE_SELL_CANCELED, DEAL_DIVIDEND, DEAL_DIVIDEND_FRANKED,
   * DEAL_TAX). See https://www.mql5.com/en/docs/constants/tradingconstants/dealproperties#enum_deal_type
   * @property {String} entryType deal entry type (one of DEAL_ENTRY_IN, DEAL_ENTRY_OUT, DEAL_ENTRY_INOUT,
   * DEAL_ENTRY_OUT_BY). See https://www.mql5.com/en/docs/constants/tradingconstants/dealproperties#enum_deal_entry
   * @property {String} [symbol] symbol deal relates to
   * @property {Number} [magic] deal magic number, identifies the EA which initiated the deal
   * @property {Date} time time the deal was conducted at
   * @property {String} brokerTime time time the deal was conducted at, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {Number} [volume] deal volume
   * @property {Number} [price] the price the deal was conducted at
   * @property {Number} [commission] deal commission
   * @property {Number} [swap] deal swap
   * @property {Number} profit deal profit
   * @property {String} [positionId] id of position the deal relates to
   * @property {String} [orderId] id of order the deal relates to
   * @property {String} [comment] deal comment. The sum of the line lengths of the comment and the clientId
   * must be less than or equal to 26. For more information see https://metaapi.cloud/docs/client/clientIdUsage/
   * @property {String} [brokerComment] current comment value on broker side (possibly overriden by the broker)
   * @property {String} [clientId] client-assigned id. The id value can be assigned when submitting a trade and
   * will be present on position, history orders and history deals related to the trade. You can use this field to bind
   * your trades to objects in your application and then track trade progress. The sum of the line lengths of the
   * comment and the clientId must be less than or equal to 26. For more information see
   * https://metaapi.cloud/docs/client/clientIdUsage/
   * @property {String} platform platform id (mt4 or mt5)
   * @property {String} [reason] optional deal execution reason. One of DEAL_REASON_CLIENT, DEAL_REASON_MOBILE,
   * DEAL_REASON_WEB, DEAL_REASON_EXPERT, DEAL_REASON_SL, DEAL_REASON_TP, DEAL_REASON_SO, DEAL_REASON_ROLLOVER,
   * DEAL_REASON_VMARGIN, DEAL_REASON_SPLIT, DEAL_REASON_UNKNOWN. See
   * https://www.mql5.com/en/docs/constants/tradingconstants/dealproperties#enum_deal_reason.
   * @property {Number} [accountCurrencyExchangeRate] current exchange rate of account currency into account base
   * currency (USD if you did not override it)
   * @property {number} [stopLoss] deal stop loss. For MT5 opening deal this is the SL of the order opening the 
   * position. For MT4 deals or MT5 closing deal this is the last known position SL.
   * @property {number} [takeProfit] deal take profit. For MT5 opening deal this is the TP of the order opening the 
   * position. For MT4 deals or MT5 closing deal this is the last known position TP.
   */ /**
   * Returns history deals with a specific ticket number
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} ticket ticket number (deal id for MT5 or order id for MT4)
   * @returns {Promise<MetatraderDeals>} promise resolving with request results containing deals found
   */ async getDealsByTicket(accountId, ticket) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getDealsByTicket",
            ticket
        });
        return {
            deals: response.deals,
            synchronizing: response.synchronizing
        };
    }
    /**
   * Returns history deals for a specific position id
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} positionId position id
   * @returns {Promise<MetatraderDeals>} promise resolving with request results containing deals found
   */ async getDealsByPosition(accountId, positionId) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getDealsByPosition",
            positionId
        });
        return {
            deals: response.deals,
            synchronizing: response.synchronizing
        };
    }
    /**
   * Returns history deals with for a specific time range
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {Date} startTime start of time range, inclusive
   * @param {Date} endTime end of time range, exclusive
   * @param {Number} offset pagination offset, default is 0
   * @param {Number} limit pagination limit, default is 1000
   * @returns {Promise<MetatraderDeals>} promise resolving with request results containing deals found
   */ async getDealsByTimeRange(accountId, startTime, endTime, offset = 0, limit = 1000) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getDealsByTimeRange",
            startTime,
            endTime,
            offset,
            limit
        });
        return {
            deals: response.deals,
            synchronizing: response.synchronizing
        };
    }
    /**
   * Clears the order and transaction history of a specified application and removes the application
   * @param {String} accountId id of the MetaTrader account to remove history and application for
   * @return {Promise} promise resolving when the history is cleared
   */ removeApplication(accountId) {
        return this.rpcRequest(accountId, {
            type: "removeApplication"
        });
    }
    /**
   * MetaTrader trade response
   * @typedef {Object} MetatraderTradeResponse
   * @property {Number} numericCode numeric response code, see
   * https://www.mql5.com/en/docs/constants/errorswarnings/enum_trade_return_codes and
   * https://book.mql4.com/appendix/errors. Response codes which indicate success are 0, 10008-10010, 10025. The rest
   * codes are errors
   * @property {String} stringCode string response code, see
   * https://www.mql5.com/en/docs/constants/errorswarnings/enum_trade_return_codes and
   * https://book.mql4.com/appendix/errors. Response codes which indicate success are ERR_NO_ERROR,
   * TRADE_RETCODE_PLACED, TRADE_RETCODE_DONE, TRADE_RETCODE_DONE_PARTIAL, TRADE_RETCODE_NO_CHANGES. The rest codes are
   * errors.
   * @property {String} message human-readable response message
   * @property {String} orderId order id which was created/modified during the trade
   * @property {String} positionId position id which was modified during the trade
   */ /**
   * Execute a trade on a connected MetaTrader account
   * @param {String} accountId id of the MetaTrader account to execute trade for
   * @param {MetatraderTrade} trade trade to execute (see docs for possible trade types)
   * @param {String} [application] application to use
   * @param {String} [reliability] account reliability
   * @returns {Promise<MetatraderTradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */ // eslint-disable-next-line complexity
    async trade(accountId, trade, application, reliability) {
        let response;
        if (application === "RPC") {
            response = await this.rpcRequest(accountId, {
                type: "trade",
                trade,
                application
            });
        } else {
            response = await this.rpcRequestAllInstances(accountId, {
                type: "trade",
                trade,
                application: application || this._application,
                requestId: _randomstring.default.generate(32)
            }, reliability);
        }
        response.response = response.response || {};
        response.response.stringCode = response.response.stringCode || response.response.description;
        response.response.numericCode = response.response.numericCode !== undefined ? response.response.numericCode : response.response.error;
        if ([
            "ERR_NO_ERROR",
            "TRADE_RETCODE_PLACED",
            "TRADE_RETCODE_DONE",
            "TRADE_RETCODE_DONE_PARTIAL",
            "TRADE_RETCODE_NO_CHANGES"
        ].includes(response.response.stringCode || response.response.description)) {
            return response.response;
        } else {
            throw new _tradeError.default(response.response.message, response.response.numericCode, response.response.stringCode);
        }
    }
    /**
   * Creates a task that ensures the account gets subscribed to the server
   * @param {String} accountId account id to subscribe
   * @param {Number} instanceNumber instance index number
   */ ensureSubscribe(accountId, instanceNumber) {
        this._subscriptionManager.scheduleSubscribe(accountId, instanceNumber);
    }
    /**
   * Subscribes to the Metatrader terminal events
   * @param {String} accountId id of the MetaTrader account to subscribe to
   * @param {Number} instanceNumber instance index number
   * @returns {Promise} promise which resolves when subscription started
   */ subscribe(accountId, instanceNumber) {
        return this._subscriptionManager.subscribe(accountId, instanceNumber);
    }
    /**
   * Requests the terminal to start synchronization process
   * @param {String} accountId id of the MetaTrader account to synchronize
   * @param {Number} instanceIndex instance index
   * @param {String} host name of host to synchronize with
   * @param {String} synchronizationId synchronization request id
   * @param {Date} startingHistoryOrderTime from what date to start synchronizing history orders from. If not specified,
   * the entire order history will be downloaded.
   * @param {Date} startingDealTime from what date to start deal synchronization from. If not specified, then all
   * history deals will be downloaded.
   * @param {Function} getHashes function to get terminal state hashes
   * @returns {Promise} promise which resolves when synchronization started
   */ async synchronize(accountId, instanceIndex, host, synchronizationId, startingHistoryOrderTime, startingDealTime, hashes) {
        if (this._getSocketInstanceByAccount(accountId, instanceIndex) === undefined) {
            this._logger.debug(`${accountId}:${instanceIndex}: creating socket instance on synchronize`);
            await this._createSocketInstanceByAccount(accountId, instanceIndex);
        }
        const syncThrottler = this._getSocketInstanceByAccount(accountId, instanceIndex).synchronizationThrottler;
        this._synchronizationHashes[synchronizationId] = hashes;
        this._synchronizationHashes[synchronizationId].lastUpdated = Date.now();
        return syncThrottler.scheduleSynchronize(accountId, {
            requestId: synchronizationId,
            version: 2,
            type: "synchronize",
            startingHistoryOrderTime,
            startingDealTime,
            instanceIndex,
            host
        }, hashes);
    }
    /**
   * Waits for server-side terminal state synchronization to complete
   * @param {String} accountId id of the MetaTrader account to synchronize
   * @param {Number} [instanceNumber] instance index number
   * @param {String} applicationPattern MetaApi application regular expression pattern, default is .*
   * @param {Number} timeoutInSeconds timeout in seconds, default is 300 seconds
   * @param {String} [application] application to synchronize with
   * @returns {Promise} promise which resolves when synchronization started
   */ waitSynchronized(accountId, instanceNumber, applicationPattern, timeoutInSeconds, application) {
        return this.rpcRequest(accountId, {
            type: "waitSynchronized",
            applicationPattern,
            timeoutInSeconds,
            instanceIndex: instanceNumber,
            application: application || this._application
        }, timeoutInSeconds + 1);
    }
    /**
   * Market data subscription
   * @typedef {Object} MarketDataSubscription
   * @property {string} type subscription type, one of quotes, candles, ticks, or marketDepth
   * @property {string} [timeframe] when subscription type is candles, defines the timeframe according to which the
   * candles must be generated. Allowed values for MT5 are 1m, 2m, 3m, 4m, 5m, 6m, 10m, 12m, 15m, 20m, 30m, 1h, 2h, 3h,
   * 4h, 6h, 8h, 12h, 1d, 1w, 1mn. Allowed values for MT4 are 1m, 5m, 15m 30m, 1h, 4h, 1d, 1w, 1mn
   * @property {number} [intervalInMilliseconds] defines how frequently the terminal will stream data to client. If not
   * set, then the value configured in account will be used
   */ /**
   * Subscribes on market data of specified symbol
   * @param {String} accountId id of the MetaTrader account
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @param {Array<MarketDataSubscription>} subscriptions array of market data subscription to create or update
   * @param {String} [reliability] account reliability
   * @returns {Promise} promise which resolves when subscription request was processed
   */ subscribeToMarketData(accountId, symbol, subscriptions, reliability) {
        return this.rpcRequestAllInstances(accountId, {
            type: "subscribeToMarketData",
            symbol,
            subscriptions
        }, reliability);
    }
    /**
   * Refreshes market data subscriptions on the server to prevent them from expiring
   * @param {String} accountId id of the MetaTrader account
   * @param {Number} instanceNumber instance index number
   * @param {Array} subscriptions array of subscriptions to refresh
   */ refreshMarketDataSubscriptions(accountId, instanceNumber, subscriptions) {
        return this.rpcRequest(accountId, {
            type: "refreshMarketDataSubscriptions",
            subscriptions,
            instanceIndex: instanceNumber
        });
    }
    /**
   * Market data unsubscription
   * @typedef {Object} MarketDataUnsubscription
   * @property {string} type subscription type, one of quotes, candles, ticks, or marketDepth
   */ /**
   * Unsubscribes from market data of specified symbol
   * @param {String} accountId id of the MetaTrader account
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @param {Array<MarketDataUnsubscription>} subscriptions array of subscriptions to cancel
   * @param {String} [reliability] account reliability
   * @returns {Promise} promise which resolves when unsubscription request was processed
   */ unsubscribeFromMarketData(accountId, symbol, subscriptions, reliability) {
        return this.rpcRequestAllInstances(accountId, {
            type: "unsubscribeFromMarketData",
            symbol,
            subscriptions
        }, reliability);
    }
    /**
   * Retrieves symbols available on an account
   * @param {String} accountId id of the MetaTrader account to retrieve symbols for
   * @returns {Promise<Array<string>>} promise which resolves when symbols are retrieved
   */ async getSymbols(accountId) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getSymbols"
        });
        return response.symbols;
    }
    /**
   * Retrieves specification for a symbol
   * @param {String} accountId id of the MetaTrader account to retrieve symbol specification for
   * @param {String} symbol symbol to retrieve specification for
   * @returns {Promise<MetatraderSymbolSpecification>} promise which resolves when specification is retrieved
   */ async getSymbolSpecification(accountId, symbol) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getSymbolSpecification",
            symbol
        });
        return response.specification;
    }
    /**
   * Retrieves price for a symbol
   * @param {String} accountId id of the MetaTrader account to retrieve symbol price for
   * @param {String} symbol symbol to retrieve price for
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderSymbolPrice>} promise which resolves when price is retrieved
   */ async getSymbolPrice(accountId, symbol, keepSubscription = false) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getSymbolPrice",
            symbol,
            keepSubscription
        });
        return response.price;
    }
    /**
   * Retrieves price for a symbol
   * @param {string} accountId id of the MetaTrader account to retrieve candle for
   * @param {string} symbol symbol to retrieve candle for
   * @param {string} timeframe defines the timeframe according to which the candle must be generated. Allowed values for
   * MT5 are 1m, 2m, 3m, 4m, 5m, 6m, 10m, 12m, 15m, 20m, 30m, 1h, 2h, 3h, 4h, 6h, 8h, 12h, 1d, 1w, 1mn. Allowed values
   * for MT4 are 1m, 5m, 15m 30m, 1h, 4h, 1d, 1w, 1mn
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderCandle>} promise which resolves when candle is retrieved
   */ async getCandle(accountId, symbol, timeframe, keepSubscription = false) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getCandle",
            symbol,
            timeframe,
            keepSubscription
        });
        return response.candle;
    }
    /**
   * Retrieves latest tick for a symbol. MT4 G1 accounts do not support this API
   * @param {string} accountId id of the MetaTrader account to retrieve symbol tick for
   * @param {string} symbol symbol to retrieve tick for
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderTick>} promise which resolves when tick is retrieved
   */ async getTick(accountId, symbol, keepSubscription = false) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getTick",
            symbol,
            keepSubscription
        });
        return response.tick;
    }
    /**
   * Retrieves latest order book for a symbol. MT4 accounts do not support this API
   * @param {string} accountId id of the MetaTrader account to retrieve symbol order book for
   * @param {string} symbol symbol to retrieve order book for
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderBook>} promise which resolves when order book is retrieved
   */ async getBook(accountId, symbol, keepSubscription = false) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getBook",
            symbol,
            keepSubscription
        });
        return response.book;
    }
    /**
   * Forces refresh of most recent quote updates for symbols subscribed to by the terminal
   * @param {string} accountId id of the MetaTrader account
   * @returns {Promise<string[]>} promise which resolves with recent quote symbols that was initiated to process
   */ async refreshTerminalState(accountId) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "refreshTerminalState"
        });
        return response.symbols;
    }
    /**
   * Forces refresh and retrieves latest quotes for a subset of symbols the terminal is subscribed to
   * @param {string} accountId id of the MetaTrader account
   * @param {string[]} symbols quote symbols to refresh
   * @returns {Promise<RefreshedQuotes>} refreshed quotes and basic account information info
   */ async refreshSymbolQuotes(accountId, symbols) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "refreshSymbolQuotes",
            symbols
        });
        return response.refreshedQuotes;
    }
    /**
   * Sends client uptime stats to the server.
   * @param {String} accountId id of the MetaTrader account to save uptime
   * @param {Object} uptime uptime statistics to send to the server
   * @returns {Promise} promise which resolves when uptime statistics is submitted
   */ saveUptime(accountId, uptime) {
        return this.rpcRequest(accountId, {
            type: "saveUptime",
            uptime
        });
    }
    /**
   * Unsubscribe from account
   * @param {String} accountId id of the MetaTrader account to unsubscribe
   * @returns {Promise} promise which resolves when socket unsubscribed
   */ async unsubscribe(accountId) {
        const region = this.getAccountRegion(accountId);
        this._latencyService.onUnsubscribe(accountId);
        const updateEventsToRemove = Object.keys(this._updateEvents).filter((key)=>key.startsWith(accountId));
        updateEventsToRemove.forEach((key)=>delete this._updateEvents[key]);
        if (this._socketInstances[region]) {
            await Promise.all(Object.keys(this._socketInstances[region]).map(async (instanceNumber)=>{
                try {
                    await this._subscriptionManager.unsubscribe(accountId, Number(instanceNumber));
                    delete this._socketInstancesByAccounts[instanceNumber][accountId];
                } catch (err) {
                    if (![
                        "TimeoutError",
                        "NotFoundError"
                    ].includes(err.name)) {
                        this._logger.warn(`${accountId}:${instanceNumber}: failed to unsubscribe`, err);
                    }
                }
            }));
        }
    }
    /**
   * Current server time (see https://metaapi.cloud/docs/client/models/serverTime/)
   * @typedef {Object} ServerTime
   * @property {Date} time current server time
   * @property {String} brokerTime current broker time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {Date} [lastQuoteTime] last quote time
   * @property {String} [lastQuoteBrokerTime] last quote time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   */ /**
   * Returns server time for a specified MetaTrader account
   * @param {string} accountId id of the MetaTrader account to return server time for
   * @returns {Promise<ServerTime>} promise resolving with server time
   */ async getServerTime(accountId) {
        let response = await this.rpcRequest(accountId, {
            application: "RPC",
            type: "getServerTime"
        });
        return response.serverTime;
    }
    /**
   * Margin required to open a trade (see https://metaapi.cloud/docs/client/models/margin/)
   * @typedef {Object} Margin
   * @property {number} [margin] margin required to open a trade. If margin can not be calculated, then this field is
   * not defined
   */ /**
   * Contains order to calculate margin for (see https://metaapi.cloud/docs/client/models/marginOrder/)
   * @typedef {Object} MarginOrder
   * @property {string} symbol order symbol
   * @property {string} type order type, one of ORDER_TYPE_BUY or ORDER_TYPE_SELL
   * @property {number} volume order volume, must be greater than 0
   * @property {number} openPrice order open price, must be greater than 0
   */ /**
   * Calculates margin required to open a trade on the specified trading account
   * @param {string} accountId id of the trading account to calculate margin for
   * @param {string} application application to send the request to
   * @param {string} reliability account reliability
   * @param {MarginOrder} order order to calculate margin for
   * @returns {Promise<Margin>} promise resolving with margin calculation result
   */ async calculateMargin(accountId, application, reliability, order) {
        let response;
        if (application === "RPC") {
            response = await this.rpcRequest(accountId, {
                application,
                type: "calculateMargin",
                order
            });
        } else {
            response = await this.rpcRequestAllInstances(accountId, {
                application,
                type: "calculateMargin",
                order
            }, reliability);
        }
        return response.margin;
    }
    /**
   * Calls onUnsubscribeRegion listener event 
   * @param {String} accountId account id
   * @param {String} region account region to unsubscribe
   */ async unsubscribeAccountRegion(accountId, region) {
        const unsubscribePromises = [];
        for (let listener of this._synchronizationListeners[accountId] || []){
            unsubscribePromises.push(Promise.resolve((async ()=>{
                await this._processEvent(()=>listener.onUnsubscribeRegion(region), `${accountId}:${region}:onUnsubscribeRegion`, true);
            })()).catch((err)=>this._logger.error(`${accountId}:${region}: Failed to notify listener ` + "about onUnsubscribeRegion event", err)));
        }
        await Promise.all(unsubscribePromises);
    }
    /**
   * Adds synchronization listener for specific account
   * @param {String} accountId account id
   * @param {SynchronizationListener} listener synchronization listener to add
   */ addSynchronizationListener(accountId, listener) {
        this._logger.trace(`${accountId}: Added synchronization listener`);
        let listeners = this._synchronizationListeners[accountId];
        if (!listeners) {
            listeners = [];
            this._synchronizationListeners[accountId] = listeners;
        }
        listeners.push(listener);
    }
    /**
   * Removes synchronization listener for specific account
   * @param {String} accountId account id
   * @param {SynchronizationListener} listener synchronization listener to remove
   */ removeSynchronizationListener(accountId, listener) {
        this._logger.trace(`${accountId}: Removed synchronization listener`);
        let listeners = this._synchronizationListeners[accountId];
        if (!listeners) {
            listeners = [];
        }
        listeners = listeners.filter((l)=>l !== listener);
        this._synchronizationListeners[accountId] = listeners;
    }
    /**
   * Adds latency listener
   * @param {LatencyListener} listener latency listener to add
   */ addLatencyListener(listener) {
        this._latencyListeners.push(listener);
    }
    /**
   * Removes latency listener
   * @param {LatencyListener} listener latency listener to remove
   */ removeLatencyListener(listener) {
        this._latencyListeners = this._latencyListeners.filter((l)=>l !== listener);
    }
    /**
   * Adds reconnect listener
   * @param {ReconnectListener} listener reconnect listener to add
   * @param {String} accountId account id of listener
   */ addReconnectListener(listener, accountId) {
        this._reconnectListeners.push({
            accountId,
            listener
        });
    }
    /**
   * Removes reconnect listener
   * @param {ReconnectListener} listener listener to remove
   */ removeReconnectListener(listener) {
        this._reconnectListeners = this._reconnectListeners.filter((l)=>l.listener !== listener);
    }
    /**
   * Removes all listeners. Intended for use in unit tests.
   */ removeAllListeners() {
        this._synchronizationListeners = {};
        this._reconnectListeners = [];
    }
    /**
   * Clears account or replica data from client records and unsubscribes
   * @param {string} accountId account id to process the removal of
   */ onAccountDeleted(accountId) {
        this._subscriptionManager.cancelAccount(accountId);
        this._latencyService.onUnsubscribe(accountId);
        const masterAccountId = this._accountsByReplicaId[accountId];
        if (masterAccountId) {
            if (masterAccountId === accountId) {
                const regionData = this._accountReplicas[masterAccountId];
                const replicas = Object.values(regionData);
                Object.keys(this._synchronizationIdByInstance).filter((instance)=>instance.startsWith(`${masterAccountId}`)).forEach((instance)=>delete this._synchronizationIdByInstance[instance]);
                replicas.forEach((replica)=>{
                    Object.values(this._socketInstancesByAccounts).forEach((instance)=>{
                        delete instance[replica];
                    });
                    delete this._accountsByReplicaId[replica];
                    delete this._regionsByAccounts[replica];
                });
                delete this._accountReplicas[masterAccountId];
                this._logger.debug(`${masterAccountId}: processed primary account removal`);
            } else {
                Object.values(this._socketInstancesByAccounts).forEach((instance)=>{
                    delete instance[accountId];
                });
                const regionData = this._regionsByAccounts[accountId];
                if (regionData) {
                    const region = regionData.region;
                    Object.keys(this._synchronizationIdByInstance).filter((instance)=>instance.startsWith(`${masterAccountId}:${region}`)).forEach((instance)=>delete this._synchronizationIdByInstance[instance]);
                    delete this._accountReplicas[masterAccountId][region];
                    this._logger.debug(`${masterAccountId}: processed removal of replica ${accountId}`);
                }
                delete this._accountsByReplicaId[accountId];
                delete this._regionsByAccounts[accountId];
            }
        }
    }
    /**
   * Queues an account packet for processing
   * @param {Object} packet packet to process
   */ queuePacket(packet) {
        const accountId = packet.accountId;
        const packets = this._packetOrderer.restoreOrder(packet).filter((p)=>p.type !== "noop");
        if (this._sequentialEventProcessing && packet.sequenceNumber !== undefined) {
            const events = packets.map((packetItem)=>()=>Promise.resolve(this._processSynchronizationPacket(packetItem)));
            if (!this._eventQueues[accountId]) {
                this._eventQueues[accountId] = events;
                this._callAccountEvents(accountId);
            } else {
                this._eventQueues[accountId] = this._eventQueues[accountId].concat(events);
            }
        } else {
            packets.forEach((packetItem)=>this._processSynchronizationPacket(packetItem));
        }
    }
    /**
   * Queues account event for processing
   * @param {String} accountId account id
   * @param {String} name event label name
   * @param {Function} callable async or regular function to execute
   */ queueEvent(accountId, name, callable) {
        let event = ()=>this._processEvent(callable, `${accountId}:${name}`);
        if (this._sequentialEventProcessing) {
            if (!this._eventQueues[accountId]) {
                this._eventQueues[accountId] = [
                    event
                ];
                this._callAccountEvents(accountId);
            } else {
                this._eventQueues[accountId].push(event);
            }
        } else {
            event();
        }
    }
    async _callAccountEvents(accountId) {
        if (this._eventQueues[accountId]) {
            while(this._eventQueues[accountId].length){
                await this._eventQueues[accountId][0]();
                this._eventQueues[accountId].shift();
            }
            delete this._eventQueues[accountId];
        }
    }
    async _reconnect(instanceNumber, socketInstanceIndex, region) {
        const instance = this.socketInstances[region][instanceNumber][socketInstanceIndex];
        if (instance) {
            while(!instance.socket.connected && !instance.isReconnecting && instance.connected){
                await this._tryReconnect(instanceNumber, socketInstanceIndex, region);
            }
        }
    }
    _tryReconnect(instanceNumber, socketInstanceIndex, region) {
        const instance = this.socketInstances[region][instanceNumber][socketInstanceIndex];
        instance.reconnectWaitTime = Math.min(instance.reconnectWaitTime * 2, 30000);
        return new Promise((resolve)=>setTimeout(async ()=>{
                if (!instance.socket.connected && !instance.isReconnecting && instance.connected) {
                    try {
                        instance.sessionId = _randomstring.default.generate(32);
                        const clientId = Math.random();
                        instance.socket.close();
                        instance.socket.io.opts.extraHeaders["Client-Id"] = clientId;
                        instance.socket.io.opts.query.clientId = clientId;
                        instance.isReconnecting = true;
                        instance.socket.io.uri = await this._getServerUrl(instanceNumber, socketInstanceIndex, region);
                        instance.socket.connect();
                    } catch (error) {
                        instance.isReconnecting = false;
                    }
                }
                resolve();
            }, instance.reconnectWaitTime));
    }
    /**
   * Simulataneously sends RPC requests to all synchronized instances
   * @param {String} accountId metatrader account id
   * @param {Object} request base request data
   * @param {String} [reliability] account reliability
   * @param {Number} [timeoutInSeconds] request timeout in seconds
   */ async rpcRequestAllInstances(accountId, request, reliability, timeoutInSeconds) {
        if (reliability === "high") {
            try {
                return await (0, _promiseany.default)([
                    0,
                    1
                ].map((instanceNumber)=>{
                    return this.rpcRequest(accountId, Object.assign({}, request, {
                        instanceIndex: instanceNumber
                    }), timeoutInSeconds);
                }));
            } catch (error) {
                throw error.errors[0];
            }
        } else {
            return await this.rpcRequest(accountId, request, timeoutInSeconds);
        }
    }
    /**
   * Makes a RPC request
   * @param {String} accountId metatrader account id
   * @param {Object} request base request data
   * @param {Number} [timeoutInSeconds] request timeout in seconds
   */ //eslint-disable-next-line complexity, max-statements
    async rpcRequest(accountId, request, timeoutInSeconds) {
        const ignoredRequestTypes = [
            "subscribe",
            "synchronize",
            "refreshMarketDataSubscriptions",
            "unsubscribe"
        ];
        const primaryAccountId = this._accountsByReplicaId[accountId];
        let connectedInstance = this._latencyService.getActiveAccountInstances(primaryAccountId)[0];
        if (!ignoredRequestTypes.includes(request.type)) {
            if (!connectedInstance) {
                connectedInstance = await this._latencyService.waitConnectedInstance(accountId);
            }
            const activeRegion = connectedInstance.split(":")[1];
            accountId = this._accountReplicas[primaryAccountId][activeRegion];
        }
        let socketInstanceIndex = null;
        let instanceNumber = 0;
        const region = this.getAccountRegion(accountId);
        this._refreshAccountRegion(accountId);
        if (request.instanceIndex !== undefined) {
            instanceNumber = request.instanceIndex;
        } else {
            if (connectedInstance) {
                instanceNumber = Number(connectedInstance.split(":")[2]);
            }
            if (request.application !== "RPC") {
                request = Object.assign({}, request, {
                    instanceIndex: instanceNumber
                });
            }
        }
        if (!this._socketInstancesByAccounts[instanceNumber]) {
            this._socketInstancesByAccounts[instanceNumber] = {};
        }
        if (!this._socketInstances[region]) {
            this._socketInstances[region] = {};
        }
        if (!this._socketInstances[region][instanceNumber]) {
            this._socketInstances[region][instanceNumber] = [];
        }
        if (this._socketInstancesByAccounts[instanceNumber][accountId] !== undefined) {
            socketInstanceIndex = this._socketInstancesByAccounts[instanceNumber][accountId];
        } else {
            this._logger.debug(`${accountId}:${instanceNumber}: creating socket instance on RPC request`);
            await this._createSocketInstanceByAccount(accountId, instanceNumber);
            socketInstanceIndex = this._socketInstancesByAccounts[instanceNumber][accountId];
        }
        const instance = this._socketInstances[region][instanceNumber][socketInstanceIndex];
        if (!instance.connected) {
            this._logger.debug(`${accountId}:${instanceNumber}: connecting socket instance on RPC request`);
            await this.connect(instanceNumber, region);
        } else if (!this.connected(instanceNumber, socketInstanceIndex, region)) {
            await instance.connectResult;
        }
        if (request.type === "subscribe") {
            request.sessionId = instance.sessionId;
        }
        if ([
            "trade",
            "subscribe"
        ].includes(request.type)) {
            return this._makeRequest(accountId, instanceNumber, request, timeoutInSeconds);
        }
        let retryCounter = 0;
        while(true){
            try {
                return await this._makeRequest(accountId, instanceNumber, request, timeoutInSeconds);
            } catch (err) {
                if (err.name === "TooManyRequestsError") {
                    let calcRetryCounter = retryCounter;
                    let calcRequestTime = 0;
                    while(calcRetryCounter < this._retries){
                        calcRetryCounter++;
                        calcRequestTime += Math.min(Math.pow(2, calcRetryCounter) * this._minRetryDelayInSeconds, this._maxRetryDelayInSeconds) * 1000;
                    }
                    const retryTime = new Date(err.metadata.recommendedRetryTime).getTime();
                    if (Date.now() + calcRequestTime > retryTime && retryCounter < this._retries) {
                        if (Date.now() < retryTime) {
                            await new Promise((res)=>setTimeout(res, retryTime - Date.now()));
                        }
                        retryCounter++;
                    } else {
                        throw err;
                    }
                } else if ([
                    "NotSynchronizedError",
                    "TimeoutError",
                    "NotAuthenticatedError",
                    "InternalError"
                ].includes(err.name) && retryCounter < this._retries) {
                    await new Promise((res)=>setTimeout(res, Math.min(Math.pow(2, retryCounter) * this._minRetryDelayInSeconds, this._maxRetryDelayInSeconds) * 1000));
                    retryCounter++;
                } else {
                    throw err;
                }
                if (this._socketInstancesByAccounts[instanceNumber][accountId] === undefined) {
                    throw err;
                }
            }
        }
    }
    _makeRequest(accountId, instanceNumber, request, timeoutInSeconds) {
        const socketInstance = this._getSocketInstanceByAccount(accountId, instanceNumber);
        let requestId = request.requestId || _randomstring.default.generate(32);
        request.timestamps = {
            clientProcessingStarted: new Date()
        };
        let result = Promise.race([
            new Promise((resolve, reject)=>socketInstance.requestResolves[requestId] = {
                    resolve,
                    reject,
                    type: request.type
                }),
            new Promise((resolve, reject)=>setTimeout(()=>{
                    reject(new _timeoutError.default(`MetaApi websocket client request ${request.requestId} of type ${request.type} ` + "timed out. Please make sure your account is connected to broker before retrying your request."));
                    delete socketInstance.requestResolves[requestId];
                }, timeoutInSeconds * 1000 || this._requestTimeout))
        ]);
        request.accountId = accountId;
        request.application = request.application || this._application;
        if (!request.requestId) {
            request.requestId = requestId;
        }
        if (request.type === "unsubscribe" || request.application === "RPC" || request.instanceIndex === socketInstance.instanceNumber) {
            this._logger.debug(()=>`${accountId}: Sending request: ${JSON.stringify(request)}`);
            socketInstance.socket.emit("request", request);
            return result;
        } else {
            this._logger.trace(()=>`${accountId}:${request.instanceIndex}: skipping request because it is being sent to ` + `the socket of the wrong instance index, request=${JSON.stringify(request)}`);
            return result;
        }
    }
    // eslint-disable-next-line complexity
    _convertError(data) {
        if (data.error === "ValidationError") {
            return new _errorHandler.ValidationError(data.message, data.details);
        } else if (data.error === "NotFoundError") {
            return new _errorHandler.NotFoundError(data.message);
        } else if (data.error === "NotSynchronizedError") {
            return new _notSynchronizedError.default(data.message);
        } else if (data.error === "TimeoutError") {
            return new _timeoutError.default(data.message);
        } else if (data.error === "NotAuthenticatedError") {
            return new _notConnectedError.default(data.message);
        } else if (data.error === "ForbiddenError") {
            return new _errorHandler.ForbiddenError(data.message);
        } else if (data.error === "TradeError") {
            return new _tradeError.default(data.message, data.numericCode, data.stringCode);
        } else if (data.error === "UnauthorizedError") {
            this.close();
            return new _errorHandler.UnauthorizedError(data.message);
        } else if (data.error === "TooManyRequestsError") {
            return new _errorHandler.TooManyRequestsError(data.message, data.metadata);
        } else {
            return new _errorHandler.InternalError(data.message);
        }
    }
    // eslint-disable-next-line complexity
    _convertIsoTimeToDate(packet) {
        // eslint-disable-next-line guard-for-in
        for(let field in packet){
            let value = packet[field];
            if (typeof value === "string" && field.match(/time$|Time$/) && !field.match(/brokerTime$|BrokerTime$|timeframe$/)) {
                packet[field] = new Date(value);
            }
            if (Array.isArray(value)) {
                for (let item of value){
                    this._convertIsoTimeToDate(item);
                }
            }
            if (typeof value === "object") {
                this._convertIsoTimeToDate(value);
            }
        }
        if (packet && packet.timestamps) {
            // eslint-disable-next-line guard-for-in
            for(let field in packet.timestamps){
                packet.timestamps[field] = new Date(packet.timestamps[field]);
            }
        }
        if (packet && packet.type === "prices") {
            for (let price of packet.prices || []){
                if (price.timestamps) {
                    // eslint-disable-next-line guard-for-in
                    for(let field in price.timestamps){
                        price.timestamps[field] = new Date(price.timestamps[field]);
                    }
                }
            }
        }
    }
    /**
   * MetaTrader symbol price. Contains current price for a symbol (see
   * https://metaapi.cloud/docs/client/models/metatraderSymbolPrice/)
   * @typedef {Object} MetatraderSymbolPrice
   * @property {String} symbol symbol (e.g. a currency pair or an index)
   * @property {Number} bid bid price
   * @property {Number} ask ask price
   * @property {Number} profitTickValue tick value for a profitable position
   * @property {Number} lossTickValue tick value for a losing position
   * @property {Number} [accountCurrencyExchangeRate] current exchange rate of account currency into account base
   * currency (USD if you did not override it)
   * @property {Date} time quote time, in ISO format
   * @property {String} brokerTime time quote time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   */ /**
   * MetaTrader candle
   * @typedef {Object} MetatraderCandle
   * @property {string} symbol symbol (e.g. currency pair or an index)
   * @property {string} timeframe timeframe candle was generated for, e.g. 1h. One of 1m, 2m, 3m, 4m, 5m, 6m, 10m, 12m,
   * 15m, 20m, 30m, 1h, 2h, 3h, 4h, 6h, 8h, 12h, 1d, 1w, 1mn
   * @property {Date} time candle opening time
   * @property {string} brokerTime candle opening time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {number} open open price
   * @property {number} high high price
   * @property {number} low low price
   * @property {number} close close price
   * @property {number} tickVolume tick volume, i.e. number of ticks inside the candle
   * @property {number} spread spread in points
   * @property {number} volume trade volume
   */ /**
   * MetaTrader tick data
   * @typedef {Object} MetatraderTick
   * @property {string} symbol symbol (e.g. a currency pair or an index)
   * @property {Date} time time
   * @property {string} brokerTime time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {number} [bid] bid price
   * @property {number} [ask] ask price
   * @property {number} [last] last deal price
   * @property {number} [volume] volume for the current last deal price
   * @property {string} side is tick a result of buy or sell deal, one of buy or sell
   */ /**
   * MetaTrader order book
   * @typedef {Object} MetatraderBook
   * @property {string} symbol symbol (e.g. a currency pair or an index)
   * @property {Date} time time
   * @property {string} brokerTime time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {Array<MetatraderBookEntry>} book list of order book entries
   */ /**
   * MetaTrader order book entry
   * @typedef {Object} MetatraderBookEntry
   * @property {string} type entry type, one of BOOK_TYPE_SELL, BOOK_TYPE_BUY, BOOK_TYPE_SELL_MARKET,
   * BOOK_TYPE_BUY_MARKET
   * @property {number} price price
   * @property {number} volume volume
   */ // eslint-disable-next-line complexity,max-statements
    async _processSynchronizationPacket(data) {
        try {
            const instanceNumber = data.instanceIndex || 0;
            const socketInstance = this._getSocketInstanceByAccount(data.accountId, instanceNumber);
            if (data.synchronizationId && socketInstance) {
                socketInstance.synchronizationThrottler.updateSynchronizationId(data.synchronizationId);
            }
            const region = this.getAccountRegion(data.accountId);
            const primaryAccountId = this._accountsByReplicaId[data.accountId];
            let instanceId = primaryAccountId + ":" + region + ":" + instanceNumber + ":" + (data.host || 0);
            let instanceIndex = region + ":" + instanceNumber + ":" + (data.host || 0);
            const isOnlyActiveInstance = ()=>{
                const activeInstanceIds = Object.keys(this._connectedHosts).filter((instance)=>instance.startsWith(primaryAccountId + ":" + region + ":" + instanceNumber));
                return !activeInstanceIds.length || activeInstanceIds.length === 1 && activeInstanceIds[0] === instanceId;
            };
            const cancelDisconnectTimer = ()=>{
                if (this._statusTimers[instanceId]) {
                    clearTimeout(this._statusTimers[instanceId]);
                }
            };
            const resetDisconnectTimer = ()=>{
                cancelDisconnectTimer();
                this._statusTimers[instanceId] = setTimeout(()=>{
                    this._logger.warn(`${data.accountId}:${instanceIndex}: timed out waiting for connection status`);
                    if (isOnlyActiveInstance()) {
                        this._subscriptionManager.onTimeout(data.accountId, 0);
                        this._subscriptionManager.onTimeout(data.accountId, 1);
                    }
                    this.queueEvent(primaryAccountId, `${instanceIndex}:onDisconnected`, ()=>onDisconnected(true));
                    clearTimeout(this._statusTimers[instanceId]);
                }, 60000);
            };
            // eslint-disable-next-line complexity
            const onDisconnected = async (isTimeout = false)=>{
                if (this._connectedHosts[instanceId]) {
                    this._latencyService.onDisconnected(instanceId);
                    if (isOnlyActiveInstance()) {
                        for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                            await this._processEvent(()=>listener.onDisconnected(instanceIndex), `${primaryAccountId}:${instanceIndex}:onDisconnected`);
                        }
                    }
                    this._packetOrderer.onStreamClosed(instanceId);
                    if (socketInstance) {
                        socketInstance.synchronizationThrottler.removeIdByParameters(data.accountId, instanceNumber, data.host);
                    }
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onStreamClosed(instanceIndex), `${primaryAccountId}:${instanceIndex}:onStreamClosed`);
                    }
                    delete this._connectedHosts[instanceId];
                    if (isOnlyActiveInstance() && !isTimeout) {
                        await this._subscriptionManager.onDisconnected(data.accountId, 0);
                        await this._subscriptionManager.onDisconnected(data.accountId, 1);
                    }
                }
            };
            if (data.type === "authenticated") {
                resetDisconnectTimer();
                if (!data.sessionId || socketInstance && data.sessionId === socketInstance.sessionId) {
                    this._latencyService.onConnected(instanceId);
                    this._connectedHosts[instanceId] = data.host;
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onConnected(instanceIndex, data.replicas), `${primaryAccountId}:${instanceIndex}:onConnected`);
                    }
                    this._subscriptionManager.cancelSubscribe(data.accountId + ":" + instanceNumber);
                    if (data.replicas === 1) {
                        this._subscriptionManager.cancelAccount(data.accountId);
                    } else {
                        this._subscriptionManager.cancelSubscribe(data.accountId + ":" + instanceNumber);
                    }
                }
            } else if (data.type === "disconnected") {
                cancelDisconnectTimer();
                await onDisconnected();
            } else if (data.type === "synchronizationStarted") {
                this._updateEvents[instanceId] = [];
                this._synchronizationFlags[data.synchronizationId] = {
                    accountId: data.accountId,
                    instanceNumber,
                    specificationsUpdated: data.specificationsHashIndex === undefined,
                    positionsUpdated: data.positionsHashIndex === undefined,
                    ordersUpdated: data.ordersHashIndex === undefined
                };
                this._synchronizationIdByInstance[instanceId] = data.synchronizationId;
                const specificationsHash = data.specificationsHashIndex !== undefined ? this._synchronizationHashes[data.synchronizationId] && this._synchronizationHashes[data.synchronizationId].specificationsHashes[data.specificationsHashIndex] : undefined;
                const positionsHash = data.positionsHashIndex !== undefined ? this._synchronizationHashes[data.synchronizationId] && this._synchronizationHashes[data.synchronizationId].positionsHashes[data.positionsHashIndex] : undefined;
                const ordersHash = data.ordersHashIndex !== undefined ? this._synchronizationHashes[data.synchronizationId] && this._synchronizationHashes[data.synchronizationId].ordersHashes[data.ordersHashIndex] : undefined;
                delete this._synchronizationHashes[data.synchronizationId];
                for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                    await this._processEvent(()=>listener.onSynchronizationStarted(instanceIndex, specificationsHash, positionsHash, ordersHash, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onSynchronizationStarted`);
                }
            } else if (data.type === "accountInformation") {
                if (data.synchronizationId && data.synchronizationId !== this._synchronizationIdByInstance[instanceId]) {
                    return;
                }
                if (data.accountInformation) {
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        try {
                            await this._processEvent(()=>listener.onAccountInformationUpdated(instanceIndex, data.accountInformation), `${primaryAccountId}:${instanceIndex}:onAccountInformationUpdated`, true);
                            // eslint-disable-next-line max-depth
                            if (this._synchronizationFlags[data.synchronizationId] && !this._synchronizationFlags[data.synchronizationId].positionsUpdated) {
                                await this._processEvent(()=>listener.onPositionsSynchronized(instanceIndex, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onPositionsSynchronized`, true);
                                // eslint-disable-next-line max-depth
                                if (!this._synchronizationFlags[data.synchronizationId].ordersUpdated) {
                                    await this._processEvent(()=>listener.onPendingOrdersSynchronized(instanceIndex, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onPendingOrdersSynchronized`, true);
                                }
                            }
                        } catch (err) {
                            this._logger.error(`${primaryAccountId}:${instanceIndex}: Failed to notify listener ` + "about accountInformation event", err);
                        }
                    }
                    if (this._synchronizationFlags[data.synchronizationId] && !this._synchronizationFlags[data.synchronizationId].positionsUpdated && !this._synchronizationFlags[data.synchronizationId].ordersUpdated) {
                        delete this._synchronizationFlags[data.synchronizationId];
                    }
                }
            } else if (data.type === "deals") {
                if (data.synchronizationId && data.synchronizationId !== this._synchronizationIdByInstance[instanceId]) {
                    return;
                }
                for (let deal of data.deals || []){
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onDealAdded(instanceIndex, deal), `${primaryAccountId}:${instanceIndex}:onDealAdded`);
                    }
                }
            } else if (data.type === "orders") {
                if (data.synchronizationId && data.synchronizationId !== this._synchronizationIdByInstance[instanceId]) {
                    return;
                }
                for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                    try {
                        await this._processEvent(()=>listener.onPendingOrdersReplaced(instanceIndex, data.orders || []), `${primaryAccountId}:${instanceIndex}:onPendingOrdersReplaced`, true);
                        await this._processEvent(()=>listener.onPendingOrdersSynchronized(instanceIndex, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onPendingOrdersSynchronized`, true);
                    } catch (err) {
                        this._logger.error(`${primaryAccountId}:${instanceIndex}: Failed to notify listener ` + "about orders event", err);
                    }
                }
                if (this._synchronizationFlags[data.synchronizationId]) {
                    delete this._synchronizationFlags[data.synchronizationId];
                }
            } else if (data.type === "historyOrders") {
                if (data.synchronizationId && data.synchronizationId !== this._synchronizationIdByInstance[instanceId]) {
                    return;
                }
                for (let historyOrder of data.historyOrders || []){
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onHistoryOrderAdded(instanceIndex, historyOrder), `${primaryAccountId}:${instanceIndex}:onHistoryOrderAdded`);
                    }
                }
            } else if (data.type === "positions") {
                if (data.synchronizationId && data.synchronizationId !== this._synchronizationIdByInstance[instanceId]) {
                    return;
                }
                for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                    try {
                        await this._processEvent(()=>listener.onPositionsReplaced(instanceIndex, data.positions || []), `${primaryAccountId}:${instanceIndex}:onPositionsReplaced`, true);
                        await this._processEvent(()=>listener.onPositionsSynchronized(instanceIndex, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onPositionsSynchronized`, true);
                        if (this._synchronizationFlags[data.synchronizationId] && !this._synchronizationFlags[data.synchronizationId].ordersUpdated) {
                            await this._processEvent(()=>listener.onPendingOrdersSynchronized(instanceIndex, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onPendingOrdersSynchronized`, true);
                        }
                    } catch (err) {
                        this._logger.error(`${primaryAccountId}:${instanceIndex}: Failed to notify listener ` + "about positions event", err);
                    }
                }
                if (this._synchronizationFlags[data.synchronizationId] && !this._synchronizationFlags[data.synchronizationId].ordersUpdated) {
                    delete this._synchronizationFlags[data.synchronizationId];
                }
            } else if (data.type === "update") {
                if (this._updateEvents[instanceId]) {
                    this._updateEvents[instanceId].push(data);
                }
                if (data.accountInformation) {
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onAccountInformationUpdated(instanceIndex, data.accountInformation), `${primaryAccountId}:${instanceIndex}:onAccountInformationUpdated`);
                    }
                }
                const updatedPositions = data.updatedPositions || [];
                const removedPositionIds = data.removedPositionIds || [];
                if (updatedPositions.length || removedPositionIds.length) {
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onPositionsUpdated(instanceIndex, updatedPositions, removedPositionIds), `${primaryAccountId}:${instanceIndex}:onPositionsUpdated`);
                    }
                }
                for (let position of updatedPositions){
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onPositionUpdated(instanceIndex, position), `${primaryAccountId}:${instanceIndex}:onPositionUpdated`);
                    }
                }
                for (let positionId of removedPositionIds){
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onPositionRemoved(instanceIndex, positionId), `${primaryAccountId}:${instanceIndex}:onPositionRemoved`);
                    }
                }
                const updatedOrders = data.updatedOrders || [];
                const completedOrderIds = data.completedOrderIds || [];
                if (updatedOrders.length || completedOrderIds.length) {
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onPendingOrdersUpdated(instanceIndex, updatedOrders, completedOrderIds), `${primaryAccountId}:${instanceIndex}:onPendingOrdersUpdated`);
                    }
                }
                for (let order of updatedOrders){
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onPendingOrderUpdated(instanceIndex, order), `${primaryAccountId}:${instanceIndex}:onPendingOrderUpdated`);
                    }
                }
                for (let orderId of completedOrderIds){
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onPendingOrderCompleted(instanceIndex, orderId), `${primaryAccountId}:${instanceIndex}:onPendingOrderCompleted`);
                    }
                }
                for (let historyOrder of data.historyOrders || []){
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onHistoryOrderAdded(instanceIndex, historyOrder), `${primaryAccountId}:${instanceIndex}:onHistoryOrderAdded`);
                    }
                }
                for (let deal of data.deals || []){
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onDealAdded(instanceIndex, deal), `${primaryAccountId}:${instanceIndex}:onDealAdded`);
                    }
                }
                if (data.timestamps) {
                    data.timestamps.clientProcessingFinished = new Date();
                    // eslint-disable-next-line max-depth
                    for (let listener of this._latencyListeners || []){
                        await this._processEvent(()=>listener.onUpdate(data.accountId, data.timestamps), `${primaryAccountId}:${instanceIndex}:onUpdate`);
                    }
                }
            } else if (data.type === "dealSynchronizationFinished") {
                if (data.synchronizationId && data.synchronizationId !== this._synchronizationIdByInstance[instanceId]) {
                    delete this._synchronizationIdByInstance[instanceId];
                    return;
                }
                this._latencyService.onDealsSynchronized(instanceId);
                for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                    if (socketInstance) {
                        socketInstance.synchronizationThrottler.removeSynchronizationId(data.synchronizationId);
                    }
                    await this._processEvent(()=>listener.onDealsSynchronized(instanceIndex, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onDealsSynchronized`);
                }
                if (this._updateEvents[instanceId]) {
                    this._updateEvents[instanceId] = this._updateEvents[instanceId].map((packet)=>()=>Promise.resolve(this._processSynchronizationPacket(packet)));
                    if (this._eventQueues[primaryAccountId]) {
                        this._eventQueues[primaryAccountId] = this._updateEvents[instanceId].concat(this._eventQueues[primaryAccountId]);
                        delete this._updateEvents[instanceId];
                    } else {
                        this._eventQueues[primaryAccountId] = this._updateEvents[instanceId];
                        delete this._updateEvents[instanceId];
                        this._callAccountEvents(primaryAccountId);
                    }
                }
            } else if (data.type === "orderSynchronizationFinished") {
                if (data.synchronizationId && data.synchronizationId !== this._synchronizationIdByInstance[instanceId]) {
                    return;
                }
                for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                    await this._processEvent(()=>listener.onHistoryOrdersSynchronized(instanceIndex, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onHistoryOrdersSynchronized`);
                }
            } else if (data.type === "status") {
                if (!this._connectedHosts[instanceId]) {
                    if (this._statusTimers[instanceId] && data.authenticated && (this._subscriptionManager.isDisconnectedRetryMode(data.accountId, instanceNumber) || !this._subscriptionManager.isAccountSubscribing(data.accountId, instanceNumber))) {
                        this._subscriptionManager.cancelSubscribe(data.accountId + ":" + instanceNumber);
                        await new Promise((res)=>setTimeout(res, 10));
                        // eslint-disable-next-line no-console
                        this._logger.info("it seems like we are not connected to a running API " + "server yet, retrying subscription for account " + instanceId);
                        this.ensureSubscribe(data.accountId, instanceNumber);
                    }
                } else {
                    resetDisconnectTimer();
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onBrokerConnectionStatusChanged(instanceIndex, !!data.connected), `${primaryAccountId}:${instanceIndex}:onBrokerConnectionStatusChanged`);
                    }
                    if (data.healthStatus) {
                        // eslint-disable-next-line max-depth
                        for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                            await this._processEvent(()=>listener.onHealthStatus(instanceIndex, data.healthStatus), `${primaryAccountId}:${instanceIndex}:onHealthStatus`);
                        }
                    }
                }
            } else if (data.type === "downgradeSubscription") {
                this._logger.info(`${primaryAccountId}:${instanceIndex}: Market data subscriptions for symbol ` + `${data.symbol} were downgraded by the server due to rate limits. Updated subscriptions: ` + `${JSON.stringify(data.updates)}, removed subscriptions: ${JSON.stringify(data.unsubscriptions)}. ` + "Please read https://metaapi.cloud/docs/client/rateLimiting/ for more details.");
                for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                    await this._processEvent(()=>listener.onSubscriptionDowngraded(instanceIndex, data.symbol, data.updates, data.unsubscriptions), `${primaryAccountId}:${instanceIndex}:onSubscriptionDowngraded`);
                }
            } else if (data.type === "specifications") {
                if (data.synchronizationId && data.synchronizationId !== this._synchronizationIdByInstance[instanceId]) {
                    return;
                }
                for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                    await this._processEvent(()=>listener.onSymbolSpecificationsUpdated(instanceIndex, data.specifications || [], data.removedSymbols || []), `${primaryAccountId}:${instanceIndex}:onSymbolSpecificationsUpdated`);
                }
                for (let specification of data.specifications || []){
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onSymbolSpecificationUpdated(instanceIndex, specification), `${primaryAccountId}:${instanceIndex}:onSymbolSpecificationUpdated`);
                    }
                }
                for (let removedSymbol of data.removedSymbols || []){
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onSymbolSpecificationRemoved(instanceIndex, removedSymbol), `${primaryAccountId}:${instanceIndex}:onSymbolSpecificationRemoved`);
                    }
                }
            } else if (data.type === "prices") {
                if (data.synchronizationId && data.synchronizationId !== this._synchronizationIdByInstance[instanceId]) {
                    return;
                }
                let prices = data.prices || [];
                let candles = data.candles || [];
                let ticks = data.ticks || [];
                let books = data.books || [];
                for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                    if (prices.length) {
                        await this._processEvent(()=>listener.onSymbolPricesUpdated(instanceIndex, prices, data.equity, data.margin, data.freeMargin, data.marginLevel, data.accountCurrencyExchangeRate), `${primaryAccountId}:${instanceIndex}:onSymbolPricesUpdated`);
                    }
                    if (candles.length) {
                        await this._processEvent(()=>listener.onCandlesUpdated(instanceIndex, candles, data.equity, data.margin, data.freeMargin, data.marginLevel, data.accountCurrencyExchangeRate), `${primaryAccountId}:${instanceIndex}:onCandlesUpdated`);
                    }
                    if (ticks.length) {
                        await this._processEvent(()=>listener.onTicksUpdated(instanceIndex, ticks, data.equity, data.margin, data.freeMargin, data.marginLevel, data.accountCurrencyExchangeRate), `${primaryAccountId}:${instanceIndex}:onTicksUpdated`);
                    }
                    if (books.length) {
                        await this._processEvent(()=>listener.onBooksUpdated(instanceIndex, books, data.equity, data.margin, data.freeMargin, data.marginLevel, data.accountCurrencyExchangeRate), `${primaryAccountId}:${instanceIndex}:onBooksUpdated`);
                    }
                }
                for (let price of prices){
                    for (let listener of this._synchronizationListeners[primaryAccountId] || []){
                        await this._processEvent(()=>listener.onSymbolPriceUpdated(instanceIndex, price), `${primaryAccountId}:${instanceIndex}:onSymbolPriceUpdated`);
                    }
                }
                for (let price of prices){
                    if (price.timestamps) {
                        price.timestamps.clientProcessingFinished = new Date();
                        // eslint-disable-next-line max-depth
                        for (let listener of this._latencyListeners || []){
                            await this._processEvent(()=>listener.onSymbolPrice(data.accountId, price.symbol, price.timestamps), `${primaryAccountId}:${instanceIndex}:onSymbolPrice`);
                        }
                    }
                }
            }
        } catch (err) {
            this._logger.error("Failed to process incoming synchronization packet", err);
        }
    }
    async _processEvent(callable, label, throwError) {
        const startTime = Date.now();
        let isLongEvent = false;
        let isEventDone = false;
        const checkLongEvent = async ()=>{
            await new Promise((res)=>setTimeout(res, 1000));
            if (!isEventDone) {
                isLongEvent = true;
                this._logger.warn(`${label}: event is taking more than 1 second to process`);
            }
        };
        checkLongEvent();
        try {
            await callable();
        } catch (err) {
            if (throwError) {
                throw err;
            }
            this._logger.error(`${label}: event failed with error`, err);
        }
        isEventDone = true;
        if (isLongEvent) {
            this._logger.warn(`${label}: finished in ${Math.floor((Date.now() - startTime) / 1000)} seconds`);
        }
    }
    async _fireReconnected(instanceNumber, socketInstanceIndex, region) {
        try {
            const reconnectListeners = [];
            for (let listener of this._reconnectListeners){
                if (this._socketInstancesByAccounts[instanceNumber][listener.accountId] === socketInstanceIndex && this.getAccountRegion(listener.accountId) === region) {
                    reconnectListeners.push(listener);
                }
            }
            Object.keys(this._synchronizationFlags).forEach((synchronizationId)=>{
                const accountId = this._synchronizationFlags[synchronizationId].accountId;
                if (this._socketInstancesByAccounts[instanceNumber][accountId] === socketInstanceIndex && this._synchronizationFlags[synchronizationId].instanceNumber === instanceNumber && this._regionsByAccounts[accountId] && this._regionsByAccounts[accountId].region === region) {
                    delete this._synchronizationFlags[synchronizationId];
                }
            });
            const reconnectAccountIds = reconnectListeners.map((listener)=>listener.accountId);
            this._subscriptionManager.onReconnected(instanceNumber, socketInstanceIndex, reconnectAccountIds);
            this._packetOrderer.onReconnected(reconnectAccountIds);
            for (let listener of reconnectListeners){
                Promise.resolve(listener.listener.onReconnected(region, instanceNumber)).catch((err)=>this._logger.error("Failed to notify reconnect listener", err));
            }
        } catch (err) {
            this._logger.error("Failed to process reconnected event", err);
        }
    }
    _getSocketInstanceByAccount(accountId, instanceNumber) {
        const region = this.getAccountRegion(accountId);
        return this._socketInstances[region][instanceNumber][this._socketInstancesByAccounts[instanceNumber][accountId]];
    }
    async getUrlSettings(instanceNumber, region) {
        if (this._url) {
            return {
                url: this._url,
                isSharedClientApi: true
            };
        }
        const urlSettings = await this._domainClient.getSettings();
        const getUrl = (hostname)=>`https://${hostname}.${region}-${String.fromCharCode(97 + Number(instanceNumber))}.${urlSettings.domain}`;
        let url;
        if (this._useSharedClientApi) {
            url = getUrl(this._hostname);
        } else {
            url = getUrl(urlSettings.hostname);
        }
        const isSharedClientApi = url === getUrl(this._hostname);
        return {
            url,
            isSharedClientApi
        };
    }
    // eslint-disable-next-line complexity
    async _getServerUrl(instanceNumber, socketInstanceIndex, region) {
        if (this._url) {
            return this._url;
        }
        while(this.socketInstances[region][instanceNumber][socketInstanceIndex].connected){
            try {
                const urlSettings = await this.getUrlSettings(instanceNumber, region);
                const url = urlSettings.url;
                const isSharedClientApi = urlSettings.isSharedClientApi;
                let logMessage = "Connecting MetaApi websocket client to the MetaApi server " + `via ${url} ${isSharedClientApi ? "shared" : "dedicated"} server.`;
                if (this._firstConnect && !isSharedClientApi) {
                    logMessage += " Please note that it can take up to 3 minutes for your dedicated server to start for the " + "first time. During this time it is OK if you see some connection errors.";
                    this._firstConnect = false;
                }
                this._logger.info(logMessage);
                return url;
            } catch (err) {
                this._logger.error("Failed to retrieve server URL", err);
                await new Promise((res)=>setTimeout(res, 1000));
            }
        }
    }
    _throttleRequest(type, accountId, instanceNumber, timeInMs) {
        this._lastRequestsTime[instanceNumber] = this._lastRequestsTime[instanceNumber] || {};
        this._lastRequestsTime[instanceNumber][type] = this._lastRequestsTime[instanceNumber][type] || {};
        let lastTime = this._lastRequestsTime[instanceNumber][type][accountId];
        if (!lastTime || lastTime < Date.now() - timeInMs) {
            this._lastRequestsTime[instanceNumber][type][accountId] = Date.now();
            return !!lastTime;
        }
        return false;
    }
    _refreshAccountRegion(accountId) {
        if (this._regionsByAccounts[accountId]) {
            this._regionsByAccounts[accountId].lastUsed = Date.now();
        }
    }
    //eslint-disable-next-line complexity
    async _createSocketInstanceByAccount(accountId, instanceNumber) {
        const region = this.getAccountRegion(accountId);
        if (this._socketInstancesByAccounts[instanceNumber][accountId] === undefined) {
            let socketInstanceIndex = null;
            while(this._subscribeLock && (new Date(this._subscribeLock.recommendedRetryTime).getTime() > Date.now() && this.subscribedAccountIds(instanceNumber, undefined, region).length < this._subscribeLock.lockedAtAccounts || new Date(this._subscribeLock.lockedAtTime).getTime() + this._subscribeCooldownInSeconds * 1000 > Date.now() && this.subscribedAccountIds(instanceNumber, undefined, region).length >= this._subscribeLock.lockedAtAccounts)){
                await new Promise((res)=>setTimeout(res, 1000));
            }
            for(let index = 0; index < this._socketInstances[region][instanceNumber].length; index++){
                const accountCounter = this.getAssignedAccounts(instanceNumber, index, region).length;
                const instance = this.socketInstances[region][instanceNumber][index];
                if (instance.subscribeLock) {
                    if (instance.subscribeLock.type === "LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_USER_PER_SERVER" && (new Date(instance.subscribeLock.recommendedRetryTime).getTime() > Date.now() || this.subscribedAccountIds(instanceNumber, index, region).length >= instance.subscribeLock.lockedAtAccounts)) {
                        continue;
                    }
                    if (instance.subscribeLock.type === "LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_SERVER" && new Date(instance.subscribeLock.recommendedRetryTime).getTime() > Date.now() && this.subscribedAccountIds(instanceNumber, index, region).length >= instance.subscribeLock.lockedAtAccounts) {
                        continue;
                    }
                }
                if (accountCounter < this._maxAccountsPerInstance) {
                    socketInstanceIndex = index;
                    break;
                }
            }
            if (socketInstanceIndex === null) {
                socketInstanceIndex = this._socketInstances[region][instanceNumber].length;
                await this.connect(instanceNumber, region);
            }
            this._socketInstancesByAccounts[instanceNumber][accountId] = socketInstanceIndex;
        }
    }
    _clearAccountCacheJob() {
        const date = Date.now();
        Object.keys(this._regionsByAccounts).forEach((replicaId)=>{
            const data = this._regionsByAccounts[replicaId];
            if (data && data.connections === 0 && date - data.lastUsed > 2 * 60 * 60 * 1000) {
                const primaryAccountId = this._accountsByReplicaId[replicaId];
                const replicas = Object.values(this._accountReplicas[primaryAccountId] || {});
                replicas.forEach((replica)=>{
                    delete this._accountsByReplicaId[replica];
                    delete this._regionsByAccounts[replica];
                });
                delete this._accountReplicas[primaryAccountId];
                this._logger.debug(`${primaryAccountId}: removed expired account replicas data`);
            }
        });
    }
    _clearInactiveSyncDataJob() {
        const date = Date.now();
        Object.keys(this._synchronizationHashes).keys((synchronizationId)=>{
            if (this._synchronizationHashes[synchronizationId].lastUpdated < date - 30 * 60 * 1000) {
                delete this._synchronizationHashes[synchronizationId];
            }
        });
    }
    /**
   * @typedef MetaApiWebsocketClientOptions MetaApi websocket client options
   * @property {string} [region] region to connect
   * @property {Boolean} [disableInternalJobs] whether to not run internal interval jobs. Used for tests only
   */ /**
   * Constructs MetaApi websocket API client instance
   * @param {MetaApi} metaApi metaApi instance
   * @param {DomainClient} domainClient domain client
   * @param {String} token authorization token
   * @param {MetaApiWebsocketClientOptions} opts websocket client options
   */ // eslint-disable-next-line complexity,max-statements
    constructor(metaApi, domainClient, token, opts){
        const validator = new _optionsValidator.default();
        opts = opts || {};
        opts.packetOrderingTimeout = validator.validateNonZero(opts.packetOrderingTimeout, 60, "packetOrderingTimeout");
        opts.synchronizationThrottler = opts.synchronizationThrottler || {};
        this._domainClient = domainClient;
        this._application = opts.application || "MetaApi";
        this._domain = opts.domain || "agiliumtrade.agiliumtrade.ai";
        this._region = opts.region;
        this._hostname = "mt-client-api-v1";
        this._metaApi = metaApi;
        this._url = null;
        this._requestTimeout = validator.validateNonZero(opts.requestTimeout, 60, "requestTimeout") * 1000;
        this._connectTimeout = validator.validateNonZero(opts.connectTimeout, 60, "connectTimeout") * 1000;
        const retryOpts = opts.retryOpts || {};
        this._retries = validator.validateNumber(retryOpts.retries, 5, "retryOpts.retries");
        this._minRetryDelayInSeconds = validator.validateNonZero(retryOpts.minDelayInSeconds, 1, "retryOpts.minDelayInSeconds");
        this._maxRetryDelayInSeconds = validator.validateNonZero(retryOpts.maxDelayInSeconds, 30, "retryOpts.maxDelayInSeconds");
        this._maxAccountsPerInstance = 100;
        this._subscribeCooldownInSeconds = validator.validateNonZero(retryOpts.subscribeCooldownInSeconds, 600, "retryOpts.subscribeCooldownInSeconds");
        this._sequentialEventProcessing = true;
        this._useSharedClientApi = validator.validateBoolean(opts.useSharedClientApi, false, "useSharedClientApi");
        this._unsubscribeThrottlingInterval = validator.validateNonZero(opts.unsubscribeThrottlingIntervalInSeconds, 10, "unsubscribeThrottlingIntervalInSeconds") * 1000;
        this._socketMinimumReconnectTimeout = 500;
        this._latencyService = new _latencyService.default(this, token, this._connectTimeout);
        this._token = token;
        this._synchronizationListeners = {};
        this._latencyListeners = [];
        this._reconnectListeners = [];
        this._connectedHosts = {};
        this._socketInstances = {};
        this._socketInstancesByAccounts = {};
        this._regionsByAccounts = {};
        this._accountsByReplicaId = {};
        this._accountReplicas = {};
        this._synchronizationThrottlerOpts = opts.synchronizationThrottler;
        this._subscriptionManager = new _subscriptionManager.default(this, metaApi);
        this._statusTimers = {};
        this._eventQueues = {};
        this._synchronizationFlags = {};
        this._synchronizationIdByInstance = {};
        this._subscribeLock = null;
        this._firstConnect = true;
        this._lastRequestsTime = {};
        this._packetOrderer = new _packetOrderer.default(this, opts.packetOrderingTimeout);
        this._packetOrderer.start();
        this._synchronizationHashes = {};
        this._updateEvents = {};
        if (opts.packetLogger && opts.packetLogger.enabled) {
            this._packetLogger = new PacketLogger(opts.packetLogger);
            this._packetLogger.start();
        }
        this._logger = _logger.default.getLogger("MetaApiWebsocketClient");
        if (!opts.disableInternalJobs) {
            this._clearAccountCacheInterval = setInterval(this._clearAccountCacheJob.bind(this), 30 * 60 * 1000);
            this._clearInactiveSyncDataJob = setInterval(this._clearInactiveSyncDataJob.bind(this), 5 * 60 * 1000);
        }
    }
};

//# sourceMappingURL=data:application/json;base64,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