"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return ClientApiClient;
    }
});
const _metaApiclient = /*#__PURE__*/ _interop_require_default(require("../metaApi.client"));
const _logger = /*#__PURE__*/ _interop_require_default(require("../../logger"));
const _errorHandler = require("../errorHandler");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let ClientApiClient = class ClientApiClient extends _metaApiclient.default {
    /**
   * Type hashing ignored field lists
   * @typedef {Object} TypeHashingIgnoredFieldLists
   * @property {String[]} specification specification ignored fields
   * @property {String[]} position position ignored fields
   * @property {String[]} order order ignored fields
   */ /**
   * Hashing ignored field lists
   * @typedef {Object} HashingIgnoredFieldLists
   * @property {TypeHashingIgnoredFieldLists} g1 g1 hashing ignored field lists
   * @property {TypeHashingIgnoredFieldLists} g2 g2 hashing ignored field lists
   */ /**
   * Refreshes hashing ignored field lists
   * @param {String} region account region
   * @returns {Promise} promise resolving when the hashing field lists are updated.
   */ async refreshIgnoredFieldLists(region) {
        if (this._ignoredFieldListsCaches[region] && this._ignoredFieldListsCaches[region].requestPromise) {
            await this._ignoredFieldListsCaches[region].requestPromise;
        } else if (this._ignoredFieldListsCaches[region] && Date.now() - this._ignoredFieldListsCaches[region].lastUpdated < this._updateInterval) {
            return;
        } else {
            if (!this._ignoredFieldListsCaches[region]) {
                this._ignoredFieldListsCaches[region] = {
                    lastUpdated: 0,
                    data: null,
                    requestPromise: null,
                    updateJob: setInterval(()=>this._refreshIgnoredFieldListsJob(region), 60000)
                };
            }
            let resolve, reject;
            this._ignoredFieldListsCaches[region].requestPromise = new Promise((res, rej)=>{
                resolve = res, reject = rej;
            });
            let isCacheUpdated = false;
            while(!isCacheUpdated){
                try {
                    const host = await this._domainClient.getUrl(this._host, region);
                    const opts = {
                        url: `${host}/hashing-ignored-field-lists`,
                        method: "GET",
                        json: true,
                        headers: {
                            "auth-token": this._token
                        }
                    };
                    const response = await this._httpClient.request(opts, "getHashingIgnoredFieldLists");
                    this._ignoredFieldListsCaches[region] = {
                        lastUpdated: Date.now(),
                        data: response,
                        requestPromise: null
                    };
                    this._ignoredFieldListsFreshestCache = response;
                    resolve(response);
                    isCacheUpdated = true;
                    this._ignoredFieldListsCaches[region].retryIntervalInSeconds = this._retryIntervalInSeconds;
                } catch (err) {
                    this._logger.error("Failed to update hashing ignored field list", err);
                    this._ignoredFieldListsCaches[region].retryIntervalInSeconds = Math.min(this._ignoredFieldListsCaches[region].retryIntervalInSeconds * 2, 300);
                    await new Promise((res)=>setTimeout(res, this._ignoredFieldListsCaches[region].retryIntervalInSeconds * 1000));
                }
            }
        }
    }
    /**
   * Retrieves hashing ignored field lists
   * @param {String} region account region
   * @returns {HashingIgnoredFieldLists} promise resolving with hashing ignored field lists
   */ getHashingIgnoredFieldLists(region) {
        if (region === "combined") {
            if (this._ignoredFieldListsFreshestCache) {
                return this._ignoredFieldListsFreshestCache;
            } else {
                throw new _errorHandler.NotFoundError("Ignored field lists not found");
            }
        }
        if (this._ignoredFieldListsCaches[region] && this._ignoredFieldListsCaches[region].data) {
            return this._ignoredFieldListsCaches[region].data;
        } else {
            throw new _errorHandler.NotFoundError(`Ignored field lists for region ${region} not found`);
        }
    }
    async _refreshIgnoredFieldListsJob(region) {
        if (!this._ignoredFieldListsCaches[region].requestPromise && Date.now() - this._ignoredFieldListsCaches[region].lastUpdated > this._updateInterval) {
            await this.refreshIgnoredFieldLists(region);
        }
    }
    /**
   * Constructs client API client instance
   * @param {HttpClient} httpClient HTTP client
   * @param {DomainClient} domainClient domain client
   */ constructor(httpClient, domainClient){
        super(httpClient, domainClient);
        this._host = "https://mt-client-api-v1";
        this._retryIntervalInSeconds = 1;
        this._updateInterval = 60 * 60 * 1000;
        this._ignoredFieldListsCaches = {};
        this._ignoredFieldListsFreshestCache = null;
        this._logger = _logger.default.getLogger("ClientApiClient");
    }
};

//# sourceMappingURL=data:application/json;base64,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