"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    HttpClientMock: function() {
        return HttpClientMock;
    },
    default: function() {
        return HttpClient;
    }
});
const _axios = /*#__PURE__*/ _interop_require_default(require("axios"));
const _errorHandler = require("./errorHandler");
const _optionsValidator = /*#__PURE__*/ _interop_require_default(require("./optionsValidator"));
const _timeoutError = /*#__PURE__*/ _interop_require_default(require("./timeoutError"));
const _logger = /*#__PURE__*/ _interop_require_default(require("../logger"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let HttpClient = class HttpClient {
    /**
   * Performs a request. Response errors are returned as ApiError or subclasses.
   * @param {Object} options request options
   * @returns {Object|String|any} request result
   */ async request(options, type = "", retryCounter = 0, endTime = Date.now() + this._maxRetryDelay * this._retries, isLongRunning = false) {
        options.timeout = this._timeout;
        let retryAfterSeconds = 0;
        options.callback = (e, res)=>{
            if (res && res.status === 202) {
                retryAfterSeconds = res.headers["retry-after"];
                if (isNaN(retryAfterSeconds)) {
                    retryAfterSeconds = Math.max((new Date(retryAfterSeconds).getTime() - Date.now()) / 1000, 1);
                }
                if (!isLongRunning) {
                    endTime = Date.now() + this._longRunningRequestTimeout;
                    isLongRunning = true;
                }
            }
        };
        let body;
        try {
            const response = await this._makeRequest(options);
            options.callback(null, response);
            body = response && response.data || undefined;
        } catch (err) {
            retryCounter = await this._handleError(err, type, retryCounter, endTime);
            return this.request(options, type, retryCounter, endTime);
        }
        if (retryAfterSeconds) {
            if (body && body.message) {
                this._logger.info(`Retrying request in ${Math.floor(retryAfterSeconds)} seconds because request ` + "returned message:", body.message);
            }
            await this._handleRetry(endTime, retryAfterSeconds * 1000);
            body = await this.request(options, type, retryCounter, endTime, isLongRunning);
        }
        return body;
    }
    _makeRequest(options) {
        return (0, _axios.default)({
            transitional: {
                clarifyTimeoutError: true
            },
            ...options
        });
    }
    async _wait(pause) {
        await new Promise((res)=>setTimeout(res, pause));
    }
    async _handleRetry(endTime, retryAfter) {
        if (endTime > Date.now() + retryAfter) {
            await this._wait(retryAfter);
        } else {
            throw new _timeoutError.default("Timed out waiting for the response");
        }
    }
    async _handleError(err, type, retryCounter, endTime) {
        const error = this._convertError(err);
        if ([
            "ConflictError",
            "InternalError",
            "ApiError",
            "TimeoutError"
        ].includes(error.name) && retryCounter < this._retries) {
            const pause = Math.min(Math.pow(2, retryCounter) * this._minRetryDelay, this._maxRetryDelay);
            await this._wait(pause);
            return retryCounter + 1;
        } else if (error.name === "TooManyRequestsError") {
            const retryTime = Date.parse(error.metadata.recommendedRetryTime);
            if (retryTime < endTime) {
                this._logger.debug(`${type} request has failed with TooManyRequestsError (HTTP status code 429). ` + `Will retry request in ${Math.ceil((retryTime - Date.now()) / 1000)} seconds`);
                await this._wait(retryTime - Date.now());
                return retryCounter;
            }
        }
        throw error;
    }
    // eslint-disable-next-line complexity
    _convertError(err) {
        var _err_config;
        const errorResponse = err.response || {};
        const errorData = errorResponse.data || {};
        const status = errorResponse.status || err.status;
        const url = err === null || err === void 0 ? void 0 : (_err_config = err.config) === null || _err_config === void 0 ? void 0 : _err_config.url;
        const errMsg = errorData.message || err.message;
        const errMsgDefault = errorData.message || err.code || err.message;
        switch(status){
            case 400:
                return new _errorHandler.ValidationError(errMsg, errorData.details || err.details, url);
            case 401:
                return new _errorHandler.UnauthorizedError(errMsg, url);
            case 403:
                return new _errorHandler.ForbiddenError(errMsg, url);
            case 404:
                return new _errorHandler.NotFoundError(errMsg, url);
            case 409:
                return new _errorHandler.ConflictError(errMsg, url);
            case 429:
                return new _errorHandler.TooManyRequestsError(errMsg, errorData.metadata || err.metadata, url);
            case 500:
                return new _errorHandler.InternalError(errMsg, url);
            default:
                return new _errorHandler.ApiError(_errorHandler.ApiError, errMsgDefault, status, url);
        }
    }
    /**
   * @typedef {Object} RetryOptions retry options
   * @property {Number} [retries] the number of attempts to retry failed request, default 5
   * @property {Number} [minDelayInSeconds] minimum delay in seconds before retrying, default 1
   * @property {Number} [maxDelayInSeconds] maximum delay in seconds before retrying, default 30
   * @property {Number} [longRunningRequestTimeoutInMinutes] timeout in minutes for long running requests, default 10
   * @property {Number} [subscribeCooldownInSeconds] time to disable new subscriptions for
   */ /**
   * Constructs HttpClient class instance
   * @param {Number} timeout request timeout in seconds
   * @param {RetryOptions} [retryOpts] retry options
   */ constructor(timeout = 60, retryOpts = {}){
        const validator = new _optionsValidator.default();
        this._timeout = timeout * 1000;
        this._retries = validator.validateNumber(retryOpts.retries, 5, "retryOpts.retries");
        this._minRetryDelay = validator.validateNonZero(retryOpts.minDelayInSeconds, 1, "retryOpts.minDelayInSeconds") * 1000;
        this._maxRetryDelay = validator.validateNonZero(retryOpts.maxDelayInSeconds, 30, "retryOpts.maxDelayInSeconds") * 1000;
        this._longRunningRequestTimeout = validator.validateNumber(retryOpts.longRunningRequestTimeoutInMinutes, 10, "retryOpts.longRunningRequestTimeoutInMinutes") * 60 * 1000;
        this._logger = _logger.default.getLogger("HttpClient");
    }
};
let HttpClientMock = class HttpClientMock extends HttpClient {
    _makeRequest() {
        return this._requestFn.apply(this, arguments);
    }
    /**
   * Constructs HTTP client mock
   * @param {Function(options:Object):Promise} requestFn mocked request function
   * @param {Number} timeout request timeout in seconds
   * @param {RetryOptions} retryOpts retry options
   */ constructor(requestFn, timeout, retryOpts){
        super(timeout, retryOpts);
        this._requestFn = requestFn;
    }
};

//# sourceMappingURL=data:application/json;base64,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