"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return DomainClient;
    }
});
const _logger = /*#__PURE__*/ _interop_require_default(require("../logger"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let DomainClient = class DomainClient {
    /**
   * Returns domain client domain
   * @returns {String} client domain
   */ get domain() {
        return this._domain;
    }
    /**
   * Returns domain client token
   * @returns {String} client token
   */ get token() {
        return this._token;
    }
    /**
   * Returns the API URL
   * @param {String} host REST API host
   * @param {String} region host region
   * @returns {String} API URL
   */ async getUrl(host, region) {
        await this._updateDomain();
        return `${host}.${region}.${this._urlCache.domain}`;
    }
    /**
   * Domain settings
   * @typedef {Object} DomainSettings
   * @property {String} hostname client api host name
   * @property {String} domain client api domain for regions
   */ /**
   * Returns domain settings
   * @returns {DomainSettings} domain settings
   */ async getSettings() {
        await this._updateDomain();
        return {
            domain: this._urlCache.domain,
            hostname: this._urlCache.hostname
        };
    }
    async _updateDomain() {
        if (!this._urlCache.domain || this._urlCache.lastUpdated < Date.now() - 1000 * 60 * 10) {
            if (this._urlCache.requestPromise) {
                await this._urlCache.requestPromise;
            } else {
                let resolve, reject;
                this._urlCache.requestPromise = new Promise((res, rej)=>{
                    resolve = res, reject = rej;
                });
                let isCacheUpdated = false;
                while(!isCacheUpdated){
                    const opts = {
                        url: `https://mt-provisioning-api-v1.${this._domain}/users/current/servers/mt-client-api`,
                        method: "GET",
                        headers: {
                            "auth-token": this._token
                        },
                        json: true
                    };
                    try {
                        const urlSettings = await this._httpClient.request(opts, "_updateDomain");
                        this._urlCache = {
                            domain: urlSettings.domain,
                            hostname: urlSettings.hostname,
                            requestPromise: null,
                            lastUpdated: Date.now()
                        };
                        resolve();
                        isCacheUpdated = true;
                        this._retryIntervalInSeconds = 1;
                    } catch (err) {
                        this._logger.error("Failed to update domain settings cache", err);
                        this._retryIntervalInSeconds = Math.min(this._retryIntervalInSeconds * 2, 300);
                        await new Promise((res)=>setTimeout(res, this._retryIntervalInSeconds * 1000));
                    }
                }
            }
        }
    }
    /**
   * Constructs domain client instance
   * @param {HttpClient} httpClient HTTP client
   * @param {String} token authorization token
   * @param {String} domain domain to connect to, default is agiliumtrade.agiliumtrade.ai
   */ constructor(httpClient, token, domain = "agiliumtrade.agiliumtrade.ai"){
        this._httpClient = httpClient;
        this._domain = domain;
        this._token = token;
        this._urlCache = {
            domain: null,
            hostname: null,
            requestPromise: null,
            lastUpdated: 0
        };
        this._retryIntervalInSeconds = 1;
        this._logger = _logger.default.getLogger("DomainClient");
    }
};

//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIjxhbm9uPiJdLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbmltcG9ydCBMb2dnZXJNYW5hZ2VyIGZyb20gJy4uL2xvZ2dlcic7XG5cbi8qKlxuICogQ29ubmVjdGlvbiBVUkwgbWFuYWdpbmcgY2xpZW50XG4gKi9cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIERvbWFpbkNsaWVudCB7XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdHMgZG9tYWluIGNsaWVudCBpbnN0YW5jZVxuICAgKiBAcGFyYW0ge0h0dHBDbGllbnR9IGh0dHBDbGllbnQgSFRUUCBjbGllbnRcbiAgICogQHBhcmFtIHtTdHJpbmd9IHRva2VuIGF1dGhvcml6YXRpb24gdG9rZW5cbiAgICogQHBhcmFtIHtTdHJpbmd9IGRvbWFpbiBkb21haW4gdG8gY29ubmVjdCB0bywgZGVmYXVsdCBpcyBhZ2lsaXVtdHJhZGUuYWdpbGl1bXRyYWRlLmFpXG4gICAqL1xuICBjb25zdHJ1Y3RvcihodHRwQ2xpZW50LCB0b2tlbiwgZG9tYWluID0gJ2FnaWxpdW10cmFkZS5hZ2lsaXVtdHJhZGUuYWknKSB7XG4gICAgdGhpcy5faHR0cENsaWVudCA9IGh0dHBDbGllbnQ7XG4gICAgdGhpcy5fZG9tYWluID0gZG9tYWluO1xuICAgIHRoaXMuX3Rva2VuID0gdG9rZW47XG4gICAgdGhpcy5fdXJsQ2FjaGUgPSB7XG4gICAgICBkb21haW46IG51bGwsXG4gICAgICBob3N0bmFtZTogbnVsbCxcbiAgICAgIHJlcXVlc3RQcm9taXNlOiBudWxsLFxuICAgICAgbGFzdFVwZGF0ZWQ6IDBcbiAgICB9O1xuICAgIHRoaXMuX3JldHJ5SW50ZXJ2YWxJblNlY29uZHMgPSAxO1xuICAgIHRoaXMuX2xvZ2dlciA9IExvZ2dlck1hbmFnZXIuZ2V0TG9nZ2VyKCdEb21haW5DbGllbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGRvbWFpbiBjbGllbnQgZG9tYWluXG4gICAqIEByZXR1cm5zIHtTdHJpbmd9IGNsaWVudCBkb21haW5cbiAgICovXG4gIGdldCBkb21haW4oKSB7XG4gICAgcmV0dXJuIHRoaXMuX2RvbWFpbjtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGRvbWFpbiBjbGllbnQgdG9rZW5cbiAgICogQHJldHVybnMge1N0cmluZ30gY2xpZW50IHRva2VuXG4gICAqL1xuICBnZXQgdG9rZW4oKSB7XG4gICAgcmV0dXJuIHRoaXMuX3Rva2VuO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIEFQSSBVUkxcbiAgICogQHBhcmFtIHtTdHJpbmd9IGhvc3QgUkVTVCBBUEkgaG9zdFxuICAgKiBAcGFyYW0ge1N0cmluZ30gcmVnaW9uIGhvc3QgcmVnaW9uXG4gICAqIEByZXR1cm5zIHtTdHJpbmd9IEFQSSBVUkxcbiAgICovXG4gIGFzeW5jIGdldFVybChob3N0LCByZWdpb24pIHtcbiAgICBhd2FpdCB0aGlzLl91cGRhdGVEb21haW4oKTtcbiAgICByZXR1cm4gYCR7aG9zdH0uJHtyZWdpb259LiR7dGhpcy5fdXJsQ2FjaGUuZG9tYWlufWA7XG4gIH1cblxuICAvKipcbiAgICogRG9tYWluIHNldHRpbmdzXG4gICAqIEB0eXBlZGVmIHtPYmplY3R9IERvbWFpblNldHRpbmdzXG4gICAqIEBwcm9wZXJ0eSB7U3RyaW5nfSBob3N0bmFtZSBjbGllbnQgYXBpIGhvc3QgbmFtZVxuICAgKiBAcHJvcGVydHkge1N0cmluZ30gZG9tYWluIGNsaWVudCBhcGkgZG9tYWluIGZvciByZWdpb25zXG4gICAqL1xuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGRvbWFpbiBzZXR0aW5nc1xuICAgKiBAcmV0dXJucyB7RG9tYWluU2V0dGluZ3N9IGRvbWFpbiBzZXR0aW5nc1xuICAgKi9cbiAgYXN5bmMgZ2V0U2V0dGluZ3MoKSB7XG4gICAgYXdhaXQgdGhpcy5fdXBkYXRlRG9tYWluKCk7XG4gICAgcmV0dXJuIHtkb21haW46IHRoaXMuX3VybENhY2hlLmRvbWFpbiwgaG9zdG5hbWU6IHRoaXMuX3VybENhY2hlLmhvc3RuYW1lfTtcbiAgfVxuXG4gIGFzeW5jIF91cGRhdGVEb21haW4oKSB7XG4gICAgaWYoIXRoaXMuX3VybENhY2hlLmRvbWFpbiB8fCB0aGlzLl91cmxDYWNoZS5sYXN0VXBkYXRlZCA8IERhdGUubm93KCkgLSAxMDAwICogNjAgKiAxMCkge1xuICAgICAgaWYodGhpcy5fdXJsQ2FjaGUucmVxdWVzdFByb21pc2UpIHtcbiAgICAgICAgYXdhaXQgdGhpcy5fdXJsQ2FjaGUucmVxdWVzdFByb21pc2U7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBsZXQgcmVzb2x2ZSwgcmVqZWN0O1xuICAgICAgICB0aGlzLl91cmxDYWNoZS5yZXF1ZXN0UHJvbWlzZSA9IG5ldyBQcm9taXNlKChyZXMsIHJlaikgPT4ge1xuICAgICAgICAgIHJlc29sdmUgPSByZXMsIHJlamVjdCA9IHJlajtcbiAgICAgICAgfSk7XG4gICAgICAgIGxldCBpc0NhY2hlVXBkYXRlZCA9IGZhbHNlO1xuICAgICAgICB3aGlsZSghaXNDYWNoZVVwZGF0ZWQpIHtcbiAgICAgICAgICBjb25zdCBvcHRzID0ge1xuICAgICAgICAgICAgdXJsOiBgaHR0cHM6Ly9tdC1wcm92aXNpb25pbmctYXBpLXYxLiR7dGhpcy5fZG9tYWlufS91c2Vycy9jdXJyZW50L3NlcnZlcnMvbXQtY2xpZW50LWFwaWAsXG4gICAgICAgICAgICBtZXRob2Q6ICdHRVQnLFxuICAgICAgICAgICAgaGVhZGVyczoge1xuICAgICAgICAgICAgICAnYXV0aC10b2tlbic6IHRoaXMuX3Rva2VuXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAganNvbjogdHJ1ZSxcbiAgICAgICAgICB9O1xuICBcbiAgICAgICAgICB0cnkge1xuICAgICAgICAgICAgY29uc3QgdXJsU2V0dGluZ3MgPSBhd2FpdCB0aGlzLl9odHRwQ2xpZW50LnJlcXVlc3Qob3B0cywgJ191cGRhdGVEb21haW4nKTtcbiAgICAgICAgICAgIHRoaXMuX3VybENhY2hlID0ge1xuICAgICAgICAgICAgICBkb21haW46IHVybFNldHRpbmdzLmRvbWFpbixcbiAgICAgICAgICAgICAgaG9zdG5hbWU6IHVybFNldHRpbmdzLmhvc3RuYW1lLFxuICAgICAgICAgICAgICByZXF1ZXN0UHJvbWlzZTogbnVsbCxcbiAgICAgICAgICAgICAgbGFzdFVwZGF0ZWQ6IERhdGUubm93KClcbiAgICAgICAgICAgIH07IFxuICAgICAgICAgICAgcmVzb2x2ZSgpO1xuICAgICAgICAgICAgaXNDYWNoZVVwZGF0ZWQgPSB0cnVlO1xuICAgICAgICAgICAgdGhpcy5fcmV0cnlJbnRlcnZhbEluU2Vjb25kcyA9IDE7XG4gICAgICAgICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICAgICAgICB0aGlzLl9sb2dnZXIuZXJyb3IoJ0ZhaWxlZCB0byB1cGRhdGUgZG9tYWluIHNldHRpbmdzIGNhY2hlJywgZXJyKTtcbiAgICAgICAgICAgIHRoaXMuX3JldHJ5SW50ZXJ2YWxJblNlY29uZHMgPSBNYXRoLm1pbih0aGlzLl9yZXRyeUludGVydmFsSW5TZWNvbmRzICogMiwgMzAwKTtcbiAgICAgICAgICAgIGF3YWl0IG5ldyBQcm9taXNlKHJlcyA9PiBzZXRUaW1lb3V0KHJlcywgdGhpcy5fcmV0cnlJbnRlcnZhbEluU2Vjb25kcyAqIDEwMDApKTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gIH1cblxufSJdLCJuYW1lcyI6WyJEb21haW5DbGllbnQiLCJkb21haW4iLCJfZG9tYWluIiwidG9rZW4iLCJfdG9rZW4iLCJnZXRVcmwiLCJob3N0IiwicmVnaW9uIiwiX3VwZGF0ZURvbWFpbiIsIl91cmxDYWNoZSIsImdldFNldHRpbmdzIiwiaG9zdG5hbWUiLCJsYXN0VXBkYXRlZCIsIkRhdGUiLCJub3ciLCJyZXF1ZXN0UHJvbWlzZSIsInJlc29sdmUiLCJyZWplY3QiLCJQcm9taXNlIiwicmVzIiwicmVqIiwiaXNDYWNoZVVwZGF0ZWQiLCJvcHRzIiwidXJsIiwibWV0aG9kIiwiaGVhZGVycyIsImpzb24iLCJ1cmxTZXR0aW5ncyIsIl9odHRwQ2xpZW50IiwicmVxdWVzdCIsIl9yZXRyeUludGVydmFsSW5TZWNvbmRzIiwiZXJyIiwiX2xvZ2dlciIsImVycm9yIiwiTWF0aCIsIm1pbiIsInNldFRpbWVvdXQiLCJjb25zdHJ1Y3RvciIsImh0dHBDbGllbnQiLCJMb2dnZXJNYW5hZ2VyIiwiZ2V0TG9nZ2VyIl0sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7OztlQU9xQkE7OzsrREFMSzs7Ozs7O0FBS1gsSUFBQSxBQUFNQSxlQUFOLE1BQU1BO0lBc0JuQjs7O0dBR0MsR0FDRCxJQUFJQyxTQUFTO1FBQ1gsT0FBTyxJQUFJLENBQUNDLE9BQU87SUFDckI7SUFFQTs7O0dBR0MsR0FDRCxJQUFJQyxRQUFRO1FBQ1YsT0FBTyxJQUFJLENBQUNDLE1BQU07SUFDcEI7SUFFQTs7Ozs7R0FLQyxHQUNELE1BQU1DLE9BQU9DLElBQUksRUFBRUMsTUFBTSxFQUFFO1FBQ3pCLE1BQU0sSUFBSSxDQUFDQyxhQUFhO1FBQ3hCLE9BQU8sQ0FBQyxFQUFFRixLQUFLLENBQUMsRUFBRUMsT0FBTyxDQUFDLEVBQUUsSUFBSSxDQUFDRSxTQUFTLENBQUNSLE1BQU0sQ0FBQyxDQUFDO0lBQ3JEO0lBRUE7Ozs7O0dBS0MsR0FFRDs7O0dBR0MsR0FDRCxNQUFNUyxjQUFjO1FBQ2xCLE1BQU0sSUFBSSxDQUFDRixhQUFhO1FBQ3hCLE9BQU87WUFBQ1AsUUFBUSxJQUFJLENBQUNRLFNBQVMsQ0FBQ1IsTUFBTTtZQUFFVSxVQUFVLElBQUksQ0FBQ0YsU0FBUyxDQUFDRSxRQUFRO1FBQUE7SUFDMUU7SUFFQSxNQUFNSCxnQkFBZ0I7UUFDcEIsSUFBRyxDQUFDLElBQUksQ0FBQ0MsU0FBUyxDQUFDUixNQUFNLElBQUksSUFBSSxDQUFDUSxTQUFTLENBQUNHLFdBQVcsR0FBR0MsS0FBS0MsR0FBRyxLQUFLLE9BQU8sS0FBSyxJQUFJO1lBQ3JGLElBQUcsSUFBSSxDQUFDTCxTQUFTLENBQUNNLGNBQWMsRUFBRTtnQkFDaEMsTUFBTSxJQUFJLENBQUNOLFNBQVMsQ0FBQ00sY0FBYztZQUNyQyxPQUFPO2dCQUNMLElBQUlDLFNBQVNDO2dCQUNiLElBQUksQ0FBQ1IsU0FBUyxDQUFDTSxjQUFjLEdBQUcsSUFBSUcsUUFBUSxDQUFDQyxLQUFLQztvQkFDaERKLFVBQVVHLEtBQUtGLFNBQVNHO2dCQUMxQjtnQkFDQSxJQUFJQyxpQkFBaUI7Z0JBQ3JCLE1BQU0sQ0FBQ0EsZUFBZ0I7b0JBQ3JCLE1BQU1DLE9BQU87d0JBQ1hDLEtBQUssQ0FBQywrQkFBK0IsRUFBRSxJQUFJLENBQUNyQixPQUFPLENBQUMsb0NBQW9DLENBQUM7d0JBQ3pGc0IsUUFBUTt3QkFDUkMsU0FBUzs0QkFDUCxjQUFjLElBQUksQ0FBQ3JCLE1BQU07d0JBQzNCO3dCQUNBc0IsTUFBTTtvQkFDUjtvQkFFQSxJQUFJO3dCQUNGLE1BQU1DLGNBQWMsTUFBTSxJQUFJLENBQUNDLFdBQVcsQ0FBQ0MsT0FBTyxDQUFDUCxNQUFNO3dCQUN6RCxJQUFJLENBQUNiLFNBQVMsR0FBRzs0QkFDZlIsUUFBUTBCLFlBQVkxQixNQUFNOzRCQUMxQlUsVUFBVWdCLFlBQVloQixRQUFROzRCQUM5QkksZ0JBQWdCOzRCQUNoQkgsYUFBYUMsS0FBS0MsR0FBRzt3QkFDdkI7d0JBQ0FFO3dCQUNBSyxpQkFBaUI7d0JBQ2pCLElBQUksQ0FBQ1MsdUJBQXVCLEdBQUc7b0JBQ2pDLEVBQUUsT0FBT0MsS0FBSzt3QkFDWixJQUFJLENBQUNDLE9BQU8sQ0FBQ0MsS0FBSyxDQUFDLDBDQUEwQ0Y7d0JBQzdELElBQUksQ0FBQ0QsdUJBQXVCLEdBQUdJLEtBQUtDLEdBQUcsQ0FBQyxJQUFJLENBQUNMLHVCQUF1QixHQUFHLEdBQUc7d0JBQzFFLE1BQU0sSUFBSVosUUFBUUMsQ0FBQUEsTUFBT2lCLFdBQVdqQixLQUFLLElBQUksQ0FBQ1csdUJBQXVCLEdBQUc7b0JBQzFFO2dCQUNGO1lBQ0Y7UUFDRjtJQUNGO0lBdEdBOzs7OztHQUtDLEdBQ0RPLFlBQVlDLFVBQVUsRUFBRW5DLEtBQUssRUFBRUYsU0FBUyw4QkFBOEIsQ0FBRTtRQUN0RSxJQUFJLENBQUMyQixXQUFXLEdBQUdVO1FBQ25CLElBQUksQ0FBQ3BDLE9BQU8sR0FBR0Q7UUFDZixJQUFJLENBQUNHLE1BQU0sR0FBR0Q7UUFDZCxJQUFJLENBQUNNLFNBQVMsR0FBRztZQUNmUixRQUFRO1lBQ1JVLFVBQVU7WUFDVkksZ0JBQWdCO1lBQ2hCSCxhQUFhO1FBQ2Y7UUFDQSxJQUFJLENBQUNrQix1QkFBdUIsR0FBRztRQUMvQixJQUFJLENBQUNFLE9BQU8sR0FBR08sZUFBYSxDQUFDQyxTQUFTLENBQUM7SUFDekM7QUFzRkYifQ==