'use strict';
function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
import SynchronizationListener from '../clients/metaApi/synchronizationListener';
import LoggerManager from '../logger';
let MetaApiConnection = class MetaApiConnection extends SynchronizationListener {
    /**
   * Opens the connection. Can only be called the first time, next calls will be ignored.
   * @param {string} instanceId connection instance id
   * @return {Promise} promise resolving when the connection is opened
   */ connect(instanceId) {
        return _async_to_generator(function*() {})();
    }
    /**
   * Closes the connection. The instance of the class should no longer be used after this method is invoked.
   * @param {string} instanceId connection instance id
   */ close(instanceId) {
        return _async_to_generator(function*() {})();
    }
    /**
   * Returns MetaApi account
   * @return {MetatraderAccount} MetaApi account
   */ get account() {
        return this._account;
    }
    /**
   * Returns connection application
   * @return {String} connection application
   */ get application() {
        return this._application;
    }
    /**
   * Schedules the refresh task
   * @param {string} region replica region
   */ scheduleRefresh(region) {
        if (!this._refreshTasks[region]) {
            var _this__options_connections;
            var _this__options_connections_refreshReplicasMaxDelayInMs;
            const delay = Math.random() * ((_this__options_connections_refreshReplicasMaxDelayInMs = (_this__options_connections = this._options.connections) === null || _this__options_connections === void 0 ? void 0 : _this__options_connections.refreshReplicasMaxDelayInMs) !== null && _this__options_connections_refreshReplicasMaxDelayInMs !== void 0 ? _this__options_connections_refreshReplicasMaxDelayInMs : 6 * 60 * 60 * 1000);
            this._refreshTasks[region] = setTimeout(this._refreshReplicas, delay);
        }
    }
    /**
   * Cancels the scheduled refresh task
   * @param {string} region replica region
   */ cancelRefresh(region) {
        clearTimeout(this._refreshTasks[region]);
        delete this._refreshTasks[region];
    }
    /**
   * Refreshes account replicas
   */ _refreshReplicas() {
        var _this = this;
        return _async_to_generator(function*() {
            Object.values(_this._refreshTasks).forEach((task)=>clearTimeout(task));
            _this._refreshTasks = {};
            const oldReplicas = {};
            _this._account.replicas.forEach((replica)=>oldReplicas[replica.region] = replica.id);
            const newReplicas = {};
            let isAccountUpdated = false;
            try {
                yield _this._account.reload();
                isAccountUpdated = true;
                _this._account.replicas.forEach((replica)=>newReplicas[replica.region] = replica.id);
            } catch (error) {
                if (error.name === 'NotFoundError') {
                    if (_this._connectionRegistry) {
                        _this._connectionRegistry.closeAllInstances(_this._account.id);
                    }
                }
            }
            if (isAccountUpdated) {
                const deletedReplicas = {};
                const addedReplicas = {};
                Object.keys(oldReplicas).forEach((key)=>{
                    if (newReplicas[key] !== oldReplicas[key]) {
                        deletedReplicas[key] = oldReplicas[key];
                    }
                });
                Object.keys(newReplicas).forEach((key)=>{
                    if (newReplicas[key] !== oldReplicas[key]) {
                        addedReplicas[key] = newReplicas[key];
                    }
                });
                if (Object.keys(deletedReplicas).length) {
                    Object.values(deletedReplicas).forEach((replicaId)=>_this._websocketClient.onAccountDeleted(replicaId));
                }
                if (Object.keys(deletedReplicas).length || Object.keys(addedReplicas).length) {
                    newReplicas[_this._account.region] = _this._account.id;
                    _this._websocketClient.updateAccountCache(_this._account.id, newReplicas);
                    Object.entries(_this._account.accountRegions).forEach(([region, instance])=>{
                        if (!_this._options.region || _this._options.region === region) {
                            _this._websocketClient.ensureSubscribe(instance, 0);
                            _this._websocketClient.ensureSubscribe(instance, 1);
                        }
                    });
                }
            }
        })();
    }
    _ensureSynchronized(instanceIndex, key) {
        var _this = this;
        return _async_to_generator(function*() {
            let state = _this._getState(instanceIndex);
            if (state && !_this._closed) {
                try {
                    const synchronizationResult = yield _this.synchronize(instanceIndex);
                    if (synchronizationResult) {
                        state.synchronized = true;
                        state.synchronizationRetryIntervalInSeconds = 1;
                    }
                } catch (err) {
                    const level = _this._latencyService.getSynchronizedAccountInstances(_this._account.id).length ? 'debug' : 'error';
                    _this._logger[level]('MetaApi websocket client for account ' + _this._account.id + ':' + instanceIndex + ' failed to synchronize', err);
                    if (state.shouldSynchronize === key) {
                        setTimeout(_this._ensureSynchronized.bind(_this, instanceIndex, key), state.synchronizationRetryIntervalInSeconds * 1000);
                        state.synchronizationRetryIntervalInSeconds = Math.min(state.synchronizationRetryIntervalInSeconds * 2, 300);
                    }
                }
            }
        })();
    }
    _getState(instanceIndex) {
        if (!this._stateByInstanceIndex['' + instanceIndex]) {
            this._stateByInstanceIndex['' + instanceIndex] = {
                instanceIndex,
                ordersSynchronized: {},
                dealsSynchronized: {},
                shouldSynchronize: undefined,
                synchronizationRetryIntervalInSeconds: 1,
                synchronized: false,
                lastDisconnectedSynchronizationId: undefined,
                lastSynchronizationId: undefined,
                disconnected: false
            };
        }
        return this._stateByInstanceIndex['' + instanceIndex];
    }
    _checkIsConnectionActive() {
        if (!this._opened) {
            throw new Error('This connection has not been initialized yet, please invoke await connection.connect()');
        }
        if (this._closed) {
            throw new Error('This connection has been closed, please create a new connection');
        }
    }
    /**
   * @typedef Config MetaApi options for connections
   * @property {Options} [connections] MetaApi connections options. Only for tests. Will be ignored when set in SDK
   */ /**
   * @typedef Options MetaApiConnection options
   * @property {number} [refreshReplicasMaxDelayInMs = 6 * 60 * 60 * 1000] max delay before refreshing replicas delay
   */ /**
   * Constructs MetaApi MetaTrader Api connection
   * @param {MetaApiOpts & Config} options MetaApi options
   * @param {MetaApiWebsocketClient} websocketClient MetaApi websocket client
   * @param {MetatraderAccount} account MetaTrader account id to connect to
   * @param {String} [application] application to use
   */ constructor(options, websocketClient, account, application){
        super();
        this._options = options;
        this._websocketClient = websocketClient;
        this._latencyService = websocketClient.latencyService;
        this._account = account;
        this._logger = LoggerManager.getLogger('MetaApiConnection');
        this._application = application;
        this._refreshReplicas = this._refreshReplicas.bind(this);
        this._refreshTasks = {};
    }
};
/**
 * Exposes MetaApi MetaTrader API connection to consumers
 */ export { MetaApiConnection as default };

//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIjxhbm9uPiJdLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbmltcG9ydCBTeW5jaHJvbml6YXRpb25MaXN0ZW5lciBmcm9tICcuLi9jbGllbnRzL21ldGFBcGkvc3luY2hyb25pemF0aW9uTGlzdGVuZXInO1xuaW1wb3J0IExvZ2dlck1hbmFnZXIgZnJvbSAnLi4vbG9nZ2VyJztcblxuLyoqXG4gKiBFeHBvc2VzIE1ldGFBcGkgTWV0YVRyYWRlciBBUEkgY29ubmVjdGlvbiB0byBjb25zdW1lcnNcbiAqL1xuZXhwb3J0IGRlZmF1bHQgY2xhc3MgTWV0YUFwaUNvbm5lY3Rpb24gZXh0ZW5kcyBTeW5jaHJvbml6YXRpb25MaXN0ZW5lciB7XG5cbiAgLyoqXG4gICAqIEB0eXBlZGVmIENvbmZpZyBNZXRhQXBpIG9wdGlvbnMgZm9yIGNvbm5lY3Rpb25zXG4gICAqIEBwcm9wZXJ0eSB7T3B0aW9uc30gW2Nvbm5lY3Rpb25zXSBNZXRhQXBpIGNvbm5lY3Rpb25zIG9wdGlvbnMuIE9ubHkgZm9yIHRlc3RzLiBXaWxsIGJlIGlnbm9yZWQgd2hlbiBzZXQgaW4gU0RLXG4gICAqL1xuXG4gIC8qKlxuICAgKiBAdHlwZWRlZiBPcHRpb25zIE1ldGFBcGlDb25uZWN0aW9uIG9wdGlvbnNcbiAgICogQHByb3BlcnR5IHtudW1iZXJ9IFtyZWZyZXNoUmVwbGljYXNNYXhEZWxheUluTXMgPSA2ICogNjAgKiA2MCAqIDEwMDBdIG1heCBkZWxheSBiZWZvcmUgcmVmcmVzaGluZyByZXBsaWNhcyBkZWxheVxuICAgKi9cblxuICAvKipcbiAgICogQ29uc3RydWN0cyBNZXRhQXBpIE1ldGFUcmFkZXIgQXBpIGNvbm5lY3Rpb25cbiAgICogQHBhcmFtIHtNZXRhQXBpT3B0cyAmIENvbmZpZ30gb3B0aW9ucyBNZXRhQXBpIG9wdGlvbnNcbiAgICogQHBhcmFtIHtNZXRhQXBpV2Vic29ja2V0Q2xpZW50fSB3ZWJzb2NrZXRDbGllbnQgTWV0YUFwaSB3ZWJzb2NrZXQgY2xpZW50XG4gICAqIEBwYXJhbSB7TWV0YXRyYWRlckFjY291bnR9IGFjY291bnQgTWV0YVRyYWRlciBhY2NvdW50IGlkIHRvIGNvbm5lY3QgdG9cbiAgICogQHBhcmFtIHtTdHJpbmd9IFthcHBsaWNhdGlvbl0gYXBwbGljYXRpb24gdG8gdXNlXG4gICAqL1xuICBjb25zdHJ1Y3RvcihvcHRpb25zLCB3ZWJzb2NrZXRDbGllbnQsIGFjY291bnQsIGFwcGxpY2F0aW9uKSB7XG4gICAgc3VwZXIoKTtcbiAgICB0aGlzLl9vcHRpb25zID0gb3B0aW9ucztcbiAgICB0aGlzLl93ZWJzb2NrZXRDbGllbnQgPSB3ZWJzb2NrZXRDbGllbnQ7XG4gICAgdGhpcy5fbGF0ZW5jeVNlcnZpY2UgPSB3ZWJzb2NrZXRDbGllbnQubGF0ZW5jeVNlcnZpY2U7XG4gICAgdGhpcy5fYWNjb3VudCA9IGFjY291bnQ7XG4gICAgdGhpcy5fbG9nZ2VyID0gTG9nZ2VyTWFuYWdlci5nZXRMb2dnZXIoJ01ldGFBcGlDb25uZWN0aW9uJyk7XG4gICAgdGhpcy5fYXBwbGljYXRpb24gPSBhcHBsaWNhdGlvbjtcbiAgICB0aGlzLl9yZWZyZXNoUmVwbGljYXMgPSB0aGlzLl9yZWZyZXNoUmVwbGljYXMuYmluZCh0aGlzKTtcbiAgICB0aGlzLl9yZWZyZXNoVGFza3MgPSB7fTtcbiAgfVxuXG4gIC8qKlxuICAgKiBPcGVucyB0aGUgY29ubmVjdGlvbi4gQ2FuIG9ubHkgYmUgY2FsbGVkIHRoZSBmaXJzdCB0aW1lLCBuZXh0IGNhbGxzIHdpbGwgYmUgaWdub3JlZC5cbiAgICogQHBhcmFtIHtzdHJpbmd9IGluc3RhbmNlSWQgY29ubmVjdGlvbiBpbnN0YW5jZSBpZFxuICAgKiBAcmV0dXJuIHtQcm9taXNlfSBwcm9taXNlIHJlc29sdmluZyB3aGVuIHRoZSBjb25uZWN0aW9uIGlzIG9wZW5lZFxuICAgKi9cbiAgYXN5bmMgY29ubmVjdChpbnN0YW5jZUlkKSB7fVxuXG4gIC8qKlxuICAgKiBDbG9zZXMgdGhlIGNvbm5lY3Rpb24uIFRoZSBpbnN0YW5jZSBvZiB0aGUgY2xhc3Mgc2hvdWxkIG5vIGxvbmdlciBiZSB1c2VkIGFmdGVyIHRoaXMgbWV0aG9kIGlzIGludm9rZWQuXG4gICAqIEBwYXJhbSB7c3RyaW5nfSBpbnN0YW5jZUlkIGNvbm5lY3Rpb24gaW5zdGFuY2UgaWRcbiAgICovXG4gIGFzeW5jIGNsb3NlKGluc3RhbmNlSWQpIHt9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgTWV0YUFwaSBhY2NvdW50XG4gICAqIEByZXR1cm4ge01ldGF0cmFkZXJBY2NvdW50fSBNZXRhQXBpIGFjY291bnRcbiAgICovXG4gIGdldCBhY2NvdW50KCkge1xuICAgIHJldHVybiB0aGlzLl9hY2NvdW50O1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgY29ubmVjdGlvbiBhcHBsaWNhdGlvblxuICAgKiBAcmV0dXJuIHtTdHJpbmd9IGNvbm5lY3Rpb24gYXBwbGljYXRpb25cbiAgICovXG4gIGdldCBhcHBsaWNhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy5fYXBwbGljYXRpb247XG4gIH1cblxuICAvKipcbiAgICogU2NoZWR1bGVzIHRoZSByZWZyZXNoIHRhc2tcbiAgICogQHBhcmFtIHtzdHJpbmd9IHJlZ2lvbiByZXBsaWNhIHJlZ2lvblxuICAgKi9cbiAgc2NoZWR1bGVSZWZyZXNoKHJlZ2lvbikge1xuICAgIGlmICghdGhpcy5fcmVmcmVzaFRhc2tzW3JlZ2lvbl0pIHtcbiAgICAgIGNvbnN0IGRlbGF5ID0gTWF0aC5yYW5kb20oKSAqICh0aGlzLl9vcHRpb25zLmNvbm5lY3Rpb25zPy5yZWZyZXNoUmVwbGljYXNNYXhEZWxheUluTXMgPz8gNiAqIDYwICogNjAgKiAxMDAwKTtcbiAgICAgIHRoaXMuX3JlZnJlc2hUYXNrc1tyZWdpb25dID0gc2V0VGltZW91dCh0aGlzLl9yZWZyZXNoUmVwbGljYXMsIGRlbGF5KTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2FuY2VscyB0aGUgc2NoZWR1bGVkIHJlZnJlc2ggdGFza1xuICAgKiBAcGFyYW0ge3N0cmluZ30gcmVnaW9uIHJlcGxpY2EgcmVnaW9uXG4gICAqL1xuICBjYW5jZWxSZWZyZXNoKHJlZ2lvbikge1xuICAgIGNsZWFyVGltZW91dCh0aGlzLl9yZWZyZXNoVGFza3NbcmVnaW9uXSk7XG4gICAgZGVsZXRlIHRoaXMuX3JlZnJlc2hUYXNrc1tyZWdpb25dO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlZnJlc2hlcyBhY2NvdW50IHJlcGxpY2FzXG4gICAqL1xuICBhc3luYyBfcmVmcmVzaFJlcGxpY2FzKCkge1xuICAgIE9iamVjdC52YWx1ZXModGhpcy5fcmVmcmVzaFRhc2tzKS5mb3JFYWNoKHRhc2sgPT4gY2xlYXJUaW1lb3V0KHRhc2spKTtcbiAgICB0aGlzLl9yZWZyZXNoVGFza3MgPSB7fTtcbiAgICBjb25zdCBvbGRSZXBsaWNhcyA9IHt9O1xuICAgIHRoaXMuX2FjY291bnQucmVwbGljYXMuZm9yRWFjaChyZXBsaWNhID0+IG9sZFJlcGxpY2FzW3JlcGxpY2EucmVnaW9uXSA9IHJlcGxpY2EuaWQpO1xuICAgIGNvbnN0IG5ld1JlcGxpY2FzID0ge307XG4gICAgbGV0IGlzQWNjb3VudFVwZGF0ZWQgPSBmYWxzZTtcbiAgICB0cnkge1xuICAgICAgYXdhaXQgdGhpcy5fYWNjb3VudC5yZWxvYWQoKTtcbiAgICAgIGlzQWNjb3VudFVwZGF0ZWQgPSB0cnVlO1xuICAgICAgdGhpcy5fYWNjb3VudC5yZXBsaWNhcy5mb3JFYWNoKHJlcGxpY2EgPT4gbmV3UmVwbGljYXNbcmVwbGljYS5yZWdpb25dID0gcmVwbGljYS5pZCk7XG4gICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgIGlmIChlcnJvci5uYW1lID09PSAnTm90Rm91bmRFcnJvcicpIHtcbiAgICAgICAgaWYgKHRoaXMuX2Nvbm5lY3Rpb25SZWdpc3RyeSkge1xuICAgICAgICAgIHRoaXMuX2Nvbm5lY3Rpb25SZWdpc3RyeS5jbG9zZUFsbEluc3RhbmNlcyh0aGlzLl9hY2NvdW50LmlkKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICBpZiAoaXNBY2NvdW50VXBkYXRlZCkge1xuICAgICAgY29uc3QgZGVsZXRlZFJlcGxpY2FzID0ge307XG4gICAgICBjb25zdCBhZGRlZFJlcGxpY2FzID0ge307XG4gICAgICBPYmplY3Qua2V5cyhvbGRSZXBsaWNhcykuZm9yRWFjaChrZXkgPT4ge1xuICAgICAgICBpZiAobmV3UmVwbGljYXNba2V5XSAhPT0gb2xkUmVwbGljYXNba2V5XSkge1xuICAgICAgICAgIGRlbGV0ZWRSZXBsaWNhc1trZXldID0gb2xkUmVwbGljYXNba2V5XTtcbiAgICAgICAgfVxuICAgICAgfSk7XG4gICAgICBPYmplY3Qua2V5cyhuZXdSZXBsaWNhcykuZm9yRWFjaChrZXkgPT4ge1xuICAgICAgICBpZiAobmV3UmVwbGljYXNba2V5XSAhPT0gb2xkUmVwbGljYXNba2V5XSkge1xuICAgICAgICAgIGFkZGVkUmVwbGljYXNba2V5XSA9IG5ld1JlcGxpY2FzW2tleV07XG4gICAgICAgIH1cbiAgICAgIH0pO1xuICAgICAgaWYgKE9iamVjdC5rZXlzKGRlbGV0ZWRSZXBsaWNhcykubGVuZ3RoKSB7XG4gICAgICAgIE9iamVjdC52YWx1ZXMoZGVsZXRlZFJlcGxpY2FzKS5mb3JFYWNoKHJlcGxpY2FJZCA9PiBcbiAgICAgICAgICB0aGlzLl93ZWJzb2NrZXRDbGllbnQub25BY2NvdW50RGVsZXRlZChyZXBsaWNhSWQpKTtcbiAgICAgIH1cbiAgICAgIGlmIChPYmplY3Qua2V5cyhkZWxldGVkUmVwbGljYXMpLmxlbmd0aCB8fCBPYmplY3Qua2V5cyhhZGRlZFJlcGxpY2FzKS5sZW5ndGgpIHtcbiAgICAgICAgbmV3UmVwbGljYXNbdGhpcy5fYWNjb3VudC5yZWdpb25dID0gdGhpcy5fYWNjb3VudC5pZDtcbiAgICAgICAgdGhpcy5fd2Vic29ja2V0Q2xpZW50LnVwZGF0ZUFjY291bnRDYWNoZSh0aGlzLl9hY2NvdW50LmlkLCBuZXdSZXBsaWNhcyk7XG4gICAgICAgIE9iamVjdC5lbnRyaWVzKHRoaXMuX2FjY291bnQuYWNjb3VudFJlZ2lvbnMpLmZvckVhY2goKFtyZWdpb24sIGluc3RhbmNlXSkgPT4ge1xuICAgICAgICAgIGlmICghdGhpcy5fb3B0aW9ucy5yZWdpb24gfHwgdGhpcy5fb3B0aW9ucy5yZWdpb24gPT09IHJlZ2lvbikge1xuICAgICAgICAgICAgdGhpcy5fd2Vic29ja2V0Q2xpZW50LmVuc3VyZVN1YnNjcmliZShpbnN0YW5jZSwgMCk7XG4gICAgICAgICAgICB0aGlzLl93ZWJzb2NrZXRDbGllbnQuZW5zdXJlU3Vic2NyaWJlKGluc3RhbmNlLCAxKTtcbiAgICAgICAgICB9XG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIGFzeW5jIF9lbnN1cmVTeW5jaHJvbml6ZWQoaW5zdGFuY2VJbmRleCwga2V5KSB7XG4gICAgbGV0IHN0YXRlID0gdGhpcy5fZ2V0U3RhdGUoaW5zdGFuY2VJbmRleCk7XG4gICAgaWYgKHN0YXRlICYmICF0aGlzLl9jbG9zZWQpIHtcbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IHN5bmNocm9uaXphdGlvblJlc3VsdCA9IGF3YWl0IHRoaXMuc3luY2hyb25pemUoaW5zdGFuY2VJbmRleCk7XG4gICAgICAgIGlmIChzeW5jaHJvbml6YXRpb25SZXN1bHQpIHtcbiAgICAgICAgICBzdGF0ZS5zeW5jaHJvbml6ZWQgPSB0cnVlO1xuICAgICAgICAgIHN0YXRlLnN5bmNocm9uaXphdGlvblJldHJ5SW50ZXJ2YWxJblNlY29uZHMgPSAxO1xuICAgICAgICB9XG4gICAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgICAgY29uc3QgbGV2ZWwgPSB0aGlzLl9sYXRlbmN5U2VydmljZS5nZXRTeW5jaHJvbml6ZWRBY2NvdW50SW5zdGFuY2VzKHRoaXMuX2FjY291bnQuaWQpLmxlbmd0aCA/ICdkZWJ1ZycgOiAnZXJyb3InO1xuICAgICAgICB0aGlzLl9sb2dnZXJbbGV2ZWxdKCdNZXRhQXBpIHdlYnNvY2tldCBjbGllbnQgZm9yIGFjY291bnQgJyArIHRoaXMuX2FjY291bnQuaWQgK1xuICAgICAgICAgICc6JyArIGluc3RhbmNlSW5kZXggKyAnIGZhaWxlZCB0byBzeW5jaHJvbml6ZScsIGVycik7XG4gICAgICAgIGlmIChzdGF0ZS5zaG91bGRTeW5jaHJvbml6ZSA9PT0ga2V5KSB7XG4gICAgICAgICAgc2V0VGltZW91dCh0aGlzLl9lbnN1cmVTeW5jaHJvbml6ZWQuYmluZCh0aGlzLCBpbnN0YW5jZUluZGV4LCBrZXkpLFxuICAgICAgICAgICAgc3RhdGUuc3luY2hyb25pemF0aW9uUmV0cnlJbnRlcnZhbEluU2Vjb25kcyAqIDEwMDApO1xuICAgICAgICAgIHN0YXRlLnN5bmNocm9uaXphdGlvblJldHJ5SW50ZXJ2YWxJblNlY29uZHMgPSBNYXRoLm1pbihzdGF0ZS5zeW5jaHJvbml6YXRpb25SZXRyeUludGVydmFsSW5TZWNvbmRzICogMiwgMzAwKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIF9nZXRTdGF0ZShpbnN0YW5jZUluZGV4KSB7XG4gICAgaWYgKCF0aGlzLl9zdGF0ZUJ5SW5zdGFuY2VJbmRleFsnJyArIGluc3RhbmNlSW5kZXhdKSB7XG4gICAgICB0aGlzLl9zdGF0ZUJ5SW5zdGFuY2VJbmRleFsnJyArIGluc3RhbmNlSW5kZXhdID0ge1xuICAgICAgICBpbnN0YW5jZUluZGV4LFxuICAgICAgICBvcmRlcnNTeW5jaHJvbml6ZWQ6IHt9LFxuICAgICAgICBkZWFsc1N5bmNocm9uaXplZDoge30sXG4gICAgICAgIHNob3VsZFN5bmNocm9uaXplOiB1bmRlZmluZWQsXG4gICAgICAgIHN5bmNocm9uaXphdGlvblJldHJ5SW50ZXJ2YWxJblNlY29uZHM6IDEsXG4gICAgICAgIHN5bmNocm9uaXplZDogZmFsc2UsXG4gICAgICAgIGxhc3REaXNjb25uZWN0ZWRTeW5jaHJvbml6YXRpb25JZDogdW5kZWZpbmVkLFxuICAgICAgICBsYXN0U3luY2hyb25pemF0aW9uSWQ6IHVuZGVmaW5lZCxcbiAgICAgICAgZGlzY29ubmVjdGVkOiBmYWxzZVxuICAgICAgfTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuX3N0YXRlQnlJbnN0YW5jZUluZGV4WycnICsgaW5zdGFuY2VJbmRleF07XG4gIH1cblxuICBfY2hlY2tJc0Nvbm5lY3Rpb25BY3RpdmUoKSB7XG4gICAgaWYgKCF0aGlzLl9vcGVuZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVGhpcyBjb25uZWN0aW9uIGhhcyBub3QgYmVlbiBpbml0aWFsaXplZCB5ZXQsIHBsZWFzZSBpbnZva2UgYXdhaXQgY29ubmVjdGlvbi5jb25uZWN0KCknKTtcbiAgICB9XG4gICAgaWYgKHRoaXMuX2Nsb3NlZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGlzIGNvbm5lY3Rpb24gaGFzIGJlZW4gY2xvc2VkLCBwbGVhc2UgY3JlYXRlIGEgbmV3IGNvbm5lY3Rpb24nKTtcbiAgICB9XG4gIH1cblxufVxuIl0sIm5hbWVzIjpbIlN5bmNocm9uaXphdGlvbkxpc3RlbmVyIiwiTG9nZ2VyTWFuYWdlciIsIk1ldGFBcGlDb25uZWN0aW9uIiwiY29ubmVjdCIsImluc3RhbmNlSWQiLCJjbG9zZSIsImFjY291bnQiLCJfYWNjb3VudCIsImFwcGxpY2F0aW9uIiwiX2FwcGxpY2F0aW9uIiwic2NoZWR1bGVSZWZyZXNoIiwicmVnaW9uIiwiX3JlZnJlc2hUYXNrcyIsImRlbGF5IiwiTWF0aCIsInJhbmRvbSIsIl9vcHRpb25zIiwiY29ubmVjdGlvbnMiLCJyZWZyZXNoUmVwbGljYXNNYXhEZWxheUluTXMiLCJzZXRUaW1lb3V0IiwiX3JlZnJlc2hSZXBsaWNhcyIsImNhbmNlbFJlZnJlc2giLCJjbGVhclRpbWVvdXQiLCJPYmplY3QiLCJ2YWx1ZXMiLCJmb3JFYWNoIiwidGFzayIsIm9sZFJlcGxpY2FzIiwicmVwbGljYXMiLCJyZXBsaWNhIiwiaWQiLCJuZXdSZXBsaWNhcyIsImlzQWNjb3VudFVwZGF0ZWQiLCJyZWxvYWQiLCJlcnJvciIsIm5hbWUiLCJfY29ubmVjdGlvblJlZ2lzdHJ5IiwiY2xvc2VBbGxJbnN0YW5jZXMiLCJkZWxldGVkUmVwbGljYXMiLCJhZGRlZFJlcGxpY2FzIiwia2V5cyIsImtleSIsImxlbmd0aCIsInJlcGxpY2FJZCIsIl93ZWJzb2NrZXRDbGllbnQiLCJvbkFjY291bnREZWxldGVkIiwidXBkYXRlQWNjb3VudENhY2hlIiwiZW50cmllcyIsImFjY291bnRSZWdpb25zIiwiaW5zdGFuY2UiLCJlbnN1cmVTdWJzY3JpYmUiLCJfZW5zdXJlU3luY2hyb25pemVkIiwiaW5zdGFuY2VJbmRleCIsInN0YXRlIiwiX2dldFN0YXRlIiwiX2Nsb3NlZCIsInN5bmNocm9uaXphdGlvblJlc3VsdCIsInN5bmNocm9uaXplIiwic3luY2hyb25pemVkIiwic3luY2hyb25pemF0aW9uUmV0cnlJbnRlcnZhbEluU2Vjb25kcyIsImVyciIsImxldmVsIiwiX2xhdGVuY3lTZXJ2aWNlIiwiZ2V0U3luY2hyb25pemVkQWNjb3VudEluc3RhbmNlcyIsIl9sb2dnZXIiLCJzaG91bGRTeW5jaHJvbml6ZSIsImJpbmQiLCJtaW4iLCJfc3RhdGVCeUluc3RhbmNlSW5kZXgiLCJvcmRlcnNTeW5jaHJvbml6ZWQiLCJkZWFsc1N5bmNocm9uaXplZCIsInVuZGVmaW5lZCIsImxhc3REaXNjb25uZWN0ZWRTeW5jaHJvbml6YXRpb25JZCIsImxhc3RTeW5jaHJvbml6YXRpb25JZCIsImRpc2Nvbm5lY3RlZCIsIl9jaGVja0lzQ29ubmVjdGlvbkFjdGl2ZSIsIl9vcGVuZWQiLCJFcnJvciIsImNvbnN0cnVjdG9yIiwib3B0aW9ucyIsIndlYnNvY2tldENsaWVudCIsImxhdGVuY3lTZXJ2aWNlIiwiZ2V0TG9nZ2VyIl0sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0FBRUEsT0FBT0EsNkJBQTZCLDZDQUE2QztBQUNqRixPQUFPQyxtQkFBbUIsWUFBWTtBQUt2QixJQUFBLEFBQU1DLG9CQUFOLE1BQU1BLDBCQUEwQkY7SUErQjdDOzs7O0dBSUMsR0FDRCxBQUFNRyxRQUFRQyxVQUFVO2VBQXhCLG9CQUFBLGFBQTJCOztJQUUzQjs7O0dBR0MsR0FDRCxBQUFNQyxNQUFNRCxVQUFVO2VBQXRCLG9CQUFBLGFBQXlCOztJQUV6Qjs7O0dBR0MsR0FDRCxJQUFJRSxVQUFVO1FBQ1osT0FBTyxJQUFJLENBQUNDLFFBQVE7SUFDdEI7SUFFQTs7O0dBR0MsR0FDRCxJQUFJQyxjQUFjO1FBQ2hCLE9BQU8sSUFBSSxDQUFDQyxZQUFZO0lBQzFCO0lBRUE7OztHQUdDLEdBQ0RDLGdCQUFnQkMsTUFBTSxFQUFFO1FBQ3RCLElBQUksQ0FBQyxJQUFJLENBQUNDLGFBQWEsQ0FBQ0QsT0FBTyxFQUFFO2dCQUNBO2dCQUFBO1lBQS9CLE1BQU1FLFFBQVFDLEtBQUtDLE1BQU0sS0FBTSxDQUFBLENBQUEsMERBQUEsNkJBQUEsSUFBSSxDQUFDQyxRQUFRLENBQUNDLFdBQVcsY0FBekIsaURBQUEsMkJBQTJCQywyQkFBMkIsY0FBdEQsb0VBQUEseURBQTBELElBQUksS0FBSyxLQUFLLElBQUc7WUFDMUcsSUFBSSxDQUFDTixhQUFhLENBQUNELE9BQU8sR0FBR1EsV0FBVyxJQUFJLENBQUNDLGdCQUFnQixFQUFFUDtRQUNqRTtJQUNGO0lBRUE7OztHQUdDLEdBQ0RRLGNBQWNWLE1BQU0sRUFBRTtRQUNwQlcsYUFBYSxJQUFJLENBQUNWLGFBQWEsQ0FBQ0QsT0FBTztRQUN2QyxPQUFPLElBQUksQ0FBQ0MsYUFBYSxDQUFDRCxPQUFPO0lBQ25DO0lBRUE7O0dBRUMsR0FDRCxBQUFNUzs7ZUFBTixvQkFBQTtZQUNFRyxPQUFPQyxNQUFNLENBQUMsTUFBS1osYUFBYSxFQUFFYSxPQUFPLENBQUNDLENBQUFBLE9BQVFKLGFBQWFJO1lBQy9ELE1BQUtkLGFBQWEsR0FBRyxDQUFDO1lBQ3RCLE1BQU1lLGNBQWMsQ0FBQztZQUNyQixNQUFLcEIsUUFBUSxDQUFDcUIsUUFBUSxDQUFDSCxPQUFPLENBQUNJLENBQUFBLFVBQVdGLFdBQVcsQ0FBQ0UsUUFBUWxCLE1BQU0sQ0FBQyxHQUFHa0IsUUFBUUMsRUFBRTtZQUNsRixNQUFNQyxjQUFjLENBQUM7WUFDckIsSUFBSUMsbUJBQW1CO1lBQ3ZCLElBQUk7Z0JBQ0YsTUFBTSxNQUFLekIsUUFBUSxDQUFDMEIsTUFBTTtnQkFDMUJELG1CQUFtQjtnQkFDbkIsTUFBS3pCLFFBQVEsQ0FBQ3FCLFFBQVEsQ0FBQ0gsT0FBTyxDQUFDSSxDQUFBQSxVQUFXRSxXQUFXLENBQUNGLFFBQVFsQixNQUFNLENBQUMsR0FBR2tCLFFBQVFDLEVBQUU7WUFDcEYsRUFBRSxPQUFPSSxPQUFPO2dCQUNkLElBQUlBLE1BQU1DLElBQUksS0FBSyxpQkFBaUI7b0JBQ2xDLElBQUksTUFBS0MsbUJBQW1CLEVBQUU7d0JBQzVCLE1BQUtBLG1CQUFtQixDQUFDQyxpQkFBaUIsQ0FBQyxNQUFLOUIsUUFBUSxDQUFDdUIsRUFBRTtvQkFDN0Q7Z0JBQ0Y7WUFDRjtZQUNBLElBQUlFLGtCQUFrQjtnQkFDcEIsTUFBTU0sa0JBQWtCLENBQUM7Z0JBQ3pCLE1BQU1DLGdCQUFnQixDQUFDO2dCQUN2QmhCLE9BQU9pQixJQUFJLENBQUNiLGFBQWFGLE9BQU8sQ0FBQ2dCLENBQUFBO29CQUMvQixJQUFJVixXQUFXLENBQUNVLElBQUksS0FBS2QsV0FBVyxDQUFDYyxJQUFJLEVBQUU7d0JBQ3pDSCxlQUFlLENBQUNHLElBQUksR0FBR2QsV0FBVyxDQUFDYyxJQUFJO29CQUN6QztnQkFDRjtnQkFDQWxCLE9BQU9pQixJQUFJLENBQUNULGFBQWFOLE9BQU8sQ0FBQ2dCLENBQUFBO29CQUMvQixJQUFJVixXQUFXLENBQUNVLElBQUksS0FBS2QsV0FBVyxDQUFDYyxJQUFJLEVBQUU7d0JBQ3pDRixhQUFhLENBQUNFLElBQUksR0FBR1YsV0FBVyxDQUFDVSxJQUFJO29CQUN2QztnQkFDRjtnQkFDQSxJQUFJbEIsT0FBT2lCLElBQUksQ0FBQ0YsaUJBQWlCSSxNQUFNLEVBQUU7b0JBQ3ZDbkIsT0FBT0MsTUFBTSxDQUFDYyxpQkFBaUJiLE9BQU8sQ0FBQ2tCLENBQUFBLFlBQ3JDLE1BQUtDLGdCQUFnQixDQUFDQyxnQkFBZ0IsQ0FBQ0Y7Z0JBQzNDO2dCQUNBLElBQUlwQixPQUFPaUIsSUFBSSxDQUFDRixpQkFBaUJJLE1BQU0sSUFBSW5CLE9BQU9pQixJQUFJLENBQUNELGVBQWVHLE1BQU0sRUFBRTtvQkFDNUVYLFdBQVcsQ0FBQyxNQUFLeEIsUUFBUSxDQUFDSSxNQUFNLENBQUMsR0FBRyxNQUFLSixRQUFRLENBQUN1QixFQUFFO29CQUNwRCxNQUFLYyxnQkFBZ0IsQ0FBQ0Usa0JBQWtCLENBQUMsTUFBS3ZDLFFBQVEsQ0FBQ3VCLEVBQUUsRUFBRUM7b0JBQzNEUixPQUFPd0IsT0FBTyxDQUFDLE1BQUt4QyxRQUFRLENBQUN5QyxjQUFjLEVBQUV2QixPQUFPLENBQUMsQ0FBQyxDQUFDZCxRQUFRc0MsU0FBUzt3QkFDdEUsSUFBSSxDQUFDLE1BQUtqQyxRQUFRLENBQUNMLE1BQU0sSUFBSSxNQUFLSyxRQUFRLENBQUNMLE1BQU0sS0FBS0EsUUFBUTs0QkFDNUQsTUFBS2lDLGdCQUFnQixDQUFDTSxlQUFlLENBQUNELFVBQVU7NEJBQ2hELE1BQUtMLGdCQUFnQixDQUFDTSxlQUFlLENBQUNELFVBQVU7d0JBQ2xEO29CQUNGO2dCQUNGO1lBQ0Y7UUFDRjs7SUFFTUUsb0JBQW9CQyxhQUFhLEVBQUVYLEdBQUc7O2VBQTVDLG9CQUFBO1lBQ0UsSUFBSVksUUFBUSxNQUFLQyxTQUFTLENBQUNGO1lBQzNCLElBQUlDLFNBQVMsQ0FBQyxNQUFLRSxPQUFPLEVBQUU7Z0JBQzFCLElBQUk7b0JBQ0YsTUFBTUMsd0JBQXdCLE1BQU0sTUFBS0MsV0FBVyxDQUFDTDtvQkFDckQsSUFBSUksdUJBQXVCO3dCQUN6QkgsTUFBTUssWUFBWSxHQUFHO3dCQUNyQkwsTUFBTU0scUNBQXFDLEdBQUc7b0JBQ2hEO2dCQUNGLEVBQUUsT0FBT0MsS0FBSztvQkFDWixNQUFNQyxRQUFRLE1BQUtDLGVBQWUsQ0FBQ0MsK0JBQStCLENBQUMsTUFBS3hELFFBQVEsQ0FBQ3VCLEVBQUUsRUFBRVksTUFBTSxHQUFHLFVBQVU7b0JBQ3hHLE1BQUtzQixPQUFPLENBQUNILE1BQU0sQ0FBQywwQ0FBMEMsTUFBS3RELFFBQVEsQ0FBQ3VCLEVBQUUsR0FDNUUsTUFBTXNCLGdCQUFnQiwwQkFBMEJRO29CQUNsRCxJQUFJUCxNQUFNWSxpQkFBaUIsS0FBS3hCLEtBQUs7d0JBQ25DdEIsV0FBVyxNQUFLZ0MsbUJBQW1CLENBQUNlLElBQUksUUFBT2QsZUFBZVgsTUFDNURZLE1BQU1NLHFDQUFxQyxHQUFHO3dCQUNoRE4sTUFBTU0scUNBQXFDLEdBQUc3QyxLQUFLcUQsR0FBRyxDQUFDZCxNQUFNTSxxQ0FBcUMsR0FBRyxHQUFHO29CQUMxRztnQkFDRjtZQUNGO1FBQ0Y7O0lBRUFMLFVBQVVGLGFBQWEsRUFBRTtRQUN2QixJQUFJLENBQUMsSUFBSSxDQUFDZ0IscUJBQXFCLENBQUMsS0FBS2hCLGNBQWMsRUFBRTtZQUNuRCxJQUFJLENBQUNnQixxQkFBcUIsQ0FBQyxLQUFLaEIsY0FBYyxHQUFHO2dCQUMvQ0E7Z0JBQ0FpQixvQkFBb0IsQ0FBQztnQkFDckJDLG1CQUFtQixDQUFDO2dCQUNwQkwsbUJBQW1CTTtnQkFDbkJaLHVDQUF1QztnQkFDdkNELGNBQWM7Z0JBQ2RjLG1DQUFtQ0Q7Z0JBQ25DRSx1QkFBdUJGO2dCQUN2QkcsY0FBYztZQUNoQjtRQUNGO1FBQ0EsT0FBTyxJQUFJLENBQUNOLHFCQUFxQixDQUFDLEtBQUtoQixjQUFjO0lBQ3ZEO0lBRUF1QiwyQkFBMkI7UUFDekIsSUFBSSxDQUFDLElBQUksQ0FBQ0MsT0FBTyxFQUFFO1lBQ2pCLE1BQU0sSUFBSUMsTUFBTTtRQUNsQjtRQUNBLElBQUksSUFBSSxDQUFDdEIsT0FBTyxFQUFFO1lBQ2hCLE1BQU0sSUFBSXNCLE1BQU07UUFDbEI7SUFDRjtJQS9LQTs7O0dBR0MsR0FFRDs7O0dBR0MsR0FFRDs7Ozs7O0dBTUMsR0FDREMsWUFBWUMsT0FBTyxFQUFFQyxlQUFlLEVBQUUxRSxPQUFPLEVBQUVFLFdBQVcsQ0FBRTtRQUMxRCxLQUFLO1FBQ0wsSUFBSSxDQUFDUSxRQUFRLEdBQUcrRDtRQUNoQixJQUFJLENBQUNuQyxnQkFBZ0IsR0FBR29DO1FBQ3hCLElBQUksQ0FBQ2xCLGVBQWUsR0FBR2tCLGdCQUFnQkMsY0FBYztRQUNyRCxJQUFJLENBQUMxRSxRQUFRLEdBQUdEO1FBQ2hCLElBQUksQ0FBQzBELE9BQU8sR0FBRy9ELGNBQWNpRixTQUFTLENBQUM7UUFDdkMsSUFBSSxDQUFDekUsWUFBWSxHQUFHRDtRQUNwQixJQUFJLENBQUNZLGdCQUFnQixHQUFHLElBQUksQ0FBQ0EsZ0JBQWdCLENBQUM4QyxJQUFJLENBQUMsSUFBSTtRQUN2RCxJQUFJLENBQUN0RCxhQUFhLEdBQUcsQ0FBQztJQUN4QjtBQXNKRjtBQXRMQTs7Q0FFQyxHQUNELFNBQXFCViwrQkFtTHBCIn0=