function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
import StreamingMetaApiConnection from './streamingMetaApiConnection';
import RpcMetaApiConnection from './rpcMetaApiConnection';
import StreamingMetaApiConnectionInstance from './streamingMetaApiConnectionInstance';
import RpcMetaApiConnectionInstance from './rpcMetaApiConnectionInstance';
let ConnectionRegistry = class ConnectionRegistry {
    /**
   * Creates and returns a new account streaming connection if doesnt exist, otherwise returns old
   * @param {MetatraderAccount} account MetaTrader account id to connect to
   * @param {HistoryStorage} historyStorage terminal history storage
   * @param {Date} [historyStartTime] history start time
   * @return {StreamingMetaApiConnection} streaming metaapi connection
   */ connectStreaming(account, historyStorage, historyStartTime) {
        if (!this._streamingConnections[account.id]) {
            this._streamingConnections[account.id] = new StreamingMetaApiConnection(this._options, this._metaApiWebsocketClient, this._terminalHashManager, account, historyStorage, this, historyStartTime, this._refreshSubscriptionsOpts);
        }
        const instance = new StreamingMetaApiConnectionInstance(this._metaApiWebsocketClient, this._streamingConnections[account.id]);
        this._streamingConnectionInstances[account.id] = this._streamingConnectionInstances[account.id] || [];
        this._streamingConnectionInstances[account.id].push(instance);
        return instance;
    }
    /**
   * Removes a streaming connection from registry
   * @param {MetatraderAccount} account MetaTrader account to remove from registry
   */ removeStreaming(account) {
        var _this = this;
        return _async_to_generator(function*() {
            if (_this._streamingConnections[account.id]) {
                delete _this._streamingConnections[account.id];
                delete _this._streamingConnectionInstances[account.id];
            }
            if (!_this._rpcConnections[account.id]) {
                yield _this._closeLastConnection(account);
            }
        })();
    }
    /**
   * Creates and returns a new account rpc connection if doesnt exist, otherwise returns old
   * @param {MetatraderAccount} account MetaTrader account id to connect to
   * @returns {RpcMetaApiConnection} rpc metaapi connection
   */ connectRpc(account) {
        if (!this._rpcConnections[account.id]) {
            this._rpcConnections[account.id] = new RpcMetaApiConnection(this._options, this._metaApiWebsocketClient, account, this);
        }
        const instance = new RpcMetaApiConnectionInstance(this._metaApiWebsocketClient, this._rpcConnections[account.id]);
        this._rpcConnectionInstances[account.id] = this._rpcConnectionInstances[account.id] || [];
        this._rpcConnectionInstances[account.id].push(instance);
        return instance;
    }
    /**
   * Removes an RPC connection from registry
   * @param {MetatraderAccount} account MetaTrader account to remove from registry
   */ removeRpc(account) {
        var _this = this;
        return _async_to_generator(function*() {
            if (_this._rpcConnections[account.id]) {
                delete _this._rpcConnections[account.id];
                delete _this._rpcConnectionInstances[account.id];
            }
            if (!_this._streamingConnections[account.id]) {
                yield _this._closeLastConnection(account);
            }
        })();
    }
    /**
   * Removes an account from registry
   * @param {string} accountId MetaTrader account id to remove
   */ remove(accountId) {
        delete this._rpcConnections[accountId];
        delete this._rpcConnectionInstances[accountId];
        delete this._streamingConnections[accountId];
        delete this._streamingConnectionInstances[accountId];
    }
    /**
   * Returns application type
   * @return {String} application type
   */ get application() {
        return this._application;
    }
    _closeLastConnection(account) {
        var _this = this;
        return _async_to_generator(function*() {
            const accountRegions = account.accountRegions;
            yield Promise.all(Object.values(accountRegions).map((replicaId)=>_this._metaApiWebsocketClient.unsubscribe(replicaId)));
        })();
    }
    /**
   * Returns the dictionary of streaming connections
   * @returns {{[id: string]: StreamingMetaApiConnection}}
   */ get streamingConnections() {
        return this._streamingConnections;
    }
    /**
   * Returns the dictionary of rpc connections
   * @returns {{[id: string]: RpcMetaApiConnection}}
   */ get rpcConnections() {
        return this._rpcConnections;
    }
    /**
   * Closes all connection instances for an account
   * @param {string} accountId 
   */ closeAllInstances(accountId) {
        if (this._rpcConnectionInstances[accountId]) {
            this._rpcConnectionInstances[accountId].forEach((instance)=>instance.close());
        }
        if (this._streamingConnectionInstances[accountId]) {
            this._streamingConnectionInstances[accountId].forEach((instance)=>instance.close());
        }
    }
    /**
   * Constructs a MetaTrader connection registry instance
   * @param {MetaApiOpts} options metaapi options
   * @param {MetaApiWebsocketClient} metaApiWebsocketClient MetaApi websocket client
   * @param {ClientApiClient} clientApiClient client API client
   * @param {String} application application id
   * @param {String} refreshSubscriptionsOpts subscriptions refresh options
   */ constructor(options, metaApiWebsocketClient, terminalHashManager, application = 'MetaApi', refreshSubscriptionsOpts){
        refreshSubscriptionsOpts = refreshSubscriptionsOpts || {};
        this._metaApiWebsocketClient = metaApiWebsocketClient;
        this._terminalHashManager = terminalHashManager;
        this._application = application;
        this._refreshSubscriptionsOpts = refreshSubscriptionsOpts;
        this._rpcConnections = {};
        this._rpcConnectionInstances = {};
        this._streamingConnections = {};
        this._streamingConnectionInstances = {};
        this._connectionLocks = {};
        this._options = options;
    }
};
/**
 * Manages account connections
 */ export { ConnectionRegistry as default };

//# sourceMappingURL=data:application/json;base64,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