"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    HttpClientMock: function() {
        return HttpClientMock;
    },
    /**
 * HTTP client library based on request-promise
 */ default: function() {
        return HttpClient;
    }
});
const _axios = /*#__PURE__*/ _interop_require_default(require("axios"));
const _errorHandler = require("./errorHandler");
const _timeoutError = /*#__PURE__*/ _interop_require_default(require("./timeoutError"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let HttpClient = class HttpClient {
    /**
   * @typedef {Object} RetryOptions retry options
   * @property {Number} [retries] the number of attempts to retry failed request, default 5
   * @property {Number} [minDelayInSeconds] minimum delay in seconds before retrying, default 1
   * @property {Number} [maxDelayInSeconds] maximum delay in seconds before retrying, default 30
   */ /**
   * Constructs HttpClient class instance
   * @param {Number} timeout request timeout in seconds
   * @param {RetryOptions} [retryOpts] retry options
   */ constructor(timeout = 60, retryOpts = {}){
        this._timeout = timeout * 1000;
        this._retries = retryOpts.retries || 5;
        this._minRetryDelay = (retryOpts.minDelayInSeconds || 1) * 1000;
        this._maxRetryDelay = (retryOpts.maxDelayInSeconds || 30) * 1000;
    }
    /**
   * Performs a request. Response errors are returned as ApiError or subclasses.
   * @param {Object} options request options
   * @returns {Object|String|any} request result
   */ async request(options, endTime = Date.now() + this._maxRetryDelay * this._retries) {
        options.timeout = this._timeout;
        let retryAfterSeconds = 0;
        options.callback = (e, res)=>{
            if (res && res.status === 202) {
                retryAfterSeconds = res.headers["retry-after"];
            }
        };
        let body;
        try {
            const response = await this._makeRequest(options);
            options.callback(null, response);
            body = response && response.data || undefined;
        } catch (err) {
            throw this._convertError(err);
        }
        if (retryAfterSeconds) {
            await this._handleRetry(endTime, retryAfterSeconds * 1000);
            body = await this.request(options, endTime);
        }
        return body;
    }
    /**
   * Performs a request with failover. Response errors are returned as ApiError or subclasses.
   * @param {Object} options request options
   * @returns {Object|String|any} request result
   */ async requestWithFailover(options, retryCounter = 0, endTime = Date.now() + this._maxRetryDelay * this._retries) {
        options.timeout = this._timeout;
        let retryAfterSeconds = 0;
        options.callback = (e, res)=>{
            if (res && res.status === 202) {
                retryAfterSeconds = res.headers["retry-after"];
            }
        };
        let body;
        try {
            const response = await this._makeRequest(options);
            options.callback(null, response);
            body = response && response.data || undefined;
        } catch (err) {
            retryCounter = await this._handleError(err, retryCounter, endTime);
            return this.requestWithFailover(options, retryCounter, endTime);
        }
        if (retryAfterSeconds) {
            await this._handleRetry(endTime, retryAfterSeconds * 1000);
            body = await this.requestWithFailover(options, retryCounter, endTime);
        }
        return body;
    }
    _makeRequest(options) {
        return (0, _axios.default)({
            transitional: {
                clarifyTimeoutError: true
            },
            ...options
        });
    }
    async _wait(pause) {
        await new Promise((res)=>setTimeout(res, pause));
    }
    async _handleRetry(endTime, retryAfter) {
        if (endTime > Date.now() + retryAfter) {
            await this._wait(retryAfter);
        } else {
            throw new _timeoutError.default("Timed out waiting for the end of the process of calculating metrics");
        }
    }
    async _handleError(err, retryCounter, endTime) {
        const error = this._convertError(err);
        if ([
            "InternalError",
            "ApiError"
        ].includes(error.name) && retryCounter < this._retries) {
            const pause = Math.min(Math.pow(2, retryCounter) * this._minRetryDelay, this._maxRetryDelay);
            await this._wait(pause);
            return retryCounter + 1;
        } else if (error.name === "TooManyRequestsError") {
            const retryTime = Date.parse(error.metadata.recommendedRetryTime);
            if (retryTime < endTime) {
                await this._wait(retryTime - Date.now());
                return retryCounter;
            }
        }
        throw error;
    }
    // eslint-disable-next-line complexity
    _convertError(err) {
        const errorResponse = err.response || {};
        const errorData = errorResponse.data || {};
        const status = errorResponse.status || err.status;
        const url = err?.config?.url;
        const errMsg = errorData.message || err.message;
        const errMsgDefault = errorData.message || err.code || err.message;
        switch(status){
            case 400:
                return new _errorHandler.ValidationError(errMsg, errorData.details || err.details, url);
            case 401:
                return new _errorHandler.UnauthorizedError(errMsg, url);
            case 403:
                return new _errorHandler.ForbiddenError(errMsg, url);
            case 404:
                return new _errorHandler.NotFoundError(errMsg, url);
            case 429:
                return new _errorHandler.TooManyRequestsError(errMsg, errorData.metadata || err.metadata, url);
            case 500:
                return new _errorHandler.InternalError(errMsg, url);
            default:
                return new _errorHandler.ApiError(_errorHandler.ApiError, errMsgDefault, status, url);
        }
    }
};
let HttpClientMock = class HttpClientMock extends HttpClient {
    /**
   * Constructs HTTP client mock
   * @param {Function(options:Object):Promise} requestFn mocked request function
   * @param {Number} timeout request timeout in seconds
   * @param {RetryOptions} retryOpts retry options
   */ constructor(requestFn, timeout, retryOpts){
        super(timeout, retryOpts);
        this._requestFn = requestFn;
    }
    _makeRequest() {
        return this._requestFn.apply(this, arguments);
    }
};

//# sourceMappingURL=data:application/json;base64,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