import MetaApiWebsocketClient, {
  MetatraderAccountInformation, MetatraderOrder, MetatraderPosition, MetatraderSymbolPrice,
  MetatraderSymbolSpecification
} from "../clients/metaApi/metaApiWebsocket.client";
import SynchronizationListener from "../clients/metaApi/synchronizationListener";
import MetatraderAccount from './metatraderAccount';
import TerminalHashManager from './terminalHashManager';

/**
 * Responsible for storing a local copy of remote terminal state
 */
export default class TerminalState extends SynchronizationListener {
  
  /**
   * Constructs the instance of terminal state class
   * @param {MetatraderAccount} account account
   * @param {TerminalHashManager} terminalHashManager terminal hash manager
   * @param {MetaApiWebsocketClient} websocketClient websocket client
   */
  constructor(
    account: MetatraderAccount, terminalHashManager: TerminalHashManager, websocketClient: MetaApiWebsocketClient
  );
  
  /**
   * Returns true if MetaApi have connected to MetaTrader terminal
   * @return {boolean} true if MetaApi have connected to MetaTrader terminal
   */
  get connected(): boolean;
  
  /**
   * Returns true if MetaApi have connected to MetaTrader terminal and MetaTrader terminal is connected to broker
   * @return {boolean} true if MetaApi have connected to MetaTrader terminal and MetaTrader terminal is connected to
   * broker
   */
  get connectedToBroker(): boolean;
  
  /**
   * Returns a local copy of account information
   * @returns {MetatraderAccountInformation} local copy of account information
   */
  get accountInformation(): MetatraderAccountInformation;
  
  /**
   * Returns a local copy of MetaTrader positions opened
   * @returns {Array<MetatraderPosition>} a local copy of MetaTrader positions opened
   */
  get positions(): Array<MetatraderPosition>;
  
  /**
   * Returns a local copy of MetaTrader orders opened
   * @returns {Array<MetatraderOrder>} a local copy of MetaTrader orders opened
   */
  get orders(): Array<MetatraderOrder>;
  
  /**
   * Returns a local copy of symbol specifications available in MetaTrader trading terminal
   * @returns {Array<MetatraderSymbolSpecification>} a local copy of symbol specifications available in MetaTrader
   * trading terminal
   */
  get specifications(): Array<MetatraderSymbolSpecification>;
  
  /**
   * Returns hashes of terminal state data for incremental synchronization
   * @returns {Promise<Object>} promise resolving with hashes of terminal state data
   */
  getHashes(): Promise<Object>;
  
  /**
   * Returns MetaTrader symbol specification by symbol
   * @param {string} symbol symbol (e.g. currency pair or an index)
   * @return {MetatraderSymbolSpecification} MetatraderSymbolSpecification found or undefined if specification for a
   * symbol is not found
   */
  specification(symbol: string): MetatraderSymbolSpecification;
  
  /**
   * Returns MetaTrader symbol price by symbol
   * @param {string} symbol symbol (e.g. currency pair or an index)
   * @return {MetatraderSymbolPrice} MetatraderSymbolPrice found or undefined if price for a symbol is not found
   */
  price(symbol: string): MetatraderSymbolPrice;

  /**
   * Returns time of the last received quote
   * @return {QuoteTime} time of the last received quote
   */
  get lastQuoteTime(): QuoteTime;

  /**
   * Waits for price to be received
   * @param {string} symbol symbol (e.g. currency pair or an index)
   * @param {number} [timeoutInSeconds] timeout in seconds, default is 30
   * @return {Promise<MetatraderSymbolPrice>} promise resolving with price or undefined if price has not been received
   */
  waitForPrice(symbol: string, timeoutInSeconds?: number): Promise<MetatraderSymbolPrice>;
  
  /**
   * Invoked when connection to MetaTrader terminal established
   * @param {string} instanceIndex index of an account instance connected
   */
  onConnected(instanceIndex: string): Promise<any>;
  
  /**
   * Invoked when connection to MetaTrader terminal terminated
   * @param {string} instanceIndex index of an account instance connected
   */
  onDisconnected(instanceIndex: string): Promise<any>;
  
  /**
   * Invoked when broker connection status have changed
   * @param {string} instanceIndex index of an account instance connected
   * @param {boolean} connected is MetaTrader terminal is connected to broker
   */
  onBrokerConnectionStatusChanged(instanceIndex: string, connected: boolean): Promise<any>;

  /**
   * Invoked when MetaTrader terminal state synchronization is started
   * @param {string} instanceIndex index of an account instance connected
   * @param {string} specificationsHash specifications hash
   * @param {string} positionsHash positions hash
   * @param {string} ordersHash orders hash
   * @param {string} synchronizationId synchronization id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  onSynchronizationStarted(instanceIndex: string, specificationsHash: string, positionsHash: string, ordersHash: string, synchronizationId: string): Promise<any>;
  
  /**
   * Invoked when MetaTrader account information is updated
   * @param {string} instanceIndex index of an account instance connected
   * @param {MetatraderAccountInformation} accountInformation updated MetaTrader account information
   */
  onAccountInformationUpdated(instanceIndex: string, accountInformation: MetatraderAccountInformation): Promise<any>;
  
  /**
   * Invoked when the positions are replaced as a result of initial terminal state synchronization
   * @param {string} instanceIndex index of an account instance connected
   * @param {Array<MetatraderPosition>} positions updated array of positions
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  onPositionsReplaced(instanceIndex: string, positions: Array<MetatraderPosition>): Promise<any>;
  
  /**
   * Invoked when position synchronization fnished to indicate progress of an initial terminal state synchronization
   * @param {string} instanceIndex index of an account instance connected
   * @param {string} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  onPositionsSynchronized(instanceIndex: string, synchronizationId: string): Promise<any>;

  /**
   * Invoked when MetaTrader positions are updated
   * @param {string} instanceIndex index of an account instance connected
   * @param {MetatraderPosition[]} positions updated MetaTrader positions
   * @param {string[]} removedPositionIds removed position ids
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  onPositionsUpdated(instanceIndex: string, positions: MetatraderPosition[], removedPositionIds: string[]): Promise<any>;
  
  /**
   * Invoked when the orders are replaced as a result of initial terminal state synchronization
   * @param {string} instanceIndex index of an account instance connected
   * @param {Array<MetatraderOrder>} orders updated array of pending orders
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  onPendingOrdersReplaced(instanceIndex: string, orders: Array<MetatraderOrder>): Promise<any>;
  
  /**
   * Invoked when pending order synchronization fnished to indicate progress of an initial terminal state
   * synchronization
   * @param {string} instanceIndex index of an account instance connected
   * @param {string} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  onPendingOrdersSynchronized(instanceIndex: string, synchronizationId: string): Promise<any>;

  /**
   * Invoked when MetaTrader pending orders are updated or completed
   * @param {string} instanceIndex index of an account instance connected
   * @param {MetatraderOrder[]} orders updated MetaTrader pending orders
   * @param {string[]} completedOrderIds completed MetaTrader pending order ids
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  onPendingOrdersUpdated(instanceIndex: string, orders: MetatraderOrder[], completedOrderIds: string[]): Promise<any>;
  
  /**
   * Invoked when a symbol specification was updated
   * @param {string} instanceIndex index of account instance connected
   * @param {Array<MetatraderSymbolSpecification>} specifications updated specifications
   * @param {Array<string>} removedSymbols removed symbols
   */
  onSymbolSpecificationsUpdated(instanceIndex: string, specifications: Array<MetatraderSymbolSpecification>, removedSymbols: Array<string>): Promise<any>;
  
  /**
   * Invoked when prices for several symbols were updated
   * @param {string} instanceIndex index of an account instance connected
   * @param {Array<MetatraderSymbolPrice>} prices updated MetaTrader symbol prices
   * @param {number} equity account liquidation value
   * @param {number} margin margin used
   * @param {number} freeMargin free margin
   * @param {number} marginLevel margin level calculated as % of equity/margin
   */
  onSymbolPricesUpdated(instanceIndex: string, prices: Array<MetatraderSymbolPrice>, equity: number, margin: number, freeMargin: number, marginLevel: number): Promise<any>;
  
  /**
   * Invoked when a stream for an instance index is closed
   * @param {string} instanceIndex index of an account instance connected
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  onStreamClosed(instanceIndex: string): Promise<any>;

  /**
   * Forces refresh of most recent quote updates for symbols subscribed to by the terminal, and waits for them all to
   * be processed by this terminal state. This method does not waits for all other listeners to receive and process the
   * quote updates
   * @param {RefreshTerminalStateOptions} [options] additional options
   * @returns {Promise} promise resolving when the terminal state received and processed the latest quotes
   */
  refreshTerminalState(options: RefreshTerminalStateOptions): Promise<void>;

  /**
   * Removes connection related data from terminal hash manager
   */
  close(): void;
}

/**
 * Quote time
 */
export declare type QuoteTime = {

  /**
   * Quote time
   */
  time: Date,

  /**
   * Quote broker time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   */
  brokerTime: string

}

/**
 * Options for refreshing terminal state
 */
export declare type RefreshTerminalStateOptions = {
  /**
   * Timeout in seconds. Defaults to `10`
   */
  timeoutInSeconds?: number
}
