'use strict';
import TimeoutError from '../clients/timeoutError';
/**
 * Creates a promise that can be used as a handle. It will not raise errors when rejected until it is explicitly
 * awaited or catch is set. The promise additionally has following methods:
 * - `resolve(value)` resolves the promise with specified value
 * - `reject(error)` rejects the promise with specified error
 * - `timeout(milliseconds, errorMessage)` adds a timeout to reject the promise with `TimeoutError`
 * @return {Promise} Modified handle promise
 */ export function createHandlePromise() {
    let resolve, reject;
    let promise = new Promise((res, rej)=>{
        resolve = res;
        reject = rej;
    });
    promise.resolve = (result)=>{
        if (promise.completed) {
            return;
        }
        promise.resolved = true;
        promise.completed = true;
        resolve(result);
    };
    promise.reject = (err)=>{
        if (promise.completed) {
            return;
        }
        promise.rejected = true;
        promise.completed = true;
        reject(err);
    };
    promise.timeout = (milliseconds, errorMessage)=>{
        if (promise.completed) {
            return;
        }
        let timeout = setTimeout(()=>promise.reject(new TimeoutError(errorMessage)), milliseconds);
        promise.finally(()=>clearTimeout(timeout)).catch(()=>{});
        return promise;
    };
    promise.catch(()=>{});
    return promise;
}
/**
 * Waits specified delay
 * @param {Number} ms Milliseconds to wait
 * @return {Promise} Promise resolving when the delay has ended
 */ export function delay(ms) {
    return new Promise((res)=>setTimeout(res, ms));
}
/**
 * Assembles log4js config from logging level map
 * @param {Object} [config] log4js config
 * @param {String} [config.defaultLevel = 'INFO'] Default logging level
 * @param {Object} [config.levels] Logging levels
 * @return {Object} Log4js config
 */ export function assembleLog4jsConfig(config = {}) {
    let appenders = {
        console: {
            type: 'console'
        }
    };
    let categories = {
        default: {
            appenders: Object.keys(appenders),
            level: config.defaultLevel || 'INFO'
        }
    };
    Object.keys(config.levels || {}).forEach((category)=>{
        categories[category] = {
            appenders: Object.keys(appenders),
            level: config.levels[category]
        };
    });
    return {
        appenders,
        categories
    };
}

//# sourceMappingURL=data:application/json;base64,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