'use strict';
import { ValidationError } from './errorHandler';
let OptionsValidator = class OptionsValidator {
    /**
   * Validates a number parameter
   * @param {Number} value value to validate
   * @param {Number} defaultValue default value for an option
   * @param {String} name option name
   * @returns {Number} validated value
   * @throws {ValidationError} if value is invalid
   */ validateNumber(value, defaultValue, name) {
        if (value === undefined) {
            return defaultValue;
        }
        if (typeof value !== 'number') {
            throw new ValidationError(`Parameter ${name} must be a number`);
        }
        if (value < 0) {
            throw new ValidationError(`Parameter ${name} cannot be lower than 0`);
        }
        return value;
    }
    /**
   * Validates a number parameter to be above zero
   * @param {Number} value value to validate
   * @param {Number} defaultValue default value for an option
   * @param {String} name option name
   * @returns {Number} validated value
   * @throws {ValidationError} if value is invalid
   */ validateNonZero(value, defaultValue, name) {
        value = this.validateNumber(value, defaultValue, name);
        if (value === 0) {
            throw new ValidationError(`Parameter ${name} must be bigger than 0`);
        }
        return value;
    }
    /**
   * Validates a parameter to be boolean
   * @param {Boolean} value value to validate
   * @param {Boolean} defaultValue default value for an option
   * @param {String} name option name
   * @returns {Boolean} validated value
   * @throws {ValidationError} if value is invalid
   */ validateBoolean(value, defaultValue, name) {
        if (value === undefined) {
            return defaultValue;
        }
        if (typeof value !== 'boolean') {
            throw new ValidationError(`Parameter ${name} must be a boolean`);
        }
        return value;
    }
};
/**
 * Class for validating API options.
 */ export { OptionsValidator as default };

//# sourceMappingURL=data:application/json;base64,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