'use strict';
function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
function _define_property(obj, key, value) {
    if (key in obj) {
        Object.defineProperty(obj, key, {
            value: value,
            enumerable: true,
            configurable: true,
            writable: true
        });
    } else {
        obj[key] = value;
    }
    return obj;
}
function _object_spread(target) {
    for(var i = 1; i < arguments.length; i++){
        var source = arguments[i] != null ? arguments[i] : {};
        var ownKeys = Object.keys(source);
        if (typeof Object.getOwnPropertySymbols === "function") {
            ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function(sym) {
                return Object.getOwnPropertyDescriptor(source, sym).enumerable;
            }));
        }
        ownKeys.forEach(function(key) {
            _define_property(target, key, source[key]);
        });
    }
    return target;
}
import randomstring from 'randomstring';
import socketIO from 'socket.io-client';
import TimeoutError from '../timeoutError';
import { ValidationError, NotFoundError, InternalError, UnauthorizedError, TooManyRequestsError, ForbiddenError } from '../errorHandler';
import OptionsValidator from '../optionsValidator';
import NotSynchronizedError from './notSynchronizedError';
import NotConnectedError from './notConnectedError';
import TradeError from './tradeError';
import PacketOrderer from './packetOrderer';
import SynchronizationThrottler from './synchronizationThrottler';
import SubscriptionManager from './subscriptionManager';
import LoggerManager from '../../logger';
import any from 'promise.any';
import LatencyService from './latencyService';
import { MetatraderSymbolSpecification } from './metaApiWebsocket.client';
let PacketLogger;
if (typeof window === 'undefined') {
    PacketLogger = require('./packetLogger').default;
}
let MetaApiWebsocketClient = class MetaApiWebsocketClient {
    /**
   * Restarts the account synchronization process on an out of order packet
   * @param {String} accountId account id
   * @param {Number} instanceIndex instance index
   * @param {Number} expectedSequenceNumber expected s/n
   * @param {Number} actualSequenceNumber actual s/n
   * @param {Object} packet packet data
   * @param {Date} receivedAt time the packet was received at
   */ onOutOfOrderPacket(accountId, instanceIndex, expectedSequenceNumber, actualSequenceNumber, packet, receivedAt) {
        const primaryAccountId = this._accountsByReplicaId[accountId];
        if (this._subscriptionManager.isSubscriptionActive(accountId)) {
            const level = this._latencyService.getSynchronizedAccountInstances(primaryAccountId).length ? 'debug' : 'error';
            this._logger[level]('MetaApi websocket client received an out of order ' + `packet type ${packet.type} for account id ${accountId}:${instanceIndex}. Expected s/n ` + `${expectedSequenceNumber} does not match the actual of ${actualSequenceNumber}`);
            this.ensureSubscribe(accountId, instanceIndex);
        }
    }
    /**
   * Patch server URL for use in unit tests
   * @param {String} url patched server URL
   */ set url(url) {
        this._url = url;
    }
    /**
   * Websocket client predefined region
   * @returns {String} predefined region
   */ get region() {
        return this._region;
    }
    /**
   * Returns the list of socket instance dictionaries
   * @return {Object[]} list of socket instance dictionaries
   */ get socketInstances() {
        return this._socketInstances;
    }
    /**
   * Returns the dictionary of socket instances by account ids
   * @return {Object} dictionary of socket instances by account ids
   */ get socketInstancesByAccounts() {
        return this._socketInstancesByAccounts;
    }
    /**
   * Returns the dictionary of account replicas by region
   * @return {Object} dictionary of account replicas by region
   */ get accountReplicas() {
        return this._accountReplicas;
    }
    /**
   * Returns the dictionary of primary account ids by replica ids
   * @return {Object} dictionary of primary account ids by replica ids
   */ get accountsByReplicaId() {
        return this._accountsByReplicaId;
    }
    /**
   * Returns clear account cache job. Used for tests
   * @return {Function} clear account cache job
   */ get clearAccountCacheJob() {
        return this._clearAccountCacheJob.bind(this);
    }
    /**
   * Returns latency service
   * @returns {LatencyService} latency service
   */ get latencyService() {
        return this._latencyService;
    }
    /**
   * Returns the list of subscribed account ids
   * @param {Number} instanceNumber instance index number
   * @param {String} socketInstanceIndex socket instance index
   * @param {String} region server region
   * @return {string[]} list of subscribed account ids
   */ subscribedAccountIds(instanceNumber, socketInstanceIndex, region) {
        const connectedIds = [];
        if (this._socketInstancesByAccounts[instanceNumber]) {
            Object.keys(this._connectedHosts).forEach((instanceId)=>{
                const accountId = instanceId.split(':')[0];
                const accountRegion = this.getAccountRegion(accountId);
                if (!connectedIds.includes(accountId) && this._socketInstancesByAccounts[instanceNumber][accountId] !== undefined && (this._socketInstancesByAccounts[instanceNumber][accountId] === socketInstanceIndex || socketInstanceIndex === undefined) && accountRegion === region) {
                    connectedIds.push(accountId);
                }
            });
        }
        return connectedIds;
    }
    /**
   * Returns websocket client connection status
   * @param {Number} instanceNumber instance index number
   * @param {Number} socketInstanceIndex socket instance index
   * @param {String} region server region
   * @returns {Boolean} websocket client connection status
   */ connected(instanceNumber, socketInstanceIndex, region) {
        const instance = this._socketInstances[region] && this._socketInstances[region][instanceNumber].length > socketInstanceIndex ? this._socketInstances[region][instanceNumber][socketInstanceIndex] : null;
        return instance && instance.socket && instance.socket.connected || false;
    }
    /**
   * Returns list of accounts assigned to instance
   * @param {Number} instanceNumber instance index number
   * @param {String} socketInstanceIndex socket instance index
   * @param {String} region server region
   * @returns 
   */ getAssignedAccounts(instanceNumber, socketInstanceIndex, region) {
        const accountIds = [];
        Object.keys(this._socketInstancesByAccounts[instanceNumber]).forEach((key)=>{
            const accountRegion = this.getAccountRegion(key);
            if (accountRegion === region && this._socketInstancesByAccounts[instanceNumber][key] === socketInstanceIndex) {
                accountIds.push(key);
            }
        });
        return accountIds;
    }
    /**
   * Returns account region by id
   * @param {String} accountId account id
   * @returns {String} account region
   */ getAccountRegion(accountId) {
        return this._regionsByAccounts[accountId] && this._regionsByAccounts[accountId].region;
    }
    /**
   * Adds account cache info
   * @param {String} accountId account id
   * @param {Object} replicas account replicas
   */ addAccountCache(accountId, replicas) {
        this._accountReplicas[accountId] = replicas;
        Object.keys(replicas).forEach((region)=>{
            const replicaId = replicas[region];
            if (!this._regionsByAccounts[replicaId]) {
                this._regionsByAccounts[replicaId] = {
                    region,
                    connections: 1,
                    lastUsed: Date.now()
                };
            } else {
                this._regionsByAccounts[replicaId].connections++;
            }
            this._accountsByReplicaId[replicaId] = accountId;
        });
        this._logger.debug(`${accountId}: added account cache`);
    }
    /**
   * Updates account cache info
   * @param {String} accountId account id
   * @param {Object} replicas account replicas
   */ updateAccountCache(accountId, replicas) {
        const oldReplicas = this._accountReplicas[accountId];
        if (oldReplicas) {
            const connectionCount = this._regionsByAccounts[accountId].connections;
            Object.keys(oldReplicas).forEach((region)=>{
                const replicaId = replicas[region];
                delete this._accountsByReplicaId[replicaId];
                delete this._regionsByAccounts[replicaId];
            });
            this._accountReplicas[accountId] = replicas;
            Object.keys(replicas).forEach((region)=>{
                const replicaId = replicas[region];
                this._regionsByAccounts[replicaId] = {
                    region,
                    connections: connectionCount,
                    lastUsed: Date.now()
                };
                this._accountsByReplicaId[replicaId] = accountId;
            });
            this._logger.debug(`${accountId}: updated account cache`);
        }
    }
    /**
   * Removes account region info
   * @param {String} accountId account id
   */ removeAccountCache(accountId) {
        if (this._regionsByAccounts[accountId]) {
            if (this._regionsByAccounts[accountId].connections > 0) {
                this._regionsByAccounts[accountId].connections--;
            }
        }
    }
    /**
   * Locks subscription for a socket instance based on TooManyRequestsError metadata
   * @param {Number} instanceNumber instance index number
   * @param {String} socketInstanceIndex socket instance index
   * @param {String} region server region
   * @param {Object} metadata TooManyRequestsError metadata
   */ lockSocketInstance(instanceNumber, socketInstanceIndex, region, metadata) {
        var _this = this;
        return _async_to_generator(function*() {
            if (metadata.type === 'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_USER') {
                _this._subscribeLock = {
                    recommendedRetryTime: metadata.recommendedRetryTime,
                    lockedAtAccounts: _this.subscribedAccountIds(instanceNumber, undefined, region).length,
                    lockedAtTime: Date.now()
                };
            } else {
                const subscribedAccounts = _this.subscribedAccountIds(instanceNumber, socketInstanceIndex, region);
                if (subscribedAccounts.length === 0) {
                    const socketInstance = _this.socketInstances[region][instanceNumber][socketInstanceIndex];
                    socketInstance.socket.close();
                    yield _this._reconnect(instanceNumber, socketInstanceIndex, region);
                } else {
                    const instance = _this.socketInstances[region][instanceNumber][socketInstanceIndex];
                    instance.subscribeLock = {
                        recommendedRetryTime: metadata.recommendedRetryTime,
                        type: metadata.type,
                        lockedAtAccounts: subscribedAccounts.length
                    };
                }
            }
        })();
    }
    /**
   * Connects to MetaApi server via socket.io protocol
   * @param {Number} instanceNumber instance index number
   * @param {String} region server region
   * @returns {Promise} promise which resolves when connection is established
   */ connect(instanceNumber, region) {
        var _this = this;
        return _async_to_generator(function*() {
            var _this__socketInstances, _region, _this__socketInstances_region, _instanceNumber;
            if (_this._region && region !== _this._region) {
                throw new ValidationError(`Trying to connect to ${region} region, but configured with ${_this._region}`);
            }
            let clientId = Math.random();
            let resolve;
            let result = new Promise((res, rej)=>{
                resolve = res;
            });
            (_this__socketInstances = _this._socketInstances)[_region = region] || (_this__socketInstances[_region] = {});
            (_this__socketInstances_region = _this._socketInstances[region])[_instanceNumber = instanceNumber] || (_this__socketInstances_region[_instanceNumber] = []);
            const socketInstanceIndex = _this._socketInstances[region][instanceNumber].length;
            const instance = {
                id: socketInstanceIndex,
                reconnectWaitTime: _this._socketMinimumReconnectTimeout,
                connected: false,
                requestResolves: {},
                resolved: false,
                connectResult: result,
                sessionId: randomstring.generate(32),
                isReconnecting: false,
                socket: null,
                synchronizationThrottler: new SynchronizationThrottler(_this, socketInstanceIndex, instanceNumber, region, _this._synchronizationThrottlerOpts),
                subscribeLock: null,
                instanceNumber
            };
            instance.connected = true;
            _this._socketInstances[region][instanceNumber].push(instance);
            instance.synchronizationThrottler.start();
            const serverUrl = yield _this._getServerUrl(instanceNumber, socketInstanceIndex, region);
            const socketInstance = socketIO(serverUrl, {
                path: '/ws',
                reconnection: true,
                reconnectionDelay: 1000,
                reconnectionDelayMax: 5000,
                reconnectionAttempts: Infinity,
                timeout: _this._connectTimeout,
                extraHeaders: {
                    'Client-Id': clientId
                },
                query: {
                    'auth-token': _this._token,
                    clientId: clientId,
                    protocol: 3
                }
            });
            instance.socket = socketInstance;
            socketInstance.on('connect', /*#__PURE__*/ _async_to_generator(function*() {
                // eslint-disable-next-line no-console
                _this._logger.info(`${region}:${instanceNumber}: MetaApi websocket client connected to the MetaApi server`);
                instance.reconnectWaitTime = _this._socketMinimumReconnectTimeout;
                instance.isReconnecting = false;
                if (!instance.resolved) {
                    instance.resolved = true;
                    resolve();
                } else {
                    yield _this._fireReconnected(instanceNumber, instance.id, region);
                }
                if (!instance.connected) {
                    instance.socket.close();
                }
            }));
            socketInstance.on('reconnect', /*#__PURE__*/ _async_to_generator(function*() {
                instance.isReconnecting = false;
                _this._logger.info(`${region}:${instanceNumber}: MetaApi websocket client reconnected`);
                yield _this._fireReconnected(instanceNumber, instance.id, region);
            }));
            socketInstance.on('connect_error', function() {
                var _ref = _async_to_generator(function*(err) {
                    // eslint-disable-next-line no-console
                    _this._logger.error(`${region}:${instanceNumber}: MetaApi websocket client connection error`, err);
                    instance.isReconnecting = false;
                    yield _this._reconnect(instanceNumber, instance.id, region);
                });
                return function(err) {
                    return _ref.apply(this, arguments);
                };
            }());
            socketInstance.on('connect_timeout', function() {
                var _ref = _async_to_generator(function*(timeout) {
                    // eslint-disable-next-line no-console
                    _this._logger.error(`${region}:${instanceNumber}: MetaApi websocket client connection timeout`);
                    instance.isReconnecting = false;
                    if (!instance.resolved) {
                        yield _this._reconnect(instanceNumber, instance.id, region);
                    }
                });
                return function(timeout) {
                    return _ref.apply(this, arguments);
                };
            }());
            socketInstance.on('disconnect', function() {
                var _ref = _async_to_generator(function*(reason) {
                    instance.synchronizationThrottler.onDisconnect();
                    // eslint-disable-next-line no-console
                    _this._logger.info(`${region}:${instanceNumber}: MetaApi websocket client disconnected from the ` + `MetaApi server because of ${reason}`);
                    instance.isReconnecting = false;
                    yield _this._reconnect(instanceNumber, instance.id, region);
                });
                return function(reason) {
                    return _ref.apply(this, arguments);
                };
            }());
            socketInstance.on('error', function() {
                var _ref = _async_to_generator(function*(error) {
                    // eslint-disable-next-line no-console
                    _this._logger.error(`${region}:${instanceNumber}: MetaApi websocket client error`, error);
                    instance.isReconnecting = false;
                    yield _this._reconnect(instanceNumber, instance.id, region);
                });
                return function(error) {
                    return _ref.apply(this, arguments);
                };
            }());
            socketInstance.on('response', (data)=>{
                if (typeof data === 'string') {
                    data = JSON.parse(data);
                }
                _this._logger.debug(()=>`${data.accountId}: Response received: ${JSON.stringify({
                        requestId: data.requestId,
                        timestamps: data.timestamps
                    })}`);
                let requestResolve = instance.requestResolves[data.requestId] || {
                    resolve: ()=>{},
                    reject: ()=>{}
                };
                delete instance.requestResolves[data.requestId];
                _this._convertIsoTimeToDate(data);
                requestResolve.resolve(data);
                if (data.timestamps && requestResolve.type) {
                    data.timestamps.clientProcessingFinished = new Date();
                    for (let listener of _this._latencyListeners){
                        Promise.resolve().then(()=>requestResolve.type === 'trade' ? listener.onTrade(data.accountId, data.timestamps) : listener.onResponse(data.accountId, requestResolve.type, data.timestamps)).catch((error)=>_this._logger.error('Failed to process onResponse event for account ' + data.accountId + ', request type ' + requestResolve.type, error));
                    }
                }
            });
            socketInstance.on('processingError', (data)=>{
                let requestResolve = instance.requestResolves[data.requestId] || {
                    resolve: ()=>{},
                    reject: ()=>{}
                };
                delete instance.requestResolves[data.requestId];
                requestResolve.reject(_this._convertError(data));
            });
            // eslint-disable-next-line complexity
            socketInstance.on('synchronization', function() {
                var _ref = _async_to_generator(function*(data) {
                    if (typeof data === 'string') {
                        data = JSON.parse(data);
                    }
                    if (data.instanceIndex && data.instanceIndex !== instanceNumber) {
                        _this._logger.trace(()=>`${data.accountId}:${data.instanceNumber}: received packet with wrong instance ` + `index via a socket with instance number of ${instanceNumber}, data=${JSON.stringify({
                                type: data.type,
                                sequenceNumber: data.sequenceNumber,
                                sequenceTimestamp: data.sequenceTimestamp,
                                synchronizationId: data.synchronizationId,
                                application: data.application,
                                host: data.host,
                                specificationsUpdated: data.specificationsUpdated,
                                positionsUpdated: data.positionsUpdated,
                                ordersUpdated: data.ordersUpdated,
                                specifications: data.specifications ? (data.specifications || []).length : undefined
                            })}`);
                        return;
                    }
                    if (!_this._regionsByAccounts[data.accountId]) {
                        _this._regionsByAccounts[data.accountId] = {
                            region,
                            connections: 0,
                            lastUsed: Date.now()
                        };
                    }
                    _this._logger.trace(()=>`${data.accountId}:${data.instanceIndex}: Sync packet received: ${JSON.stringify({
                            type: data.type,
                            sequenceNumber: data.sequenceNumber,
                            sequenceTimestamp: data.sequenceTimestamp,
                            synchronizationId: data.synchronizationId,
                            application: data.application,
                            host: data.host,
                            specificationsUpdated: data.specificationsUpdated,
                            positionsUpdated: data.positionsUpdated,
                            ordersUpdated: data.ordersUpdated,
                            specifications: data.specifications ? (data.specifications || []).length : undefined
                        })}, ` + `active listeners: ${(_this._synchronizationListeners[data.accountId] || []).length}`);
                    let activeSynchronizationIds = instance.synchronizationThrottler.activeSynchronizationIds;
                    if (!data.synchronizationId || activeSynchronizationIds.includes(data.synchronizationId)) {
                        if (_this._packetLogger) {
                            yield _this._packetLogger.logPacket(data);
                        }
                        const ignoredPacketTypes = [
                            'disconnected',
                            'status',
                            'keepalive'
                        ];
                        if (!_this._subscriptionManager.isSubscriptionActive(data.accountId) && !ignoredPacketTypes.includes(data.type)) {
                            _this._logger.debug(`${data.accountId}: Packet arrived to inactive connection, attempting` + ` unsubscribe, packet: ${data.type}`);
                            if (_this._throttleRequest('unsubscribe', data.accountId, data.instanceIndex, _this._unsubscribeThrottlingInterval)) {
                                _this.unsubscribe(data.accountId).catch((err)=>{
                                    _this._logger.warn(`${data.accountId}:${data.instanceIndex || 0}: failed to unsubscribe`, err);
                                });
                            }
                            return;
                        }
                        _this._convertIsoTimeToDate(data);
                    } else {
                        data.type = 'noop';
                    }
                    _this.queuePacket(data);
                });
                return function(data) {
                    return _ref.apply(this, arguments);
                };
            }());
            return result;
        })();
    }
    /**
   * Closes connection to MetaApi server
   */ close() {
        Object.keys(this._socketInstances).forEach((region)=>{
            Object.keys(this._socketInstances[region]).forEach((instanceNumber)=>{
                this._socketInstances[region][instanceNumber].forEach(function() {
                    var _ref = _async_to_generator(function*(instance) {
                        if (instance.connected) {
                            instance.connected = false;
                            yield instance.socket.close();
                            for (let requestResolve of Object.values(instance.requestResolves)){
                                requestResolve.reject(new Error('MetaApi connection closed'));
                            }
                            instance.requestResolves = {};
                        }
                    });
                    return function(instance) {
                        return _ref.apply(this, arguments);
                    };
                }());
                this._socketInstancesByAccounts[instanceNumber] = {};
                this._socketInstances[region][instanceNumber] = [];
            });
        });
        this._synchronizationListeners = {};
        this._latencyListeners = [];
        this._packetOrderer.stop();
    }
    /**
   * Stops the client
   */ stop() {
        clearInterval(this._clearAccountCacheInterval);
        clearInterval(this._clearInactiveSyncDataJob);
        this._latencyService.stop();
    }
    /**
   * Returns account information for a specified MetaTrader account.
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {GetAccountInformationOptions} [options] additional request options
   * @returns {Promise<MetatraderAccountInformation>} promise resolving with account information
   */ getAccountInformation(accountId, options) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, _object_spread({
                application: 'RPC',
                type: 'getAccountInformation'
            }, options));
            return response.accountInformation;
        })();
    }
    /**
   * Returns positions for a specified MetaTrader account.
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {GetPositionsOptions} [options] additional request options
   * @returns {Promise<Array<MetatraderPosition>} promise resolving with array of open positions
   */ getPositions(accountId, options) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, _object_spread({
                application: 'RPC',
                type: 'getPositions'
            }, options));
            return response.positions;
        })();
    }
    /**
   * Returns specific position for a MetaTrader account.
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} positionId position id
   * @param {GetPositionOptions} [options] additional request options
   * @return {Promise<MetatraderPosition>} promise resolving with MetaTrader position found
   */ getPosition(accountId, positionId, options) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, _object_spread({
                application: 'RPC',
                type: 'getPosition',
                positionId
            }, options));
            return response.position;
        })();
    }
    /**
   * Returns open orders for a specified MetaTrader account.
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {GetOrdersOptions} [options] additional request options
   * @return {Promise<Array<MetatraderOrder>>} promise resolving with open MetaTrader orders
   */ getOrders(accountId, options) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, _object_spread({
                application: 'RPC',
                type: 'getOrders'
            }, options));
            return response.orders;
        })();
    }
    /**
   * Returns specific open order for a MetaTrader account.
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} orderId order id (ticket number)
   * @param {GetOrderOptions} [options] additional request options
   * @return {Promise<MetatraderOrder>} promise resolving with metatrader order found
   */ getOrder(accountId, orderId, options) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, _object_spread({
                application: 'RPC',
                type: 'getOrder',
                orderId
            }, options));
            return response.order;
        })();
    }
    /**
   * MetaTrader history orders search query response
   * @typedef {Object} MetatraderHistoryOrders
   * @property {Array<MetatraderOrder>} historyOrders array of history orders returned
   * @property {Boolean} synchronizing flag indicating that history order initial synchronization is still in progress
   * and thus search results may be incomplete
   */ /**
   * Returns the history of completed orders for a specific ticket number.
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} ticket ticket number (order id)
   * @returns {Promise<MetatraderHistoryOrders>} promise resolving with request results containing history orders found
   */ getHistoryOrdersByTicket(accountId, ticket) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, {
                application: 'RPC',
                type: 'getHistoryOrdersByTicket',
                ticket
            });
            return {
                historyOrders: response.historyOrders,
                synchronizing: response.synchronizing
            };
        })();
    }
    /**
   * Returns the history of completed orders for a specific position id
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} positionId position id
   * @returns {Promise<MetatraderHistoryOrders>} promise resolving with request results containing history orders found
   */ getHistoryOrdersByPosition(accountId, positionId) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, {
                application: 'RPC',
                type: 'getHistoryOrdersByPosition',
                positionId
            });
            return {
                historyOrders: response.historyOrders,
                synchronizing: response.synchronizing
            };
        })();
    }
    /**
   * Returns the history of completed orders for a specific time range
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {Date} startTime start of time range, inclusive
   * @param {Date} endTime end of time range, exclusive
   * @param {Number} offset pagination offset, default is 0
   * @param {Number} limit pagination limit, default is 1000
   * @returns {Promise<MetatraderHistoryOrders>} promise resolving with request results containing history orders found
   */ getHistoryOrdersByTimeRange(accountId, startTime, endTime, offset = 0, limit = 1000) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, {
                application: 'RPC',
                type: 'getHistoryOrdersByTimeRange',
                startTime,
                endTime,
                offset,
                limit
            });
            return {
                historyOrders: response.historyOrders,
                synchronizing: response.synchronizing
            };
        })();
    }
    /**
   * MetaTrader history deals search query response
   * @typedef {Object} MetatraderDeals
   * @property {Array<MetatraderDeal>} deals array of history deals returned
   * @property {Boolean} synchronizing flag indicating that deal initial synchronization is still in progress
   * and thus search results may be incomplete
   */ /**
   * MetaTrader deal
   * @typedef {Object} MetatraderDeal
   * @property {String} id deal id (ticket number)
   * @property {String} type deal type (one of DEAL_TYPE_BUY, DEAL_TYPE_SELL, DEAL_TYPE_BALANCE, DEAL_TYPE_CREDIT,
   * DEAL_TYPE_CHARGE, DEAL_TYPE_CORRECTION, DEAL_TYPE_BONUS, DEAL_TYPE_COMMISSION, DEAL_TYPE_COMMISSION_DAILY,
   * DEAL_TYPE_COMMISSION_MONTHLY, DEAL_TYPE_COMMISSION_AGENT_DAILY, DEAL_TYPE_COMMISSION_AGENT_MONTHLY,
   * DEAL_TYPE_INTEREST, DEAL_TYPE_BUY_CANCELED, DEAL_TYPE_SELL_CANCELED, DEAL_DIVIDEND, DEAL_DIVIDEND_FRANKED,
   * DEAL_TAX). See https://www.mql5.com/en/docs/constants/tradingconstants/dealproperties#enum_deal_type
   * @property {String} entryType deal entry type (one of DEAL_ENTRY_IN, DEAL_ENTRY_OUT, DEAL_ENTRY_INOUT,
   * DEAL_ENTRY_OUT_BY). See https://www.mql5.com/en/docs/constants/tradingconstants/dealproperties#enum_deal_entry
   * @property {String} [symbol] symbol deal relates to
   * @property {Number} [magic] deal magic number, identifies the EA which initiated the deal
   * @property {Date} time time the deal was conducted at
   * @property {String} brokerTime time time the deal was conducted at, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {Number} [volume] deal volume
   * @property {Number} [price] the price the deal was conducted at
   * @property {Number} [commission] deal commission
   * @property {Number} [swap] deal swap
   * @property {Number} profit deal profit
   * @property {String} [positionId] id of position the deal relates to
   * @property {String} [orderId] id of order the deal relates to
   * @property {String} [comment] deal comment. The sum of the line lengths of the comment and the clientId
   * must be less than or equal to 26. For more information see https://metaapi.cloud/docs/client/clientIdUsage/
   * @property {String} [brokerComment] current comment value on broker side (possibly overriden by the broker)
   * @property {String} [clientId] client-assigned id. The id value can be assigned when submitting a trade and
   * will be present on position, history orders and history deals related to the trade. You can use this field to bind
   * your trades to objects in your application and then track trade progress. The sum of the line lengths of the
   * comment and the clientId must be less than or equal to 26. For more information see
   * https://metaapi.cloud/docs/client/clientIdUsage/
   * @property {String} platform platform id (mt4 or mt5)
   * @property {String} [reason] optional deal execution reason. One of DEAL_REASON_CLIENT, DEAL_REASON_MOBILE,
   * DEAL_REASON_WEB, DEAL_REASON_EXPERT, DEAL_REASON_SL, DEAL_REASON_TP, DEAL_REASON_SO, DEAL_REASON_ROLLOVER,
   * DEAL_REASON_VMARGIN, DEAL_REASON_SPLIT, DEAL_REASON_UNKNOWN. See
   * https://www.mql5.com/en/docs/constants/tradingconstants/dealproperties#enum_deal_reason.
   * @property {Number} [accountCurrencyExchangeRate] current exchange rate of account currency into account base
   * currency (USD if you did not override it)
   * @property {number} [stopLoss] deal stop loss. For MT5 opening deal this is the SL of the order opening the 
   * position. For MT4 deals or MT5 closing deal this is the last known position SL.
   * @property {number} [takeProfit] deal take profit. For MT5 opening deal this is the TP of the order opening the 
   * position. For MT4 deals or MT5 closing deal this is the last known position TP.
   */ /**
   * Returns history deals with a specific ticket number
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} ticket ticket number (deal id for MT5 or order id for MT4)
   * @returns {Promise<MetatraderDeals>} promise resolving with request results containing deals found
   */ getDealsByTicket(accountId, ticket) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, {
                application: 'RPC',
                type: 'getDealsByTicket',
                ticket
            });
            return {
                deals: response.deals,
                synchronizing: response.synchronizing
            };
        })();
    }
    /**
   * Returns history deals for a specific position id
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {String} positionId position id
   * @returns {Promise<MetatraderDeals>} promise resolving with request results containing deals found
   */ getDealsByPosition(accountId, positionId) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, {
                application: 'RPC',
                type: 'getDealsByPosition',
                positionId
            });
            return {
                deals: response.deals,
                synchronizing: response.synchronizing
            };
        })();
    }
    /**
   * Returns history deals with for a specific time range
   * @param {String} accountId id of the MetaTrader account to return information for
   * @param {Date} startTime start of time range, inclusive
   * @param {Date} endTime end of time range, exclusive
   * @param {Number} offset pagination offset, default is 0
   * @param {Number} limit pagination limit, default is 1000
   * @returns {Promise<MetatraderDeals>} promise resolving with request results containing deals found
   */ getDealsByTimeRange(accountId, startTime, endTime, offset = 0, limit = 1000) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, {
                application: 'RPC',
                type: 'getDealsByTimeRange',
                startTime,
                endTime,
                offset,
                limit
            });
            return {
                deals: response.deals,
                synchronizing: response.synchronizing
            };
        })();
    }
    /**
   * Clears the order and transaction history of a specified application and removes the application
   * @param {String} accountId id of the MetaTrader account to remove history and application for
   * @return {Promise} promise resolving when the history is cleared
   */ removeApplication(accountId) {
        return this.rpcRequest(accountId, {
            type: 'removeApplication'
        });
    }
    /**
   * MetaTrader trade response
   * @typedef {Object} MetatraderTradeResponse
   * @property {Number} numericCode numeric response code, see
   * https://www.mql5.com/en/docs/constants/errorswarnings/enum_trade_return_codes and
   * https://book.mql4.com/appendix/errors. Response codes which indicate success are 0, 10008-10010, 10025. The rest
   * codes are errors
   * @property {String} stringCode string response code, see
   * https://www.mql5.com/en/docs/constants/errorswarnings/enum_trade_return_codes and
   * https://book.mql4.com/appendix/errors. Response codes which indicate success are ERR_NO_ERROR,
   * TRADE_RETCODE_PLACED, TRADE_RETCODE_DONE, TRADE_RETCODE_DONE_PARTIAL, TRADE_RETCODE_NO_CHANGES. The rest codes are
   * errors.
   * @property {String} message human-readable response message
   * @property {String} orderId order id which was created/modified during the trade
   * @property {String} positionId position id which was modified during the trade
   */ /**
   * Execute a trade on a connected MetaTrader account
   * @param {String} accountId id of the MetaTrader account to execute trade for
   * @param {MetatraderTrade} trade trade to execute (see docs for possible trade types)
   * @param {String} [application] application to use
   * @param {String} [reliability] account reliability
   * @returns {Promise<MetatraderTradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */ // eslint-disable-next-line complexity
    trade(accountId, trade, application, reliability) {
        var _this = this;
        return _async_to_generator(function*() {
            let response;
            if (application === 'RPC') {
                response = yield _this.rpcRequest(accountId, {
                    type: 'trade',
                    trade,
                    application
                });
            } else {
                response = yield _this.rpcRequestAllInstances(accountId, {
                    type: 'trade',
                    trade,
                    application: application || _this._application,
                    requestId: randomstring.generate(32)
                }, reliability);
            }
            response.response = response.response || {};
            response.response.stringCode = response.response.stringCode || response.response.description;
            response.response.numericCode = response.response.numericCode !== undefined ? response.response.numericCode : response.response.error;
            if ([
                'ERR_NO_ERROR',
                'TRADE_RETCODE_PLACED',
                'TRADE_RETCODE_DONE',
                'TRADE_RETCODE_DONE_PARTIAL',
                'TRADE_RETCODE_NO_CHANGES'
            ].includes(response.response.stringCode || response.response.description)) {
                return response.response;
            } else {
                throw new TradeError(response.response.message, response.response.numericCode, response.response.stringCode);
            }
        })();
    }
    /**
   * Creates a task that ensures the account gets subscribed to the server
   * @param {String} accountId account id to subscribe
   * @param {Number} instanceNumber instance index number
   */ ensureSubscribe(accountId, instanceNumber) {
        this._subscriptionManager.scheduleSubscribe(accountId, instanceNumber);
    }
    /**
   * Subscribes to the Metatrader terminal events
   * @param {String} accountId id of the MetaTrader account to subscribe to
   * @param {Number} instanceNumber instance index number
   * @returns {Promise} promise which resolves when subscription started
   */ subscribe(accountId, instanceNumber) {
        return this._subscriptionManager.subscribe(accountId, instanceNumber);
    }
    /**
   * Requests the terminal to start synchronization process
   * @param {String} accountId id of the MetaTrader account to synchronize
   * @param {Number} instanceIndex instance index
   * @param {String} host name of host to synchronize with
   * @param {String} synchronizationId synchronization request id
   * @param {Date} startingHistoryOrderTime from what date to start synchronizing history orders from. If not specified,
   * the entire order history will be downloaded.
   * @param {Date} startingDealTime from what date to start deal synchronization from. If not specified, then all
   * history deals will be downloaded.
   * @param {Function} getHashes function to get terminal state hashes
   * @returns {Promise} promise which resolves when synchronization started
   */ synchronize(accountId, instanceIndex, host, synchronizationId, startingHistoryOrderTime, startingDealTime, hashes) {
        var _this = this;
        return _async_to_generator(function*() {
            if (_this._getSocketInstanceByAccount(accountId, instanceIndex) === undefined) {
                _this._logger.debug(`${accountId}:${instanceIndex}: creating socket instance on synchronize`);
                yield _this._createSocketInstanceByAccount(accountId, instanceIndex);
            }
            const syncThrottler = _this._getSocketInstanceByAccount(accountId, instanceIndex).synchronizationThrottler;
            _this._synchronizationHashes[synchronizationId] = hashes;
            _this._synchronizationHashes[synchronizationId].lastUpdated = Date.now();
            return syncThrottler.scheduleSynchronize(accountId, {
                requestId: synchronizationId,
                version: 2,
                type: 'synchronize',
                startingHistoryOrderTime,
                startingDealTime,
                instanceIndex,
                host
            }, hashes);
        })();
    }
    /**
   * Waits for server-side terminal state synchronization to complete
   * @param {String} accountId id of the MetaTrader account to synchronize
   * @param {Number} [instanceNumber] instance index number
   * @param {String} applicationPattern MetaApi application regular expression pattern, default is .*
   * @param {Number} timeoutInSeconds timeout in seconds, default is 300 seconds
   * @param {String} [application] application to synchronize with
   * @returns {Promise} promise which resolves when synchronization started
   */ waitSynchronized(accountId, instanceNumber, applicationPattern, timeoutInSeconds, application) {
        return this.rpcRequest(accountId, {
            type: 'waitSynchronized',
            applicationPattern,
            timeoutInSeconds,
            instanceIndex: instanceNumber,
            application: application || this._application
        }, timeoutInSeconds + 1);
    }
    /**
   * Market data subscription
   * @typedef {Object} MarketDataSubscription
   * @property {string} type subscription type, one of quotes, candles, ticks, or marketDepth
   * @property {string} [timeframe] when subscription type is candles, defines the timeframe according to which the
   * candles must be generated. Allowed values for MT5 are 1m, 2m, 3m, 4m, 5m, 6m, 10m, 12m, 15m, 20m, 30m, 1h, 2h, 3h,
   * 4h, 6h, 8h, 12h, 1d, 1w, 1mn. Allowed values for MT4 are 1m, 5m, 15m 30m, 1h, 4h, 1d, 1w, 1mn
   * @property {number} [intervalInMilliseconds] defines how frequently the terminal will stream data to client. If not
   * set, then the value configured in account will be used
   */ /**
   * Subscribes on market data of specified symbol
   * @param {String} accountId id of the MetaTrader account
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @param {Array<MarketDataSubscription>} subscriptions array of market data subscription to create or update
   * @param {String} [reliability] account reliability
   * @returns {Promise} promise which resolves when subscription request was processed
   */ subscribeToMarketData(accountId, symbol, subscriptions, reliability) {
        return this.rpcRequestAllInstances(accountId, {
            type: 'subscribeToMarketData',
            symbol,
            subscriptions
        }, reliability);
    }
    /**
   * Refreshes market data subscriptions on the server to prevent them from expiring
   * @param {String} accountId id of the MetaTrader account
   * @param {Number} instanceNumber instance index number
   * @param {Array} subscriptions array of subscriptions to refresh
   */ refreshMarketDataSubscriptions(accountId, instanceNumber, subscriptions) {
        return this.rpcRequest(accountId, {
            type: 'refreshMarketDataSubscriptions',
            subscriptions,
            instanceIndex: instanceNumber
        });
    }
    /**
   * Market data unsubscription
   * @typedef {Object} MarketDataUnsubscription
   * @property {string} type subscription type, one of quotes, candles, ticks, or marketDepth
   */ /**
   * Unsubscribes from market data of specified symbol
   * @param {String} accountId id of the MetaTrader account
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @param {Array<MarketDataUnsubscription>} subscriptions array of subscriptions to cancel
   * @param {String} [reliability] account reliability
   * @returns {Promise} promise which resolves when unsubscription request was processed
   */ unsubscribeFromMarketData(accountId, symbol, subscriptions, reliability) {
        return this.rpcRequestAllInstances(accountId, {
            type: 'unsubscribeFromMarketData',
            symbol,
            subscriptions
        }, reliability);
    }
    /**
   * Retrieves symbols available on an account
   * @param {String} accountId id of the MetaTrader account to retrieve symbols for
   * @returns {Promise<Array<string>>} promise which resolves when symbols are retrieved
   */ getSymbols(accountId) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, {
                application: 'RPC',
                type: 'getSymbols'
            });
            return response.symbols;
        })();
    }
    /**
   * Retrieves specification for a symbol
   * @param {String} accountId id of the MetaTrader account to retrieve symbol specification for
   * @param {String} symbol symbol to retrieve specification for
   * @returns {Promise<MetatraderSymbolSpecification>} promise which resolves when specification is retrieved
   */ getSymbolSpecification(accountId, symbol) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, {
                application: 'RPC',
                type: 'getSymbolSpecification',
                symbol
            });
            return response.specification;
        })();
    }
    /**
   * Retrieves price for a symbol
   * @param {String} accountId id of the MetaTrader account to retrieve symbol price for
   * @param {String} symbol symbol to retrieve price for
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderSymbolPrice>} promise which resolves when price is retrieved
   */ getSymbolPrice(accountId, symbol, keepSubscription = false) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, {
                application: 'RPC',
                type: 'getSymbolPrice',
                symbol,
                keepSubscription
            });
            return response.price;
        })();
    }
    /**
   * Retrieves price for a symbol
   * @param {string} accountId id of the MetaTrader account to retrieve candle for
   * @param {string} symbol symbol to retrieve candle for
   * @param {string} timeframe defines the timeframe according to which the candle must be generated. Allowed values for
   * MT5 are 1m, 2m, 3m, 4m, 5m, 6m, 10m, 12m, 15m, 20m, 30m, 1h, 2h, 3h, 4h, 6h, 8h, 12h, 1d, 1w, 1mn. Allowed values
   * for MT4 are 1m, 5m, 15m 30m, 1h, 4h, 1d, 1w, 1mn
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderCandle>} promise which resolves when candle is retrieved
   */ getCandle(accountId, symbol, timeframe, keepSubscription = false) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, {
                application: 'RPC',
                type: 'getCandle',
                symbol,
                timeframe,
                keepSubscription
            });
            return response.candle;
        })();
    }
    /**
   * Retrieves latest tick for a symbol. MT4 G1 accounts do not support this API
   * @param {string} accountId id of the MetaTrader account to retrieve symbol tick for
   * @param {string} symbol symbol to retrieve tick for
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderTick>} promise which resolves when tick is retrieved
   */ getTick(accountId, symbol, keepSubscription = false) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, {
                application: 'RPC',
                type: 'getTick',
                symbol,
                keepSubscription
            });
            return response.tick;
        })();
    }
    /**
   * Retrieves latest order book for a symbol. MT4 accounts do not support this API
   * @param {string} accountId id of the MetaTrader account to retrieve symbol order book for
   * @param {string} symbol symbol to retrieve order book for
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderBook>} promise which resolves when order book is retrieved
   */ getBook(accountId, symbol, keepSubscription = false) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, {
                application: 'RPC',
                type: 'getBook',
                symbol,
                keepSubscription
            });
            return response.book;
        })();
    }
    /**
   * Forces refresh of most recent quote updates for symbols subscribed to by the terminal
   * @param {string} accountId id of the MetaTrader account
   * @returns {Promise<string[]>} promise which resolves with recent quote symbols that was initiated to process
   */ refreshTerminalState(accountId) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, {
                application: 'RPC',
                type: 'refreshTerminalState'
            });
            return response.symbols;
        })();
    }
    /**
   * Forces refresh and retrieves latest quotes for a subset of symbols the terminal is subscribed to
   * @param {string} accountId id of the MetaTrader account
   * @param {string[]} symbols quote symbols to refresh
   * @returns {Promise<RefreshedQuotes>} refreshed quotes and basic account information info
   */ refreshSymbolQuotes(accountId, symbols) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, {
                application: 'RPC',
                type: 'refreshSymbolQuotes',
                symbols
            });
            return response.refreshedQuotes;
        })();
    }
    /**
   * Sends client uptime stats to the server.
   * @param {String} accountId id of the MetaTrader account to save uptime
   * @param {Object} uptime uptime statistics to send to the server
   * @returns {Promise} promise which resolves when uptime statistics is submitted
   */ saveUptime(accountId, uptime) {
        return this.rpcRequest(accountId, {
            type: 'saveUptime',
            uptime
        });
    }
    /**
   * Unsubscribe from account
   * @param {String} accountId id of the MetaTrader account to unsubscribe
   * @returns {Promise} promise which resolves when socket unsubscribed
   */ unsubscribe(accountId) {
        var _this = this;
        return _async_to_generator(function*() {
            const region = _this.getAccountRegion(accountId);
            _this._latencyService.onUnsubscribe(accountId);
            const updateEventsToRemove = Object.keys(_this._updateEvents).filter((key)=>key.startsWith(accountId));
            updateEventsToRemove.forEach((key)=>delete _this._updateEvents[key]);
            if (_this._socketInstances[region]) {
                yield Promise.all(Object.keys(_this._socketInstances[region]).map(function() {
                    var _ref = _async_to_generator(function*(instanceNumber) {
                        try {
                            yield _this._subscriptionManager.unsubscribe(accountId, Number(instanceNumber));
                            delete _this._socketInstancesByAccounts[instanceNumber][accountId];
                        } catch (err) {
                            if (![
                                'TimeoutError',
                                'NotFoundError'
                            ].includes(err.name)) {
                                _this._logger.warn(`${accountId}:${instanceNumber}: failed to unsubscribe`, err);
                            }
                        }
                    });
                    return function(instanceNumber) {
                        return _ref.apply(this, arguments);
                    };
                }()));
            }
        })();
    }
    /**
   * Current server time (see https://metaapi.cloud/docs/client/models/serverTime/)
   * @typedef {Object} ServerTime
   * @property {Date} time current server time
   * @property {String} brokerTime current broker time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {Date} [lastQuoteTime] last quote time
   * @property {String} [lastQuoteBrokerTime] last quote time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   */ /**
   * Returns server time for a specified MetaTrader account
   * @param {string} accountId id of the MetaTrader account to return server time for
   * @returns {Promise<ServerTime>} promise resolving with server time
   */ getServerTime(accountId) {
        var _this = this;
        return _async_to_generator(function*() {
            let response = yield _this.rpcRequest(accountId, {
                application: 'RPC',
                type: 'getServerTime'
            });
            return response.serverTime;
        })();
    }
    /**
   * Margin required to open a trade (see https://metaapi.cloud/docs/client/models/margin/)
   * @typedef {Object} Margin
   * @property {number} [margin] margin required to open a trade. If margin can not be calculated, then this field is
   * not defined
   */ /**
   * Contains order to calculate margin for (see https://metaapi.cloud/docs/client/models/marginOrder/)
   * @typedef {Object} MarginOrder
   * @property {string} symbol order symbol
   * @property {string} type order type, one of ORDER_TYPE_BUY or ORDER_TYPE_SELL
   * @property {number} volume order volume, must be greater than 0
   * @property {number} openPrice order open price, must be greater than 0
   */ /**
   * Calculates margin required to open a trade on the specified trading account
   * @param {string} accountId id of the trading account to calculate margin for
   * @param {string} application application to send the request to
   * @param {string} reliability account reliability
   * @param {MarginOrder} order order to calculate margin for
   * @returns {Promise<Margin>} promise resolving with margin calculation result
   */ calculateMargin(accountId, application, reliability, order) {
        var _this = this;
        return _async_to_generator(function*() {
            let response;
            if (application === 'RPC') {
                response = yield _this.rpcRequest(accountId, {
                    application,
                    type: 'calculateMargin',
                    order
                });
            } else {
                response = yield _this.rpcRequestAllInstances(accountId, {
                    application,
                    type: 'calculateMargin',
                    order
                }, reliability);
            }
            return response.margin;
        })();
    }
    /**
   * Calls onUnsubscribeRegion listener event 
   * @param {String} accountId account id
   * @param {String} region account region to unsubscribe
   */ unsubscribeAccountRegion(accountId, region) {
        var _this = this;
        return _async_to_generator(function*() {
            const unsubscribePromises = [];
            for (let listener of _this._synchronizationListeners[accountId] || []){
                unsubscribePromises.push(Promise.resolve(_async_to_generator(function*() {
                    yield _this._processEvent(()=>listener.onUnsubscribeRegion(region), `${accountId}:${region}:onUnsubscribeRegion`, true);
                })()).catch((err)=>_this._logger.error(`${accountId}:${region}: Failed to notify listener ` + 'about onUnsubscribeRegion event', err)));
            }
            yield Promise.all(unsubscribePromises);
        })();
    }
    /**
   * Adds synchronization listener for specific account
   * @param {String} accountId account id
   * @param {SynchronizationListener} listener synchronization listener to add
   */ addSynchronizationListener(accountId, listener) {
        this._logger.trace(`${accountId}: Added synchronization listener`);
        let listeners = this._synchronizationListeners[accountId];
        if (!listeners) {
            listeners = [];
            this._synchronizationListeners[accountId] = listeners;
        }
        listeners.push(listener);
    }
    /**
   * Removes synchronization listener for specific account
   * @param {String} accountId account id
   * @param {SynchronizationListener} listener synchronization listener to remove
   */ removeSynchronizationListener(accountId, listener) {
        this._logger.trace(`${accountId}: Removed synchronization listener`);
        let listeners = this._synchronizationListeners[accountId];
        if (!listeners) {
            listeners = [];
        }
        listeners = listeners.filter((l)=>l !== listener);
        this._synchronizationListeners[accountId] = listeners;
    }
    /**
   * Adds latency listener
   * @param {LatencyListener} listener latency listener to add
   */ addLatencyListener(listener) {
        this._latencyListeners.push(listener);
    }
    /**
   * Removes latency listener
   * @param {LatencyListener} listener latency listener to remove
   */ removeLatencyListener(listener) {
        this._latencyListeners = this._latencyListeners.filter((l)=>l !== listener);
    }
    /**
   * Adds reconnect listener
   * @param {ReconnectListener} listener reconnect listener to add
   * @param {String} accountId account id of listener
   */ addReconnectListener(listener, accountId) {
        this._reconnectListeners.push({
            accountId,
            listener
        });
    }
    /**
   * Removes reconnect listener
   * @param {ReconnectListener} listener listener to remove
   */ removeReconnectListener(listener) {
        this._reconnectListeners = this._reconnectListeners.filter((l)=>l.listener !== listener);
    }
    /**
   * Removes all listeners. Intended for use in unit tests.
   */ removeAllListeners() {
        this._synchronizationListeners = {};
        this._reconnectListeners = [];
    }
    /**
   * Clears account or replica data from client records and unsubscribes
   * @param {string} accountId account id to process the removal of
   */ onAccountDeleted(accountId) {
        this._subscriptionManager.cancelAccount(accountId);
        this._latencyService.onUnsubscribe(accountId);
        const masterAccountId = this._accountsByReplicaId[accountId];
        if (masterAccountId) {
            if (masterAccountId === accountId) {
                const regionData = this._accountReplicas[masterAccountId];
                const replicas = Object.values(regionData);
                Object.keys(this._synchronizationIdByInstance).filter((instance)=>instance.startsWith(`${masterAccountId}`)).forEach((instance)=>delete this._synchronizationIdByInstance[instance]);
                replicas.forEach((replica)=>{
                    Object.values(this._socketInstancesByAccounts).forEach((instance)=>{
                        delete instance[replica];
                    });
                    delete this._accountsByReplicaId[replica];
                    delete this._regionsByAccounts[replica];
                });
                delete this._accountReplicas[masterAccountId];
                this._logger.debug(`${masterAccountId}: processed primary account removal`);
            } else {
                Object.values(this._socketInstancesByAccounts).forEach((instance)=>{
                    delete instance[accountId];
                });
                const regionData = this._regionsByAccounts[accountId];
                if (regionData) {
                    const region = regionData.region;
                    Object.keys(this._synchronizationIdByInstance).filter((instance)=>instance.startsWith(`${masterAccountId}:${region}`)).forEach((instance)=>delete this._synchronizationIdByInstance[instance]);
                    delete this._accountReplicas[masterAccountId][region];
                    this._logger.debug(`${masterAccountId}: processed removal of replica ${accountId}`);
                }
                delete this._accountsByReplicaId[accountId];
                delete this._regionsByAccounts[accountId];
            }
        }
    }
    /**
   * Queues an account packet for processing
   * @param {Object} packet packet to process
   */ queuePacket(packet) {
        const accountId = packet.accountId;
        const packets = this._packetOrderer.restoreOrder(packet).filter((p)=>p.type !== 'noop');
        if (this._sequentialEventProcessing && packet.sequenceNumber !== undefined) {
            const events = packets.map((packetItem)=>()=>Promise.resolve(this._processSynchronizationPacket(packetItem)));
            if (!this._eventQueues[accountId]) {
                this._eventQueues[accountId] = events;
                this._callAccountEvents(accountId);
            } else {
                this._eventQueues[accountId] = this._eventQueues[accountId].concat(events);
            }
        } else {
            packets.forEach((packetItem)=>this._processSynchronizationPacket(packetItem));
        }
    }
    /**
   * Queues account event for processing
   * @param {String} accountId account id
   * @param {String} name event label name
   * @param {Function} callable async or regular function to execute
   */ queueEvent(accountId, name, callable) {
        let event = ()=>this._processEvent(callable, `${accountId}:${name}`);
        if (this._sequentialEventProcessing) {
            if (!this._eventQueues[accountId]) {
                this._eventQueues[accountId] = [
                    event
                ];
                this._callAccountEvents(accountId);
            } else {
                this._eventQueues[accountId].push(event);
            }
        } else {
            event();
        }
    }
    _callAccountEvents(accountId) {
        var _this = this;
        return _async_to_generator(function*() {
            if (_this._eventQueues[accountId]) {
                while(_this._eventQueues[accountId].length){
                    yield _this._eventQueues[accountId][0]();
                    _this._eventQueues[accountId].shift();
                }
                delete _this._eventQueues[accountId];
            }
        })();
    }
    _reconnect(instanceNumber, socketInstanceIndex, region) {
        var _this = this;
        return _async_to_generator(function*() {
            const instance = _this.socketInstances[region][instanceNumber][socketInstanceIndex];
            if (instance) {
                while(!instance.socket.connected && !instance.isReconnecting && instance.connected){
                    yield _this._tryReconnect(instanceNumber, socketInstanceIndex, region);
                }
            }
        })();
    }
    _tryReconnect(instanceNumber, socketInstanceIndex, region) {
        const instance = this.socketInstances[region][instanceNumber][socketInstanceIndex];
        instance.reconnectWaitTime = Math.min(instance.reconnectWaitTime * 2, 30000);
        var _this = this;
        return new Promise((resolve)=>setTimeout(/*#__PURE__*/ _async_to_generator(function*() {
                if (!instance.socket.connected && !instance.isReconnecting && instance.connected) {
                    try {
                        instance.sessionId = randomstring.generate(32);
                        const clientId = Math.random();
                        instance.socket.close();
                        instance.socket.io.opts.extraHeaders['Client-Id'] = clientId;
                        instance.socket.io.opts.query.clientId = clientId;
                        instance.isReconnecting = true;
                        instance.socket.io.uri = yield _this._getServerUrl(instanceNumber, socketInstanceIndex, region);
                        instance.socket.connect();
                    } catch (error) {
                        instance.isReconnecting = false;
                    }
                }
                resolve();
            }), instance.reconnectWaitTime));
    }
    /**
   * Simulataneously sends RPC requests to all synchronized instances
   * @param {String} accountId metatrader account id
   * @param {Object} request base request data
   * @param {String} [reliability] account reliability
   * @param {Number} [timeoutInSeconds] request timeout in seconds
   */ rpcRequestAllInstances(accountId, request, reliability, timeoutInSeconds) {
        var _this = this;
        return _async_to_generator(function*() {
            if (reliability === 'high') {
                try {
                    return yield any([
                        0,
                        1
                    ].map((instanceNumber)=>{
                        return _this.rpcRequest(accountId, Object.assign({}, request, {
                            instanceIndex: instanceNumber
                        }), timeoutInSeconds);
                    }));
                } catch (error) {
                    throw error.errors[0];
                }
            } else {
                return yield _this.rpcRequest(accountId, request, timeoutInSeconds);
            }
        })();
    }
    /**
   * Makes a RPC request
   * @param {String} accountId metatrader account id
   * @param {Object} request base request data
   * @param {Number} [timeoutInSeconds] request timeout in seconds
   */ //eslint-disable-next-line complexity, max-statements
    rpcRequest(accountId, request, timeoutInSeconds) {
        var _this = this;
        return _async_to_generator(function*() {
            const ignoredRequestTypes = [
                'subscribe',
                'synchronize',
                'refreshMarketDataSubscriptions',
                'unsubscribe'
            ];
            const primaryAccountId = _this._accountsByReplicaId[accountId];
            let connectedInstance = _this._latencyService.getActiveAccountInstances(primaryAccountId)[0];
            if (!ignoredRequestTypes.includes(request.type)) {
                if (!connectedInstance) {
                    connectedInstance = yield _this._latencyService.waitConnectedInstance(accountId);
                }
                const activeRegion = connectedInstance.split(':')[1];
                accountId = _this._accountReplicas[primaryAccountId][activeRegion];
            }
            let socketInstanceIndex = null;
            let instanceNumber = 0;
            const region = _this.getAccountRegion(accountId);
            _this._refreshAccountRegion(accountId);
            if (request.instanceIndex !== undefined) {
                instanceNumber = request.instanceIndex;
            } else {
                if (connectedInstance) {
                    instanceNumber = Number(connectedInstance.split(':')[2]);
                }
                if (request.application !== 'RPC') {
                    request = Object.assign({}, request, {
                        instanceIndex: instanceNumber
                    });
                }
            }
            if (!_this._socketInstancesByAccounts[instanceNumber]) {
                _this._socketInstancesByAccounts[instanceNumber] = {};
            }
            if (!_this._socketInstances[region]) {
                _this._socketInstances[region] = {};
            }
            if (!_this._socketInstances[region][instanceNumber]) {
                _this._socketInstances[region][instanceNumber] = [];
            }
            if (_this._socketInstancesByAccounts[instanceNumber][accountId] !== undefined) {
                socketInstanceIndex = _this._socketInstancesByAccounts[instanceNumber][accountId];
            } else {
                _this._logger.debug(`${accountId}:${instanceNumber}: creating socket instance on RPC request`);
                yield _this._createSocketInstanceByAccount(accountId, instanceNumber);
                socketInstanceIndex = _this._socketInstancesByAccounts[instanceNumber][accountId];
            }
            const instance = _this._socketInstances[region][instanceNumber][socketInstanceIndex];
            if (!instance.connected) {
                _this._logger.debug(`${accountId}:${instanceNumber}: connecting socket instance on RPC request`);
                yield _this.connect(instanceNumber, region);
            } else if (!_this.connected(instanceNumber, socketInstanceIndex, region)) {
                yield instance.connectResult;
            }
            if (request.type === 'subscribe') {
                request.sessionId = instance.sessionId;
            }
            if ([
                'trade',
                'subscribe'
            ].includes(request.type)) {
                return _this._makeRequest(accountId, instanceNumber, request, timeoutInSeconds);
            }
            let retryCounter = 0;
            while(true){
                try {
                    return yield _this._makeRequest(accountId, instanceNumber, request, timeoutInSeconds);
                } catch (err) {
                    if (err.name === 'TooManyRequestsError') {
                        let calcRetryCounter = retryCounter;
                        let calcRequestTime = 0;
                        while(calcRetryCounter < _this._retries){
                            calcRetryCounter++;
                            calcRequestTime += Math.min(Math.pow(2, calcRetryCounter) * _this._minRetryDelayInSeconds, _this._maxRetryDelayInSeconds) * 1000;
                        }
                        const retryTime = new Date(err.metadata.recommendedRetryTime).getTime();
                        if (Date.now() + calcRequestTime > retryTime && retryCounter < _this._retries) {
                            if (Date.now() < retryTime) {
                                yield new Promise((res)=>setTimeout(res, retryTime - Date.now()));
                            }
                            retryCounter++;
                        } else {
                            throw err;
                        }
                    } else if ([
                        'NotSynchronizedError',
                        'TimeoutError',
                        'NotAuthenticatedError',
                        'InternalError'
                    ].includes(err.name) && retryCounter < _this._retries) {
                        yield new Promise((res)=>setTimeout(res, Math.min(Math.pow(2, retryCounter) * _this._minRetryDelayInSeconds, _this._maxRetryDelayInSeconds) * 1000));
                        retryCounter++;
                    } else {
                        throw err;
                    }
                    if (_this._socketInstancesByAccounts[instanceNumber][accountId] === undefined) {
                        throw err;
                    }
                }
            }
        })();
    }
    _makeRequest(accountId, instanceNumber, request, timeoutInSeconds) {
        const socketInstance = this._getSocketInstanceByAccount(accountId, instanceNumber);
        let requestId = request.requestId || randomstring.generate(32);
        request.timestamps = {
            clientProcessingStarted: new Date()
        };
        let result = Promise.race([
            new Promise((resolve, reject)=>socketInstance.requestResolves[requestId] = {
                    resolve,
                    reject,
                    type: request.type
                }),
            new Promise((resolve, reject)=>setTimeout(()=>{
                    reject(new TimeoutError(`MetaApi websocket client request ${request.requestId} of type ${request.type} ` + 'timed out. Please make sure your account is connected to broker before retrying your request.'));
                    delete socketInstance.requestResolves[requestId];
                }, timeoutInSeconds * 1000 || this._requestTimeout))
        ]);
        request.accountId = accountId;
        request.application = request.application || this._application;
        if (!request.requestId) {
            request.requestId = requestId;
        }
        if (request.type === 'unsubscribe' || request.application === 'RPC' || request.instanceIndex === socketInstance.instanceNumber) {
            this._logger.debug(()=>`${accountId}: Sending request: ${JSON.stringify(request)}`);
            socketInstance.socket.emit('request', request);
            return result;
        } else {
            this._logger.trace(()=>`${accountId}:${request.instanceIndex}: skipping request because it is being sent to ` + `the socket of the wrong instance index, request=${JSON.stringify(request)}`);
            return result;
        }
    }
    // eslint-disable-next-line complexity
    _convertError(data) {
        if (data.error === 'ValidationError') {
            return new ValidationError(data.message, data.details);
        } else if (data.error === 'NotFoundError') {
            return new NotFoundError(data.message);
        } else if (data.error === 'NotSynchronizedError') {
            return new NotSynchronizedError(data.message);
        } else if (data.error === 'TimeoutError') {
            return new TimeoutError(data.message);
        } else if (data.error === 'NotAuthenticatedError') {
            return new NotConnectedError(data.message);
        } else if (data.error === 'ForbiddenError') {
            return new ForbiddenError(data.message);
        } else if (data.error === 'TradeError') {
            return new TradeError(data.message, data.numericCode, data.stringCode);
        } else if (data.error === 'UnauthorizedError') {
            this.close();
            return new UnauthorizedError(data.message);
        } else if (data.error === 'TooManyRequestsError') {
            return new TooManyRequestsError(data.message, data.metadata);
        } else {
            return new InternalError(data.message);
        }
    }
    // eslint-disable-next-line complexity
    _convertIsoTimeToDate(packet) {
        // eslint-disable-next-line guard-for-in
        for(let field in packet){
            let value = packet[field];
            if (typeof value === 'string' && field.match(/time$|Time$/) && !field.match(/brokerTime$|BrokerTime$|timeframe$/)) {
                packet[field] = new Date(value);
            }
            if (Array.isArray(value)) {
                for (let item of value){
                    this._convertIsoTimeToDate(item);
                }
            }
            if (typeof value === 'object') {
                this._convertIsoTimeToDate(value);
            }
        }
        if (packet && packet.timestamps) {
            // eslint-disable-next-line guard-for-in
            for(let field in packet.timestamps){
                packet.timestamps[field] = new Date(packet.timestamps[field]);
            }
        }
        if (packet && packet.type === 'prices') {
            for (let price of packet.prices || []){
                if (price.timestamps) {
                    // eslint-disable-next-line guard-for-in
                    for(let field in price.timestamps){
                        price.timestamps[field] = new Date(price.timestamps[field]);
                    }
                }
            }
        }
    }
    /**
   * MetaTrader symbol price. Contains current price for a symbol (see
   * https://metaapi.cloud/docs/client/models/metatraderSymbolPrice/)
   * @typedef {Object} MetatraderSymbolPrice
   * @property {String} symbol symbol (e.g. a currency pair or an index)
   * @property {Number} bid bid price
   * @property {Number} ask ask price
   * @property {Number} profitTickValue tick value for a profitable position
   * @property {Number} lossTickValue tick value for a losing position
   * @property {Number} [accountCurrencyExchangeRate] current exchange rate of account currency into account base
   * currency (USD if you did not override it)
   * @property {Date} time quote time, in ISO format
   * @property {String} brokerTime time quote time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   */ /**
   * MetaTrader candle
   * @typedef {Object} MetatraderCandle
   * @property {string} symbol symbol (e.g. currency pair or an index)
   * @property {string} timeframe timeframe candle was generated for, e.g. 1h. One of 1m, 2m, 3m, 4m, 5m, 6m, 10m, 12m,
   * 15m, 20m, 30m, 1h, 2h, 3h, 4h, 6h, 8h, 12h, 1d, 1w, 1mn
   * @property {Date} time candle opening time
   * @property {string} brokerTime candle opening time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {number} open open price
   * @property {number} high high price
   * @property {number} low low price
   * @property {number} close close price
   * @property {number} tickVolume tick volume, i.e. number of ticks inside the candle
   * @property {number} spread spread in points
   * @property {number} volume trade volume
   */ /**
   * MetaTrader tick data
   * @typedef {Object} MetatraderTick
   * @property {string} symbol symbol (e.g. a currency pair or an index)
   * @property {Date} time time
   * @property {string} brokerTime time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {number} [bid] bid price
   * @property {number} [ask] ask price
   * @property {number} [last] last deal price
   * @property {number} [volume] volume for the current last deal price
   * @property {string} side is tick a result of buy or sell deal, one of buy or sell
   */ /**
   * MetaTrader order book
   * @typedef {Object} MetatraderBook
   * @property {string} symbol symbol (e.g. a currency pair or an index)
   * @property {Date} time time
   * @property {string} brokerTime time, in broker timezone, YYYY-MM-DD HH:mm:ss.SSS format
   * @property {Array<MetatraderBookEntry>} book list of order book entries
   */ /**
   * MetaTrader order book entry
   * @typedef {Object} MetatraderBookEntry
   * @property {string} type entry type, one of BOOK_TYPE_SELL, BOOK_TYPE_BUY, BOOK_TYPE_SELL_MARKET,
   * BOOK_TYPE_BUY_MARKET
   * @property {number} price price
   * @property {number} volume volume
   */ // eslint-disable-next-line complexity,max-statements
    _processSynchronizationPacket(data) {
        var _this = this;
        return _async_to_generator(function*() {
            try {
                const instanceNumber = data.instanceIndex || 0;
                const socketInstance = _this._getSocketInstanceByAccount(data.accountId, instanceNumber);
                if (data.synchronizationId && socketInstance) {
                    socketInstance.synchronizationThrottler.updateSynchronizationId(data.synchronizationId);
                }
                const region = _this.getAccountRegion(data.accountId);
                const primaryAccountId = _this._accountsByReplicaId[data.accountId];
                let instanceId = primaryAccountId + ':' + region + ':' + instanceNumber + ':' + (data.host || 0);
                let instanceIndex = region + ':' + instanceNumber + ':' + (data.host || 0);
                const isOnlyActiveInstance = ()=>{
                    const activeInstanceIds = Object.keys(_this._connectedHosts).filter((instance)=>instance.startsWith(primaryAccountId + ':' + region + ':' + instanceNumber));
                    return !activeInstanceIds.length || activeInstanceIds.length === 1 && activeInstanceIds[0] === instanceId;
                };
                const cancelDisconnectTimer = ()=>{
                    if (_this._statusTimers[instanceId]) {
                        clearTimeout(_this._statusTimers[instanceId]);
                    }
                };
                const resetDisconnectTimer = ()=>{
                    cancelDisconnectTimer();
                    _this._statusTimers[instanceId] = setTimeout(()=>{
                        _this._logger.warn(`${data.accountId}:${instanceIndex}: timed out waiting for connection status`);
                        if (isOnlyActiveInstance()) {
                            _this._subscriptionManager.onTimeout(data.accountId, 0);
                            _this._subscriptionManager.onTimeout(data.accountId, 1);
                        }
                        _this.queueEvent(primaryAccountId, `${instanceIndex}:onDisconnected`, ()=>onDisconnected(true));
                        clearTimeout(_this._statusTimers[instanceId]);
                    }, 60000);
                };
                // eslint-disable-next-line complexity
                const onDisconnected = function() {
                    var _ref = _async_to_generator(function*(isTimeout = false) {
                        if (_this._connectedHosts[instanceId]) {
                            _this._latencyService.onDisconnected(instanceId);
                            if (isOnlyActiveInstance()) {
                                for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                                    yield _this._processEvent(()=>listener.onDisconnected(instanceIndex), `${primaryAccountId}:${instanceIndex}:onDisconnected`);
                                }
                            }
                            _this._packetOrderer.onStreamClosed(instanceId);
                            if (socketInstance) {
                                socketInstance.synchronizationThrottler.removeIdByParameters(data.accountId, instanceNumber, data.host);
                            }
                            for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                                yield _this._processEvent(()=>listener.onStreamClosed(instanceIndex), `${primaryAccountId}:${instanceIndex}:onStreamClosed`);
                            }
                            delete _this._connectedHosts[instanceId];
                            if (isOnlyActiveInstance() && !isTimeout) {
                                yield _this._subscriptionManager.onDisconnected(data.accountId, 0);
                                yield _this._subscriptionManager.onDisconnected(data.accountId, 1);
                            }
                        }
                    });
                    return function onDisconnected() {
                        return _ref.apply(this, arguments);
                    };
                }();
                if (data.type === 'authenticated') {
                    resetDisconnectTimer();
                    if (!data.sessionId || socketInstance && data.sessionId === socketInstance.sessionId) {
                        _this._latencyService.onConnected(instanceId);
                        _this._connectedHosts[instanceId] = data.host;
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onConnected(instanceIndex, data.replicas), `${primaryAccountId}:${instanceIndex}:onConnected`);
                        }
                        _this._subscriptionManager.cancelSubscribe(data.accountId + ':' + instanceNumber);
                        if (data.replicas === 1) {
                            _this._subscriptionManager.cancelAccount(data.accountId);
                        } else {
                            _this._subscriptionManager.cancelSubscribe(data.accountId + ':' + instanceNumber);
                        }
                    }
                } else if (data.type === 'disconnected') {
                    cancelDisconnectTimer();
                    yield onDisconnected();
                } else if (data.type === 'synchronizationStarted') {
                    _this._updateEvents[instanceId] = [];
                    _this._synchronizationFlags[data.synchronizationId] = {
                        accountId: data.accountId,
                        instanceNumber,
                        specificationsUpdated: data.specificationsHashIndex === undefined,
                        positionsUpdated: data.positionsHashIndex === undefined,
                        ordersUpdated: data.ordersHashIndex === undefined
                    };
                    _this._synchronizationIdByInstance[instanceId] = data.synchronizationId;
                    const specificationsHash = data.specificationsHashIndex !== undefined ? _this._synchronizationHashes[data.synchronizationId] && _this._synchronizationHashes[data.synchronizationId].specificationsHashes[data.specificationsHashIndex] : undefined;
                    const positionsHash = data.positionsHashIndex !== undefined ? _this._synchronizationHashes[data.synchronizationId] && _this._synchronizationHashes[data.synchronizationId].positionsHashes[data.positionsHashIndex] : undefined;
                    const ordersHash = data.ordersHashIndex !== undefined ? _this._synchronizationHashes[data.synchronizationId] && _this._synchronizationHashes[data.synchronizationId].ordersHashes[data.ordersHashIndex] : undefined;
                    delete _this._synchronizationHashes[data.synchronizationId];
                    for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                        yield _this._processEvent(()=>listener.onSynchronizationStarted(instanceIndex, specificationsHash, positionsHash, ordersHash, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onSynchronizationStarted`);
                    }
                } else if (data.type === 'accountInformation') {
                    if (data.synchronizationId && data.synchronizationId !== _this._synchronizationIdByInstance[instanceId]) {
                        return;
                    }
                    if (data.accountInformation) {
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            try {
                                yield _this._processEvent(()=>listener.onAccountInformationUpdated(instanceIndex, data.accountInformation), `${primaryAccountId}:${instanceIndex}:onAccountInformationUpdated`, true);
                                // eslint-disable-next-line max-depth
                                if (_this._synchronizationFlags[data.synchronizationId] && !_this._synchronizationFlags[data.synchronizationId].positionsUpdated) {
                                    yield _this._processEvent(()=>listener.onPositionsSynchronized(instanceIndex, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onPositionsSynchronized`, true);
                                    // eslint-disable-next-line max-depth
                                    if (!_this._synchronizationFlags[data.synchronizationId].ordersUpdated) {
                                        yield _this._processEvent(()=>listener.onPendingOrdersSynchronized(instanceIndex, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onPendingOrdersSynchronized`, true);
                                    }
                                }
                            } catch (err) {
                                _this._logger.error(`${primaryAccountId}:${instanceIndex}: Failed to notify listener ` + 'about accountInformation event', err);
                            }
                        }
                        if (_this._synchronizationFlags[data.synchronizationId] && !_this._synchronizationFlags[data.synchronizationId].positionsUpdated && !_this._synchronizationFlags[data.synchronizationId].ordersUpdated) {
                            delete _this._synchronizationFlags[data.synchronizationId];
                        }
                    }
                } else if (data.type === 'deals') {
                    if (data.synchronizationId && data.synchronizationId !== _this._synchronizationIdByInstance[instanceId]) {
                        return;
                    }
                    for (let deal of data.deals || []){
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onDealAdded(instanceIndex, deal), `${primaryAccountId}:${instanceIndex}:onDealAdded`);
                        }
                    }
                } else if (data.type === 'orders') {
                    if (data.synchronizationId && data.synchronizationId !== _this._synchronizationIdByInstance[instanceId]) {
                        return;
                    }
                    for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                        try {
                            yield _this._processEvent(()=>listener.onPendingOrdersReplaced(instanceIndex, data.orders || []), `${primaryAccountId}:${instanceIndex}:onPendingOrdersReplaced`, true);
                            yield _this._processEvent(()=>listener.onPendingOrdersSynchronized(instanceIndex, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onPendingOrdersSynchronized`, true);
                        } catch (err) {
                            _this._logger.error(`${primaryAccountId}:${instanceIndex}: Failed to notify listener ` + 'about orders event', err);
                        }
                    }
                    if (_this._synchronizationFlags[data.synchronizationId]) {
                        delete _this._synchronizationFlags[data.synchronizationId];
                    }
                } else if (data.type === 'historyOrders') {
                    if (data.synchronizationId && data.synchronizationId !== _this._synchronizationIdByInstance[instanceId]) {
                        return;
                    }
                    for (let historyOrder of data.historyOrders || []){
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onHistoryOrderAdded(instanceIndex, historyOrder), `${primaryAccountId}:${instanceIndex}:onHistoryOrderAdded`);
                        }
                    }
                } else if (data.type === 'positions') {
                    if (data.synchronizationId && data.synchronizationId !== _this._synchronizationIdByInstance[instanceId]) {
                        return;
                    }
                    for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                        try {
                            yield _this._processEvent(()=>listener.onPositionsReplaced(instanceIndex, data.positions || []), `${primaryAccountId}:${instanceIndex}:onPositionsReplaced`, true);
                            yield _this._processEvent(()=>listener.onPositionsSynchronized(instanceIndex, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onPositionsSynchronized`, true);
                            if (_this._synchronizationFlags[data.synchronizationId] && !_this._synchronizationFlags[data.synchronizationId].ordersUpdated) {
                                yield _this._processEvent(()=>listener.onPendingOrdersSynchronized(instanceIndex, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onPendingOrdersSynchronized`, true);
                            }
                        } catch (err) {
                            _this._logger.error(`${primaryAccountId}:${instanceIndex}: Failed to notify listener ` + 'about positions event', err);
                        }
                    }
                    if (_this._synchronizationFlags[data.synchronizationId] && !_this._synchronizationFlags[data.synchronizationId].ordersUpdated) {
                        delete _this._synchronizationFlags[data.synchronizationId];
                    }
                } else if (data.type === 'update') {
                    if (_this._updateEvents[instanceId]) {
                        _this._updateEvents[instanceId].push(data);
                    }
                    if (data.accountInformation) {
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onAccountInformationUpdated(instanceIndex, data.accountInformation), `${primaryAccountId}:${instanceIndex}:onAccountInformationUpdated`);
                        }
                    }
                    const updatedPositions = data.updatedPositions || [];
                    const removedPositionIds = data.removedPositionIds || [];
                    if (updatedPositions.length || removedPositionIds.length) {
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onPositionsUpdated(instanceIndex, updatedPositions, removedPositionIds), `${primaryAccountId}:${instanceIndex}:onPositionsUpdated`);
                        }
                    }
                    for (let position of updatedPositions){
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onPositionUpdated(instanceIndex, position), `${primaryAccountId}:${instanceIndex}:onPositionUpdated`);
                        }
                    }
                    for (let positionId of removedPositionIds){
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onPositionRemoved(instanceIndex, positionId), `${primaryAccountId}:${instanceIndex}:onPositionRemoved`);
                        }
                    }
                    const updatedOrders = data.updatedOrders || [];
                    const completedOrderIds = data.completedOrderIds || [];
                    if (updatedOrders.length || completedOrderIds.length) {
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onPendingOrdersUpdated(instanceIndex, updatedOrders, completedOrderIds), `${primaryAccountId}:${instanceIndex}:onPendingOrdersUpdated`);
                        }
                    }
                    for (let order of updatedOrders){
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onPendingOrderUpdated(instanceIndex, order), `${primaryAccountId}:${instanceIndex}:onPendingOrderUpdated`);
                        }
                    }
                    for (let orderId of completedOrderIds){
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onPendingOrderCompleted(instanceIndex, orderId), `${primaryAccountId}:${instanceIndex}:onPendingOrderCompleted`);
                        }
                    }
                    for (let historyOrder of data.historyOrders || []){
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onHistoryOrderAdded(instanceIndex, historyOrder), `${primaryAccountId}:${instanceIndex}:onHistoryOrderAdded`);
                        }
                    }
                    for (let deal of data.deals || []){
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onDealAdded(instanceIndex, deal), `${primaryAccountId}:${instanceIndex}:onDealAdded`);
                        }
                    }
                    if (data.timestamps) {
                        data.timestamps.clientProcessingFinished = new Date();
                        // eslint-disable-next-line max-depth
                        for (let listener of _this._latencyListeners || []){
                            yield _this._processEvent(()=>listener.onUpdate(data.accountId, data.timestamps), `${primaryAccountId}:${instanceIndex}:onUpdate`);
                        }
                    }
                } else if (data.type === 'dealSynchronizationFinished') {
                    if (data.synchronizationId && data.synchronizationId !== _this._synchronizationIdByInstance[instanceId]) {
                        delete _this._synchronizationIdByInstance[instanceId];
                        return;
                    }
                    _this._latencyService.onDealsSynchronized(instanceId);
                    for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                        if (socketInstance) {
                            socketInstance.synchronizationThrottler.removeSynchronizationId(data.synchronizationId);
                        }
                        yield _this._processEvent(()=>listener.onDealsSynchronized(instanceIndex, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onDealsSynchronized`);
                    }
                    if (_this._updateEvents[instanceId]) {
                        _this._updateEvents[instanceId] = _this._updateEvents[instanceId].map((packet)=>()=>Promise.resolve(_this._processSynchronizationPacket(packet)));
                        if (_this._eventQueues[primaryAccountId]) {
                            _this._eventQueues[primaryAccountId] = _this._updateEvents[instanceId].concat(_this._eventQueues[primaryAccountId]);
                            delete _this._updateEvents[instanceId];
                        } else {
                            _this._eventQueues[primaryAccountId] = _this._updateEvents[instanceId];
                            delete _this._updateEvents[instanceId];
                            _this._callAccountEvents(primaryAccountId);
                        }
                    }
                } else if (data.type === 'orderSynchronizationFinished') {
                    if (data.synchronizationId && data.synchronizationId !== _this._synchronizationIdByInstance[instanceId]) {
                        return;
                    }
                    for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                        yield _this._processEvent(()=>listener.onHistoryOrdersSynchronized(instanceIndex, data.synchronizationId), `${primaryAccountId}:${instanceIndex}:onHistoryOrdersSynchronized`);
                    }
                } else if (data.type === 'status') {
                    if (!_this._connectedHosts[instanceId]) {
                        if (_this._statusTimers[instanceId] && data.authenticated && (_this._subscriptionManager.isDisconnectedRetryMode(data.accountId, instanceNumber) || !_this._subscriptionManager.isAccountSubscribing(data.accountId, instanceNumber))) {
                            _this._subscriptionManager.cancelSubscribe(data.accountId + ':' + instanceNumber);
                            yield new Promise((res)=>setTimeout(res, 10));
                            // eslint-disable-next-line no-console
                            _this._logger.info('it seems like we are not connected to a running API ' + 'server yet, retrying subscription for account ' + instanceId);
                            _this.ensureSubscribe(data.accountId, instanceNumber);
                        }
                    } else {
                        resetDisconnectTimer();
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onBrokerConnectionStatusChanged(instanceIndex, !!data.connected), `${primaryAccountId}:${instanceIndex}:onBrokerConnectionStatusChanged`);
                        }
                        if (data.healthStatus) {
                            // eslint-disable-next-line max-depth
                            for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                                yield _this._processEvent(()=>listener.onHealthStatus(instanceIndex, data.healthStatus), `${primaryAccountId}:${instanceIndex}:onHealthStatus`);
                            }
                        }
                    }
                } else if (data.type === 'downgradeSubscription') {
                    _this._logger.info(`${primaryAccountId}:${instanceIndex}: Market data subscriptions for symbol ` + `${data.symbol} were downgraded by the server due to rate limits. Updated subscriptions: ` + `${JSON.stringify(data.updates)}, removed subscriptions: ${JSON.stringify(data.unsubscriptions)}. ` + 'Please read https://metaapi.cloud/docs/client/rateLimiting/ for more details.');
                    for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                        yield _this._processEvent(()=>listener.onSubscriptionDowngraded(instanceIndex, data.symbol, data.updates, data.unsubscriptions), `${primaryAccountId}:${instanceIndex}:onSubscriptionDowngraded`);
                    }
                } else if (data.type === 'specifications') {
                    if (data.synchronizationId && data.synchronizationId !== _this._synchronizationIdByInstance[instanceId]) {
                        return;
                    }
                    for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                        yield _this._processEvent(()=>listener.onSymbolSpecificationsUpdated(instanceIndex, data.specifications || [], data.removedSymbols || []), `${primaryAccountId}:${instanceIndex}:onSymbolSpecificationsUpdated`);
                    }
                    for (let specification of data.specifications || []){
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onSymbolSpecificationUpdated(instanceIndex, specification), `${primaryAccountId}:${instanceIndex}:onSymbolSpecificationUpdated`);
                        }
                    }
                    for (let removedSymbol of data.removedSymbols || []){
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onSymbolSpecificationRemoved(instanceIndex, removedSymbol), `${primaryAccountId}:${instanceIndex}:onSymbolSpecificationRemoved`);
                        }
                    }
                } else if (data.type === 'prices') {
                    if (data.synchronizationId && data.synchronizationId !== _this._synchronizationIdByInstance[instanceId]) {
                        return;
                    }
                    let prices = data.prices || [];
                    let candles = data.candles || [];
                    let ticks = data.ticks || [];
                    let books = data.books || [];
                    for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                        if (prices.length) {
                            yield _this._processEvent(()=>listener.onSymbolPricesUpdated(instanceIndex, prices, data.equity, data.margin, data.freeMargin, data.marginLevel, data.accountCurrencyExchangeRate), `${primaryAccountId}:${instanceIndex}:onSymbolPricesUpdated`);
                        }
                        if (candles.length) {
                            yield _this._processEvent(()=>listener.onCandlesUpdated(instanceIndex, candles, data.equity, data.margin, data.freeMargin, data.marginLevel, data.accountCurrencyExchangeRate), `${primaryAccountId}:${instanceIndex}:onCandlesUpdated`);
                        }
                        if (ticks.length) {
                            yield _this._processEvent(()=>listener.onTicksUpdated(instanceIndex, ticks, data.equity, data.margin, data.freeMargin, data.marginLevel, data.accountCurrencyExchangeRate), `${primaryAccountId}:${instanceIndex}:onTicksUpdated`);
                        }
                        if (books.length) {
                            yield _this._processEvent(()=>listener.onBooksUpdated(instanceIndex, books, data.equity, data.margin, data.freeMargin, data.marginLevel, data.accountCurrencyExchangeRate), `${primaryAccountId}:${instanceIndex}:onBooksUpdated`);
                        }
                    }
                    for (let price of prices){
                        for (let listener of _this._synchronizationListeners[primaryAccountId] || []){
                            yield _this._processEvent(()=>listener.onSymbolPriceUpdated(instanceIndex, price), `${primaryAccountId}:${instanceIndex}:onSymbolPriceUpdated`);
                        }
                    }
                    for (let price of prices){
                        if (price.timestamps) {
                            price.timestamps.clientProcessingFinished = new Date();
                            // eslint-disable-next-line max-depth
                            for (let listener of _this._latencyListeners || []){
                                yield _this._processEvent(()=>listener.onSymbolPrice(data.accountId, price.symbol, price.timestamps), `${primaryAccountId}:${instanceIndex}:onSymbolPrice`);
                            }
                        }
                    }
                }
            } catch (err) {
                _this._logger.error('Failed to process incoming synchronization packet', err);
            }
        })();
    }
    _processEvent(callable, label, throwError) {
        var _this = this;
        return _async_to_generator(function*() {
            const startTime = Date.now();
            let isLongEvent = false;
            let isEventDone = false;
            const checkLongEvent = function() {
                var _ref = _async_to_generator(function*() {
                    yield new Promise((res)=>setTimeout(res, 1000));
                    if (!isEventDone) {
                        isLongEvent = true;
                        _this._logger.warn(`${label}: event is taking more than 1 second to process`);
                    }
                });
                return function checkLongEvent() {
                    return _ref.apply(this, arguments);
                };
            }();
            checkLongEvent();
            try {
                yield callable();
            } catch (err) {
                if (throwError) {
                    throw err;
                }
                _this._logger.error(`${label}: event failed with error`, err);
            }
            isEventDone = true;
            if (isLongEvent) {
                _this._logger.warn(`${label}: finished in ${Math.floor((Date.now() - startTime) / 1000)} seconds`);
            }
        })();
    }
    _fireReconnected(instanceNumber, socketInstanceIndex, region) {
        var _this = this;
        return _async_to_generator(function*() {
            try {
                const reconnectListeners = [];
                for (let listener of _this._reconnectListeners){
                    if (_this._socketInstancesByAccounts[instanceNumber][listener.accountId] === socketInstanceIndex && _this.getAccountRegion(listener.accountId) === region) {
                        reconnectListeners.push(listener);
                    }
                }
                Object.keys(_this._synchronizationFlags).forEach((synchronizationId)=>{
                    const accountId = _this._synchronizationFlags[synchronizationId].accountId;
                    if (_this._socketInstancesByAccounts[instanceNumber][accountId] === socketInstanceIndex && _this._synchronizationFlags[synchronizationId].instanceNumber === instanceNumber && _this._regionsByAccounts[accountId] && _this._regionsByAccounts[accountId].region === region) {
                        delete _this._synchronizationFlags[synchronizationId];
                    }
                });
                const reconnectAccountIds = reconnectListeners.map((listener)=>listener.accountId);
                _this._subscriptionManager.onReconnected(instanceNumber, socketInstanceIndex, reconnectAccountIds);
                _this._packetOrderer.onReconnected(reconnectAccountIds);
                for (let listener of reconnectListeners){
                    Promise.resolve(listener.listener.onReconnected(region, instanceNumber)).catch((err)=>_this._logger.error('Failed to notify reconnect listener', err));
                }
            } catch (err) {
                _this._logger.error('Failed to process reconnected event', err);
            }
        })();
    }
    _getSocketInstanceByAccount(accountId, instanceNumber) {
        const region = this.getAccountRegion(accountId);
        return this._socketInstances[region][instanceNumber][this._socketInstancesByAccounts[instanceNumber][accountId]];
    }
    getUrlSettings(instanceNumber, region) {
        var _this = this;
        return _async_to_generator(function*() {
            if (_this._url) {
                return {
                    url: _this._url,
                    isSharedClientApi: true
                };
            }
            const urlSettings = yield _this._domainClient.getSettings();
            const getUrl = (hostname)=>`https://${hostname}.${region}-${String.fromCharCode(97 + Number(instanceNumber))}.${urlSettings.domain}`;
            let url;
            if (_this._useSharedClientApi) {
                url = getUrl(_this._hostname);
            } else {
                url = getUrl(urlSettings.hostname);
            }
            const isSharedClientApi = url === getUrl(_this._hostname);
            return {
                url,
                isSharedClientApi
            };
        })();
    }
    // eslint-disable-next-line complexity
    _getServerUrl(instanceNumber, socketInstanceIndex, region) {
        var _this = this;
        return _async_to_generator(function*() {
            if (_this._url) {
                return _this._url;
            }
            while(_this.socketInstances[region][instanceNumber][socketInstanceIndex].connected){
                try {
                    const urlSettings = yield _this.getUrlSettings(instanceNumber, region);
                    const url = urlSettings.url;
                    const isSharedClientApi = urlSettings.isSharedClientApi;
                    let logMessage = 'Connecting MetaApi websocket client to the MetaApi server ' + `via ${url} ${isSharedClientApi ? 'shared' : 'dedicated'} server.`;
                    if (_this._firstConnect && !isSharedClientApi) {
                        logMessage += ' Please note that it can take up to 3 minutes for your dedicated server to start for the ' + 'first time. During this time it is OK if you see some connection errors.';
                        _this._firstConnect = false;
                    }
                    _this._logger.info(logMessage);
                    return url;
                } catch (err) {
                    _this._logger.error('Failed to retrieve server URL', err);
                    yield new Promise((res)=>setTimeout(res, 1000));
                }
            }
        })();
    }
    _throttleRequest(type, accountId, instanceNumber, timeInMs) {
        this._lastRequestsTime[instanceNumber] = this._lastRequestsTime[instanceNumber] || {};
        this._lastRequestsTime[instanceNumber][type] = this._lastRequestsTime[instanceNumber][type] || {};
        let lastTime = this._lastRequestsTime[instanceNumber][type][accountId];
        if (!lastTime || lastTime < Date.now() - timeInMs) {
            this._lastRequestsTime[instanceNumber][type][accountId] = Date.now();
            return !!lastTime;
        }
        return false;
    }
    _refreshAccountRegion(accountId) {
        if (this._regionsByAccounts[accountId]) {
            this._regionsByAccounts[accountId].lastUsed = Date.now();
        }
    }
    //eslint-disable-next-line complexity
    _createSocketInstanceByAccount(accountId, instanceNumber) {
        var _this = this;
        return _async_to_generator(function*() {
            const region = _this.getAccountRegion(accountId);
            if (_this._socketInstancesByAccounts[instanceNumber][accountId] === undefined) {
                let socketInstanceIndex = null;
                while(_this._subscribeLock && (new Date(_this._subscribeLock.recommendedRetryTime).getTime() > Date.now() && _this.subscribedAccountIds(instanceNumber, undefined, region).length < _this._subscribeLock.lockedAtAccounts || new Date(_this._subscribeLock.lockedAtTime).getTime() + _this._subscribeCooldownInSeconds * 1000 > Date.now() && _this.subscribedAccountIds(instanceNumber, undefined, region).length >= _this._subscribeLock.lockedAtAccounts)){
                    yield new Promise((res)=>setTimeout(res, 1000));
                }
                for(let index = 0; index < _this._socketInstances[region][instanceNumber].length; index++){
                    const accountCounter = _this.getAssignedAccounts(instanceNumber, index, region).length;
                    const instance = _this.socketInstances[region][instanceNumber][index];
                    if (instance.subscribeLock) {
                        if (instance.subscribeLock.type === 'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_USER_PER_SERVER' && (new Date(instance.subscribeLock.recommendedRetryTime).getTime() > Date.now() || _this.subscribedAccountIds(instanceNumber, index, region).length >= instance.subscribeLock.lockedAtAccounts)) {
                            continue;
                        }
                        if (instance.subscribeLock.type === 'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_SERVER' && new Date(instance.subscribeLock.recommendedRetryTime).getTime() > Date.now() && _this.subscribedAccountIds(instanceNumber, index, region).length >= instance.subscribeLock.lockedAtAccounts) {
                            continue;
                        }
                    }
                    if (accountCounter < _this._maxAccountsPerInstance) {
                        socketInstanceIndex = index;
                        break;
                    }
                }
                if (socketInstanceIndex === null) {
                    socketInstanceIndex = _this._socketInstances[region][instanceNumber].length;
                    yield _this.connect(instanceNumber, region);
                }
                _this._socketInstancesByAccounts[instanceNumber][accountId] = socketInstanceIndex;
            }
        })();
    }
    _clearAccountCacheJob() {
        const date = Date.now();
        Object.keys(this._regionsByAccounts).forEach((replicaId)=>{
            const data = this._regionsByAccounts[replicaId];
            if (data && data.connections === 0 && date - data.lastUsed > 2 * 60 * 60 * 1000) {
                const primaryAccountId = this._accountsByReplicaId[replicaId];
                const replicas = Object.values(this._accountReplicas[primaryAccountId] || {});
                replicas.forEach((replica)=>{
                    delete this._accountsByReplicaId[replica];
                    delete this._regionsByAccounts[replica];
                });
                delete this._accountReplicas[primaryAccountId];
                this._logger.debug(`${primaryAccountId}: removed expired account replicas data`);
            }
        });
    }
    _clearInactiveSyncDataJob() {
        const date = Date.now();
        Object.keys(this._synchronizationHashes).keys((synchronizationId)=>{
            if (this._synchronizationHashes[synchronizationId].lastUpdated < date - 30 * 60 * 1000) {
                delete this._synchronizationHashes[synchronizationId];
            }
        });
    }
    /**
   * @typedef MetaApiWebsocketClientOptions MetaApi websocket client options
   * @property {string} [region] region to connect
   * @property {Boolean} [disableInternalJobs] whether to not run internal interval jobs. Used for tests only
   */ /**
   * Constructs MetaApi websocket API client instance
   * @param {MetaApi} metaApi metaApi instance
   * @param {DomainClient} domainClient domain client
   * @param {String} token authorization token
   * @param {MetaApiWebsocketClientOptions} opts websocket client options
   */ // eslint-disable-next-line complexity,max-statements
    constructor(metaApi, domainClient, token, opts){
        const validator = new OptionsValidator();
        opts = opts || {};
        opts.packetOrderingTimeout = validator.validateNonZero(opts.packetOrderingTimeout, 60, 'packetOrderingTimeout');
        opts.synchronizationThrottler = opts.synchronizationThrottler || {};
        this._domainClient = domainClient;
        this._application = opts.application || 'MetaApi';
        this._domain = opts.domain || 'agiliumtrade.agiliumtrade.ai';
        this._region = opts.region;
        this._hostname = 'mt-client-api-v1';
        this._metaApi = metaApi;
        this._url = null;
        this._requestTimeout = validator.validateNonZero(opts.requestTimeout, 60, 'requestTimeout') * 1000;
        this._connectTimeout = validator.validateNonZero(opts.connectTimeout, 60, 'connectTimeout') * 1000;
        const retryOpts = opts.retryOpts || {};
        this._retries = validator.validateNumber(retryOpts.retries, 5, 'retryOpts.retries');
        this._minRetryDelayInSeconds = validator.validateNonZero(retryOpts.minDelayInSeconds, 1, 'retryOpts.minDelayInSeconds');
        this._maxRetryDelayInSeconds = validator.validateNonZero(retryOpts.maxDelayInSeconds, 30, 'retryOpts.maxDelayInSeconds');
        this._maxAccountsPerInstance = 100;
        this._subscribeCooldownInSeconds = validator.validateNonZero(retryOpts.subscribeCooldownInSeconds, 600, 'retryOpts.subscribeCooldownInSeconds');
        this._sequentialEventProcessing = true;
        this._useSharedClientApi = validator.validateBoolean(opts.useSharedClientApi, false, 'useSharedClientApi');
        this._unsubscribeThrottlingInterval = validator.validateNonZero(opts.unsubscribeThrottlingIntervalInSeconds, 10, 'unsubscribeThrottlingIntervalInSeconds') * 1000;
        this._socketMinimumReconnectTimeout = 500;
        this._latencyService = new LatencyService(this, token, this._connectTimeout);
        this._token = token;
        this._synchronizationListeners = {};
        this._latencyListeners = [];
        this._reconnectListeners = [];
        this._connectedHosts = {};
        this._socketInstances = {};
        this._socketInstancesByAccounts = {};
        this._regionsByAccounts = {};
        this._accountsByReplicaId = {};
        this._accountReplicas = {};
        this._synchronizationThrottlerOpts = opts.synchronizationThrottler;
        this._subscriptionManager = new SubscriptionManager(this, metaApi);
        this._statusTimers = {};
        this._eventQueues = {};
        this._synchronizationFlags = {};
        this._synchronizationIdByInstance = {};
        this._subscribeLock = null;
        this._firstConnect = true;
        this._lastRequestsTime = {};
        this._packetOrderer = new PacketOrderer(this, opts.packetOrderingTimeout);
        this._packetOrderer.start();
        this._synchronizationHashes = {};
        this._updateEvents = {};
        if (opts.packetLogger && opts.packetLogger.enabled) {
            this._packetLogger = new PacketLogger(opts.packetLogger);
            this._packetLogger.start();
        }
        this._logger = LoggerManager.getLogger('MetaApiWebsocketClient');
        if (!opts.disableInternalJobs) {
            this._clearAccountCacheInterval = setInterval(this._clearAccountCacheJob.bind(this), 30 * 60 * 1000);
            this._clearInactiveSyncDataJob = setInterval(this._clearInactiveSyncDataJob.bind(this), 5 * 60 * 1000);
        }
    }
};
/**
 * MetaApi websocket API client (see https://metaapi.cloud/docs/client/websocket/overview/)
 */ export { MetaApiWebsocketClient as default };

//# sourceMappingURL=data:application/json;base64,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