'use strict';
function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
function _define_property(obj, key, value) {
    if (key in obj) {
        Object.defineProperty(obj, key, {
            value: value,
            enumerable: true,
            configurable: true,
            writable: true
        });
    } else {
        obj[key] = value;
    }
    return obj;
}
function _object_spread(target) {
    for(var i = 1; i < arguments.length; i++){
        var source = arguments[i] != null ? arguments[i] : {};
        var ownKeys = Object.keys(source);
        if (typeof Object.getOwnPropertySymbols === "function") {
            ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function(sym) {
                return Object.getOwnPropertyDescriptor(source, sym).enumerable;
            }));
        }
        ownKeys.forEach(function(key) {
            _define_property(target, key, source[key]);
        });
    }
    return target;
}
import axios from 'axios';
import { UnauthorizedError, ForbiddenError, ApiError, ValidationError, InternalError, NotFoundError, TooManyRequestsError, ConflictError } from './errorHandler';
import OptionsValidator from './optionsValidator';
import TimeoutError from './timeoutError';
import LoggerManager from '../logger';
let HttpClient = class HttpClient {
    /**
   * Performs a request. Response errors are returned as ApiError or subclasses.
   * @param {Object} options request options
   * @returns {Object|String|any} request result
   */ request(options, type = '', retryCounter = 0, endTime = Date.now() + this._maxRetryDelay * this._retries, isLongRunning = false) {
        var _this = this;
        return _async_to_generator(function*() {
            options.timeout = _this._timeout;
            let retryAfterSeconds = 0;
            options.callback = (e, res)=>{
                if (res && res.status === 202) {
                    retryAfterSeconds = res.headers['retry-after'];
                    if (isNaN(retryAfterSeconds)) {
                        retryAfterSeconds = Math.max((new Date(retryAfterSeconds).getTime() - Date.now()) / 1000, 1);
                    }
                    if (!isLongRunning) {
                        endTime = Date.now() + _this._longRunningRequestTimeout;
                        isLongRunning = true;
                    }
                }
            };
            let body;
            try {
                const response = yield _this._makeRequest(options);
                options.callback(null, response);
                body = response && response.data || undefined;
            } catch (err) {
                retryCounter = yield _this._handleError(err, type, retryCounter, endTime);
                return _this.request(options, type, retryCounter, endTime);
            }
            if (retryAfterSeconds) {
                if (body && body.message) {
                    _this._logger.info(`Retrying request in ${Math.floor(retryAfterSeconds)} seconds because request ` + 'returned message:', body.message);
                }
                yield _this._handleRetry(endTime, retryAfterSeconds * 1000);
                body = yield _this.request(options, type, retryCounter, endTime, isLongRunning);
            }
            return body;
        })();
    }
    _makeRequest(options) {
        return axios(_object_spread({
            transitional: {
                clarifyTimeoutError: true
            }
        }, options));
    }
    _wait(pause) {
        return _async_to_generator(function*() {
            yield new Promise((res)=>setTimeout(res, pause));
        })();
    }
    _handleRetry(endTime, retryAfter) {
        var _this = this;
        return _async_to_generator(function*() {
            if (endTime > Date.now() + retryAfter) {
                yield _this._wait(retryAfter);
            } else {
                throw new TimeoutError('Timed out waiting for the response');
            }
        })();
    }
    _handleError(err, type, retryCounter, endTime) {
        var _this = this;
        return _async_to_generator(function*() {
            const error = _this._convertError(err);
            if ([
                'ConflictError',
                'InternalError',
                'ApiError',
                'TimeoutError'
            ].includes(error.name) && retryCounter < _this._retries) {
                const pause = Math.min(Math.pow(2, retryCounter) * _this._minRetryDelay, _this._maxRetryDelay);
                yield _this._wait(pause);
                return retryCounter + 1;
            } else if (error.name === 'TooManyRequestsError') {
                const retryTime = Date.parse(error.metadata.recommendedRetryTime);
                if (retryTime < endTime) {
                    _this._logger.debug(`${type} request has failed with TooManyRequestsError (HTTP status code 429). ` + `Will retry request in ${Math.ceil((retryTime - Date.now()) / 1000)} seconds`);
                    yield _this._wait(retryTime - Date.now());
                    return retryCounter;
                }
            }
            throw error;
        })();
    }
    // eslint-disable-next-line complexity
    _convertError(err) {
        var _err_config;
        const errorResponse = err.response || {};
        const errorData = errorResponse.data || {};
        const status = errorResponse.status || err.status;
        const url = err === null || err === void 0 ? void 0 : (_err_config = err.config) === null || _err_config === void 0 ? void 0 : _err_config.url;
        const errMsg = errorData.message || err.message;
        const errMsgDefault = errorData.message || err.code || err.message;
        switch(status){
            case 400:
                return new ValidationError(errMsg, errorData.details || err.details, url);
            case 401:
                return new UnauthorizedError(errMsg, url);
            case 403:
                return new ForbiddenError(errMsg, url);
            case 404:
                return new NotFoundError(errMsg, url);
            case 409:
                return new ConflictError(errMsg, url);
            case 429:
                return new TooManyRequestsError(errMsg, errorData.metadata || err.metadata, url);
            case 500:
                return new InternalError(errMsg, url);
            default:
                return new ApiError(ApiError, errMsgDefault, status, url);
        }
    }
    /**
   * @typedef {Object} RetryOptions retry options
   * @property {Number} [retries] the number of attempts to retry failed request, default 5
   * @property {Number} [minDelayInSeconds] minimum delay in seconds before retrying, default 1
   * @property {Number} [maxDelayInSeconds] maximum delay in seconds before retrying, default 30
   * @property {Number} [longRunningRequestTimeoutInMinutes] timeout in minutes for long running requests, default 10
   * @property {Number} [subscribeCooldownInSeconds] time to disable new subscriptions for
   */ /**
   * Constructs HttpClient class instance
   * @param {Number} timeout request timeout in seconds
   * @param {RetryOptions} [retryOpts] retry options
   */ constructor(timeout = 60, retryOpts = {}){
        const validator = new OptionsValidator();
        this._timeout = timeout * 1000;
        this._retries = validator.validateNumber(retryOpts.retries, 5, 'retryOpts.retries');
        this._minRetryDelay = validator.validateNonZero(retryOpts.minDelayInSeconds, 1, 'retryOpts.minDelayInSeconds') * 1000;
        this._maxRetryDelay = validator.validateNonZero(retryOpts.maxDelayInSeconds, 30, 'retryOpts.maxDelayInSeconds') * 1000;
        this._longRunningRequestTimeout = validator.validateNumber(retryOpts.longRunningRequestTimeoutInMinutes, 10, 'retryOpts.longRunningRequestTimeoutInMinutes') * 60 * 1000;
        this._logger = LoggerManager.getLogger('HttpClient');
    }
};
/**
 * HTTP client library based on axios
 *
*/ export { HttpClient as default };
/**
 * HTTP client service mock for tests
 */ export class HttpClientMock extends HttpClient {
    _makeRequest() {
        return this._requestFn.apply(this, arguments);
    }
    /**
   * Constructs HTTP client mock
   * @param {Function(options:Object):Promise} requestFn mocked request function
   * @param {Number} timeout request timeout in seconds
   * @param {RetryOptions} retryOpts retry options
   */ constructor(requestFn, timeout, retryOpts){
        super(timeout, retryOpts);
        this._requestFn = requestFn;
    }
}

//# sourceMappingURL=data:application/json;base64,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