"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return PacketOrderer;
    }
});
let PacketOrderer = class PacketOrderer {
    /**
   * Initializes the packet orderer
   */ start() {
        this._sequenceNumberByInstance = {};
        this._lastSessionStartTimestamp = {};
        this._packetsByInstance = {};
        if (!this._outOfOrderInterval) {
            this._outOfOrderInterval = setInterval(()=>this._emitOutOfOrderEvents(), 1000);
        }
    }
    /**
   * Deinitialized the packet orderer
   */ stop() {
        clearInterval(this._outOfOrderInterval);
    }
    /**
   * Processes the packet and resolves in the order of packet sequence number
   * @param {Object} packet packet to process
   * @return {Array<Object>} ordered packets when the packets are ready to be processed in order
   */ // eslint-disable-next-line complexity
    restoreOrder(packet) {
        let instanceId = packet.accountId + ":" + (packet.instanceIndex || 0) + ":" + (packet.host || 0);
        if (packet.sequenceNumber === undefined) {
            return [
                packet
            ];
        }
        if (packet.type === "synchronizationStarted" && packet.synchronizationId && (!this._lastSessionStartTimestamp[instanceId] || this._lastSessionStartTimestamp[instanceId] < packet.sequenceTimestamp)) {
            // synchronization packet sequence just started
            this._isOutOfOrderEmitted[instanceId] = false;
            this._sequenceNumberByInstance[instanceId] = packet.sequenceNumber;
            this._lastSessionStartTimestamp[instanceId] = packet.sequenceTimestamp;
            this._packetsByInstance[instanceId] = (this._packetsByInstance[instanceId] || []).filter((waitPacket)=>waitPacket.packet.sequenceTimestamp >= packet.sequenceTimestamp);
            return [
                packet
            ].concat(this._findNextPacketsFromWaitList(instanceId));
        } else if (packet.sequenceTimestamp < this._lastSessionStartTimestamp[instanceId]) {
            // filter out previous packets
            return [];
        } else if (packet.sequenceNumber === this._sequenceNumberByInstance[instanceId]) {
            // let the duplicate s/n packet to pass through
            return [
                packet
            ];
        } else if (packet.sequenceNumber === this._sequenceNumberByInstance[instanceId] + 1) {
            // in-order packet was received
            this._sequenceNumberByInstance[instanceId]++;
            this._lastSessionStartTimestamp[instanceId] = packet.sequenceTimestamp || this._lastSessionStartTimestamp[instanceId];
            return [
                packet
            ].concat(this._findNextPacketsFromWaitList(instanceId));
        } else {
            // out-of-order packet was received, add it to the wait list
            this._packetsByInstance[instanceId] = this._packetsByInstance[instanceId] || [];
            let waitList = this._packetsByInstance[instanceId];
            waitList.push({
                instanceId,
                accountId: packet.accountId,
                instanceIndex: packet.instanceIndex || 0,
                sequenceNumber: packet.sequenceNumber,
                packet: packet,
                receivedAt: new Date()
            });
            waitList.sort((e1, e2)=>e1.sequenceNumber - e2.sequenceNumber);
            while(waitList.length > this._waitListSizeLimit){
                waitList.shift();
            }
            return [];
        }
    }
    /**
   * Resets state for instance id
   * @param {String} instanceId instance id to reset state for
   */ onStreamClosed(instanceId) {
        delete this._packetsByInstance[instanceId];
        delete this._lastSessionStartTimestamp[instanceId];
        delete this._sequenceNumberByInstance[instanceId];
    }
    /**
   * Resets state for specified accounts on reconnect
   * @param {String[]} reconnectAccountIds reconnected account ids
   */ onReconnected(reconnectAccountIds) {
        Object.keys(this._packetsByInstance).forEach((instanceId)=>{
            if (reconnectAccountIds.includes(this._getAccountIdFromInstance(instanceId))) {
                delete this._packetsByInstance[instanceId];
            }
        });
        Object.keys(this._lastSessionStartTimestamp).forEach((instanceId)=>{
            if (reconnectAccountIds.includes(this._getAccountIdFromInstance(instanceId))) {
                delete this._lastSessionStartTimestamp[instanceId];
            }
        });
        Object.keys(this._sequenceNumberByInstance).forEach((instanceId)=>{
            if (reconnectAccountIds.includes(this._getAccountIdFromInstance(instanceId))) {
                delete this._sequenceNumberByInstance[instanceId];
            }
        });
    }
    _getAccountIdFromInstance(instanceId) {
        return instanceId.split(":")[0];
    }
    // eslint-disable-next-line complexity
    _findNextPacketsFromWaitList(instanceId) {
        let result = [];
        let waitList = this._packetsByInstance[instanceId] || [];
        while(waitList.length && ([
            this._sequenceNumberByInstance[instanceId],
            this._sequenceNumberByInstance[instanceId] + 1
        ].includes(waitList[0].sequenceNumber) || waitList[0].packet.sequenceTimestamp < this._lastSessionStartTimestamp[instanceId])){
            if (waitList[0].packet.sequenceTimestamp >= this._lastSessionStartTimestamp[instanceId]) {
                result.push(waitList[0].packet);
                if (waitList[0].packet.sequenceNumber === this._sequenceNumberByInstance[instanceId] + 1) {
                    this._sequenceNumberByInstance[instanceId]++;
                    this._lastSessionStartTimestamp[instanceId] = waitList[0].packet.sequenceTimestamp || this._lastSessionStartTimestamp[instanceId];
                }
            }
            waitList.splice(0, 1);
        }
        if (!waitList.length) {
            delete this._packetsByInstance[instanceId];
        }
        return result;
    }
    _emitOutOfOrderEvents() {
        for (let waitList of Object.values(this._packetsByInstance)){
            if (waitList.length && waitList[0].receivedAt.getTime() + this._orderingTimeoutInSeconds * 1000 < Date.now()) {
                const instanceId = waitList[0].instanceId;
                if (!this._isOutOfOrderEmitted[instanceId]) {
                    this._isOutOfOrderEmitted[instanceId] = true;
                    // Do not emit onOutOfOrderPacket for packets that come before synchronizationStarted
                    if (this._sequenceNumberByInstance[instanceId] !== undefined) {
                        this._outOfOrderListener.onOutOfOrderPacket(waitList[0].accountId, waitList[0].instanceIndex, this._sequenceNumberByInstance[instanceId] + 1, waitList[0].sequenceNumber, waitList[0].packet, waitList[0].receivedAt);
                    }
                }
            }
        }
    }
    /**
   * Constructs the class
   * @param {Function} outOfOrderListener function which will receive out of order packet events
   * @param {Number} orderingTimeoutInSeconds packet ordering timeout
   */ constructor(outOfOrderListener, orderingTimeoutInSeconds){
        this._outOfOrderListener = outOfOrderListener;
        this._orderingTimeoutInSeconds = orderingTimeoutInSeconds;
        this._isOutOfOrderEmitted = {};
        this._waitListSizeLimit = 100;
        this._sequenceNumberByInstance = {};
        this._lastSessionStartTimestamp = {};
        this._packetsByInstance = {};
    }
};

//# sourceMappingURL=data:application/json;base64,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