import randomstring from 'randomstring';

/**
 * CopyFactory client for strategy signal requests
 */
export default class StrategySignalClient {

  /**
   * Constructs CopyFactory strategy signal client instance
   * @param {string} accountId strategy provider id
   * @param {string} strategyId strategy id
   * @param {Object} host host data
   * @param {DomainClient} domainClient domain client
   */
  constructor(accountId, strategyId, host, domainClient) {
    this._accountId = accountId;
    this._strategyId = strategyId;
    this._domainClient = domainClient;
    this._host = host;
  }

  /**
   * Generates random signal id
   * @return {String} signal id
   */
  generateSignalId() {
    return randomstring.generate(8);
  }

  /**
   * CopyFactory external signal update payload
   * @typedef {Object} CopyFactoryExternalSignalUpdate
   * @property {String} symbol trade symbol
   * @property {String} type trade type (one of POSITION_TYPE_BUY, POSITION_TYPE_SELL, ORDER_TYPE_BUY_LIMIT, ORDER_TYPE_SELL_LIMIT, 
   * ORDER_TYPE_BUY_STOP, ORDER_TYPE_SELL_STOP)
   * @property {Date} time time the signal was emitted at
   * @property {Date} [updateTime] last time of the signal update
   * @property {Number} volume volume traded
   * @property {Number} [magic] expert advisor id
   * @property {Number} [stopLoss] stop loss price
   * @property {Number} [takeProfit] take profit price
   * @property {Number} [openPrice] pending order open price
   */

  /**
   * CopyFactory external signal
   * @typedef {CopyFactoryExternalSignalUpdate} CopyFactoryExternalSignal
   * @property {String} id external signal id
   */

  /**
   * CopyFactory external signal remove payload
   * @typedef {Object} CopyFactoryExternalSignalRemove
   * @property {Date} time the time signal was removed (closed) at
   */

  /**
   * Returns active external signals of a strategy. Requires access to
   * copyfactory-api:rest:public:external-signals:getSignals method which is included into reader role.
   * Requires access to strategy, account resources.
   * @param {String} strategyId strategy id
   * @returns {Promise<Array<CopyFactoryExternalSignal>>}
   */
  getExternalSignals() {
    const opts = {
      url: `/users/current/strategies/${this._strategyId}/external-signals`,
      method: 'GET',
      headers: {
        'auth-token': this._domainClient.token
      },
      json: true
    };

    return this._domainClient.requestSignal(opts, this._host, this._accountId);
  }

  /**
   * Updates external signal for a strategy. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/updateExternalSignal/
   * @param {String} signalId external signal id (should be 8 alphanumerical symbols)
   * @param {CopyFactoryExternalSignalUpdate} signal signal update payload
   * @return {Promise} promise which resolves when the external signal is updated
   */
  updateExternalSignal(signalId, signal) {
    const opts = {
      url: `/users/current/strategies/${this._strategyId}/external-signals/${signalId}`,
      method: 'PUT',
      headers: {
        'auth-token': this._domainClient.token
      },
      data: signal,
      json: true
    };
    return this._domainClient.requestSignal(opts, this._host, this._accountId);
  }

  /**
   * CopyFactory external signal remove payload
   * @typedef {Object} CopyFactoryExternalSignalRemove
   * @property {Date} time the time signal was removed (closed) at
   */

  /**
   * Removes (closes) external signal for a strategy. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/removeExternalSignal/
   * @param {String} signalId external signal id
   * @param {CopyFactoryExternalSignalRemove} signal signal removal payload
   * @return {Promise} promise which resolves when the external signal is removed
   */
  removeExternalSignal(signalId, signal) {
    const opts = {
      url: `/users/current/strategies/${this._strategyId}/external-signals/${signalId}/remove`,
      method: 'POST',
      headers: {
        'auth-token': this._domainClient.token
      },
      data: signal,
      json: true
    };
    return this._domainClient.requestSignal(opts, this._host, this._accountId);
  }

}