/**
 * 
 */ 'use strict';
function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
import MetaApiClient from '../../metaapi.client';
import randomstring from 'randomstring';
import LoggerManager from '../../../logger';
let TransactionListenerManager = class TransactionListenerManager extends MetaApiClient {
    /**
   * Returns the dictionary of strategy transaction listeners
   * @returns {Object} dictionary of strategy transaction listeners
   */ get strategyTransactionListeners() {
        return this._strategyTransactionListeners;
    }
    /**
   * Returns the dictionary of subscriber transaction listeners
   * @returns {Object} dictionary of subscriber transaction listeners
   */ get subscriberTransactionListeners() {
        return this._subscriberTransactionListeners;
    }
    /**
   * Adds a strategy transaction listener
   * @param {UserTransactionListener} listener user transaction listener
   * @param {String} strategyId strategy id
   * @param {Date} [startTime] transaction search start time
   * @returns {String} strategy transaction listener id
   */ addStrategyTransactionListener(listener, strategyId, startTime) {
        const listenerId = randomstring.generate(10);
        this._strategyTransactionListeners[listenerId] = listener;
        this._startStrategyTransactionStreamJob(listenerId, listener, strategyId, startTime);
        return listenerId;
    }
    /**
   * Adds a subscriber transaction listener
   * @param {UserTransactionListener} listener user transaction listener
   * @param {String} subscriberId subscriber id
   * @param {Date} [startTime] transaction search start time
   * @returns {String} subscriber transaction listener id
   */ addSubscriberTransactionListener(listener, subscriberId, startTime) {
        const listenerId = randomstring.generate(10);
        this._subscriberTransactionListeners[listenerId] = listener;
        this._startSubscriberTransactionStreamJob(listenerId, listener, subscriberId, startTime);
        return listenerId;
    }
    /**
   * Removes strategy transaction listener by id
   * @param {String} listenerId listener id 
   */ removeStrategyTransactionListener(listenerId) {
        delete this._strategyTransactionListeners[listenerId];
    }
    /**
   * Removes subscriber transaction listener by id
   * @param {String} listenerId listener id 
   */ removeSubscriberTransactionListener(listenerId) {
        delete this._subscriberTransactionListeners[listenerId];
    }
    _startStrategyTransactionStreamJob(listenerId, listener, strategyId, startTime) {
        var _this = this;
        return _async_to_generator(function*() {
            let throttleTime = _this._errorThrottleTime;
            while(_this._strategyTransactionListeners[listenerId]){
                const opts = {
                    url: `/users/current/strategies/${strategyId}/transactions/stream`,
                    method: 'GET',
                    params: {
                        startTime,
                        limit: 1000
                    },
                    headers: {
                        'auth-token': _this._token
                    },
                    json: true
                };
                try {
                    const packets = yield _this._domainClient.requestCopyFactory(opts, true);
                    yield listener.onTransaction(packets);
                    throttleTime = _this._errorThrottleTime;
                    if (_this._strategyTransactionListeners[listenerId] && packets.length) {
                        startTime = new Date(new Date(packets[0].time).getTime() + 1);
                    }
                } catch (err) {
                    yield listener.onError(err);
                    if (err.name === 'NotFoundError') {
                        _this._logger.error(`Strategy ${strategyId} not found, removing listener ${listenerId}`);
                        delete _this._strategyTransactionListeners[listenerId];
                    } else {
                        _this._logger.error(`Failed to retrieve transactions stream for strategy ${strategyId}, ` + `listener ${listenerId}, retrying in ${Math.floor(throttleTime / 1000)} seconds`, err);
                        yield new Promise((res)=>setTimeout(res, throttleTime));
                        throttleTime = Math.min(throttleTime * 2, 30000);
                    }
                }
            }
        })();
    }
    _startSubscriberTransactionStreamJob(listenerId, listener, subscriberId, startTime) {
        var _this = this;
        return _async_to_generator(function*() {
            let throttleTime = _this._errorThrottleTime;
            while(_this._subscriberTransactionListeners[listenerId]){
                const opts = {
                    url: `/users/current/subscribers/${subscriberId}/transactions/stream`,
                    method: 'GET',
                    params: {
                        startTime,
                        limit: 1000
                    },
                    headers: {
                        'auth-token': _this._token
                    },
                    json: true
                };
                try {
                    const packets = yield _this._domainClient.requestCopyFactory(opts, true);
                    yield listener.onTransaction(packets);
                    throttleTime = _this._errorThrottleTime;
                    if (_this._subscriberTransactionListeners[listenerId] && packets.length) {
                        startTime = new Date(new Date(packets[0].time).getTime() + 1);
                    }
                } catch (err) {
                    yield listener.onError(err);
                    if (err.name === 'NotFoundError') {
                        _this._logger.error(`Subscriber ${subscriberId} not found, removing listener ${listenerId}`);
                        delete _this._subscriberTransactionListeners[listenerId];
                    } else {
                        _this._logger.error(`Failed to retrieve transactions stream for subscriber ${subscriberId}, ` + `listener ${listenerId}, retrying in ${Math.floor(throttleTime / 1000)} seconds`, err);
                        yield new Promise((res)=>setTimeout(res, throttleTime));
                        throttleTime = Math.min(throttleTime * 2, 30000);
                    }
                }
            }
        })();
    }
    /**
   * Constructs transaction event listener manager instance
   * @param {DomainClient} domainClient domain client
   */ constructor(domainClient){
        super(domainClient);
        this._domainClient = domainClient;
        this._strategyTransactionListeners = {};
        this._subscriberTransactionListeners = {};
        this._errorThrottleTime = 1000;
        this._logger = LoggerManager.getLogger('TransactionListenerManager');
    }
};
/**
 * Transaction event listener manager
 */ export { TransactionListenerManager as default };

//# sourceMappingURL=data:application/json;base64,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