/**
 * 
 */ "use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, /**
 * Transaction event listener manager
 */ "default", {
    enumerable: true,
    get: function() {
        return TransactionListenerManager;
    }
});
const _metaapiclient = /*#__PURE__*/ _interop_require_default(require("../../metaapi.client"));
const _randomstring = /*#__PURE__*/ _interop_require_default(require("randomstring"));
const _logger = /*#__PURE__*/ _interop_require_default(require("../../../logger"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let TransactionListenerManager = class TransactionListenerManager extends _metaapiclient.default {
    /**
   * Constructs transaction event listener manager instance
   * @param {DomainClient} domainClient domain client
   */ constructor(domainClient){
        super(domainClient);
        this._domainClient = domainClient;
        this._strategyTransactionListeners = {};
        this._subscriberTransactionListeners = {};
        this._errorThrottleTime = 1000;
        this._logger = _logger.default.getLogger("TransactionListenerManager");
    }
    /**
   * Returns the dictionary of strategy transaction listeners
   * @returns {Object} dictionary of strategy transaction listeners
   */ get strategyTransactionListeners() {
        return this._strategyTransactionListeners;
    }
    /**
   * Returns the dictionary of subscriber transaction listeners
   * @returns {Object} dictionary of subscriber transaction listeners
   */ get subscriberTransactionListeners() {
        return this._subscriberTransactionListeners;
    }
    /**
   * Adds a strategy transaction listener
   * @param {UserTransactionListener} listener user transaction listener
   * @param {String} strategyId strategy id
   * @param {Date} [startTime] transaction search start time
   * @returns {String} strategy transaction listener id
   */ addStrategyTransactionListener(listener, strategyId, startTime) {
        const listenerId = _randomstring.default.generate(10);
        this._strategyTransactionListeners[listenerId] = listener;
        this._startStrategyTransactionStreamJob(listenerId, listener, strategyId, startTime);
        return listenerId;
    }
    /**
   * Adds a subscriber transaction listener
   * @param {UserTransactionListener} listener user transaction listener
   * @param {String} subscriberId subscriber id
   * @param {Date} [startTime] transaction search start time
   * @returns {String} subscriber transaction listener id
   */ addSubscriberTransactionListener(listener, subscriberId, startTime) {
        const listenerId = _randomstring.default.generate(10);
        this._subscriberTransactionListeners[listenerId] = listener;
        this._startSubscriberTransactionStreamJob(listenerId, listener, subscriberId, startTime);
        return listenerId;
    }
    /**
   * Removes strategy transaction listener by id
   * @param {String} listenerId listener id 
   */ removeStrategyTransactionListener(listenerId) {
        delete this._strategyTransactionListeners[listenerId];
    }
    /**
   * Removes subscriber transaction listener by id
   * @param {String} listenerId listener id 
   */ removeSubscriberTransactionListener(listenerId) {
        delete this._subscriberTransactionListeners[listenerId];
    }
    async _startStrategyTransactionStreamJob(listenerId, listener, strategyId, startTime) {
        let throttleTime = this._errorThrottleTime;
        while(this._strategyTransactionListeners[listenerId]){
            const opts = {
                url: `/users/current/strategies/${strategyId}/transactions/stream`,
                method: "GET",
                params: {
                    startTime,
                    limit: 1000
                },
                headers: {
                    "auth-token": this._token
                },
                json: true
            };
            try {
                const packets = await this._domainClient.requestCopyFactory(opts, true);
                await listener.onTransaction(packets);
                throttleTime = this._errorThrottleTime;
                if (this._strategyTransactionListeners[listenerId] && packets.length) {
                    startTime = new Date(new Date(packets[0].time).getTime() + 1);
                }
            } catch (err) {
                await listener.onError(err);
                if (err.name === "NotFoundError") {
                    this._logger.error(`Strategy ${strategyId} not found, removing listener ${listenerId}`);
                    delete this._strategyTransactionListeners[listenerId];
                } else {
                    this._logger.error(`Failed to retrieve transactions stream for strategy ${strategyId}, ` + `listener ${listenerId}, retrying in ${Math.floor(throttleTime / 1000)} seconds`, err);
                    await new Promise((res)=>setTimeout(res, throttleTime));
                    throttleTime = Math.min(throttleTime * 2, 30000);
                }
            }
        }
    }
    async _startSubscriberTransactionStreamJob(listenerId, listener, subscriberId, startTime) {
        let throttleTime = this._errorThrottleTime;
        while(this._subscriberTransactionListeners[listenerId]){
            const opts = {
                url: `/users/current/subscribers/${subscriberId}/transactions/stream`,
                method: "GET",
                params: {
                    startTime,
                    limit: 1000
                },
                headers: {
                    "auth-token": this._token
                },
                json: true
            };
            try {
                const packets = await this._domainClient.requestCopyFactory(opts, true);
                await listener.onTransaction(packets);
                throttleTime = this._errorThrottleTime;
                if (this._subscriberTransactionListeners[listenerId] && packets.length) {
                    startTime = new Date(new Date(packets[0].time).getTime() + 1);
                }
            } catch (err) {
                await listener.onError(err);
                if (err.name === "NotFoundError") {
                    this._logger.error(`Subscriber ${subscriberId} not found, removing listener ${listenerId}`);
                    delete this._subscriberTransactionListeners[listenerId];
                } else {
                    this._logger.error(`Failed to retrieve transactions stream for subscriber ${subscriberId}, ` + `listener ${listenerId}, retrying in ${Math.floor(throttleTime / 1000)} seconds`, err);
                    await new Promise((res)=>setTimeout(res, throttleTime));
                    throttleTime = Math.min(throttleTime * 2, 30000);
                }
            }
        }
    }
};

//# sourceMappingURL=data:application/json;base64,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